/*
 * log.c
 *
 * Copyright (c) 2004 Maksim Yevmenkin <m_evmenkin@yahoo.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: log.c,v 1.2 2007/04/23 18:29:18 max Exp $
 */

#include <sys/types.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#define	SYSLOG_NAMES	1
#include <syslog.h>

#include "log.h"

#define	LOG_CHECK_PRI(p)	if ((p) > pri) return

static int	pri = -1;

void
log_open(char const *prog, char const *pri_name, int32_t log2stderr)
{
	CODE	*p;

	openlog(prog, LOG_PID|LOG_NDELAY|(log2stderr? LOG_PERROR:0), LOG_USER);

	if (pri_name != NULL) {
		for (p = &prioritynames[0]; p->c_name != NULL; p ++) {
			if (strcasecmp(p->c_name, pri_name) == 0) {
				pri = p->c_val;
				break;
			}
		}

		if (pri == -1) {
			fprintf(stderr, "Unknown log priority: %s. " \
				"Using default ERROR log priority\n", pri_name);
			pri = LOG_ERR;
		} else if (pri == INTERNAL_NOPRI)
			pri = -1;	/* log no messages */
	} else
		pri = LOG_ERR;
}

void
log_close(void)
{
	closelog();
}

void
log_emerg(char const *message, ...)
{
	va_list	ap;

	LOG_CHECK_PRI(LOG_EMERG);

	va_start(ap, message);
	vsyslog(LOG_EMERG, message, ap);
	va_end(ap);
}

void
log_alert(char const *message, ...)
{
	va_list	ap;

	LOG_CHECK_PRI(LOG_ALERT);

	va_start(ap, message);
	vsyslog(LOG_ALERT, message, ap);
	va_end(ap);
}

void
log_crit(char const *message, ...)
{
	va_list	ap;

	LOG_CHECK_PRI(LOG_CRIT);

	va_start(ap, message);
	vsyslog(LOG_CRIT, message, ap);
	va_end(ap);
}

void
log_err(char const *message, ...)
{
	va_list	ap;

	LOG_CHECK_PRI(LOG_ERR);

	va_start(ap, message);
	vsyslog(LOG_ERR, message, ap);
	va_end(ap);
}

void
log_warning(char const *message, ...)
{
	va_list	ap;

	LOG_CHECK_PRI(LOG_WARNING);

	va_start(ap, message);
	vsyslog(LOG_WARNING, message, ap);
	va_end(ap);
}

void
log_notice(char const *message, ...)
{
	va_list	ap;

	LOG_CHECK_PRI(LOG_NOTICE);

	va_start(ap, message);
	vsyslog(LOG_NOTICE, message, ap);
	va_end(ap);
}

void
log_info(char const *message, ...)
{
	va_list	ap;

	LOG_CHECK_PRI(LOG_INFO);

	va_start(ap, message);
	vsyslog(LOG_INFO, message, ap);
	va_end(ap);
}

void
log_debug(char const *message, ...)
{
	va_list	ap;

	LOG_CHECK_PRI(LOG_DEBUG);

	va_start(ap, message);
	vsyslog(LOG_DEBUG, message, ap);
	va_end(ap);
}

