/*
 * client.c
 *
 * Copyright (c) 2002 Maksim Yevmenkin <m_evmenkin@yahoo.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: client.c,v 1.20 2010/10/22 17:04:11 max Exp $
 * $FreeBSD$
 */

#include <sys/stat.h>
#include <netinet/in.h>

#include <bluetooth.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <inttypes.h>
#include <libgen.h>
#include <limits.h>
#include <obex.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "compat.h"
#include "obexapp.h"
#include "client.h"
#include "event.h"
#include "log.h"
#include "stream.h"
#include "util.h"

/*
 * OBEX request
 */

static int obexapp_client_request_connect
	(obex_t *handle, uint8_t const *target, int target_size);
static int obexapp_client_request_disconnect
	(obex_t *handle);
static int obexapp_client_request_put
	(obex_t *handle, char const *local, char const *remote, int opcode);
static int obexapp_client_request_get
	(obex_t *handle, char const *local, char const *remote, int opcode);
static int obexapp_client_request_setpath
	(obex_t *handle, char *path, int opcode);

/*
 * OBEX request handlers
 */

static obexapp_request_handler_t	obexapp_client_request_connect_done;
static obexapp_request_handler_t	obexapp_client_request_disconnect_done;
static obexapp_request_handler_t	obexapp_client_request_put_done;
static obexapp_request_handler_t	obexapp_client_request_get_done;
static obexapp_request_handler_t	obexapp_client_request_setpath_done;

static void				obexapp_client_handle_input(obex_t *handle);

/*
 * UUID of folder browsing service
 */

static uint8_t	FolderBrowsingServiceUUID[] = {
	0xf9, 0xec, 0x7b, 0xc4, 0x95, 0x3c, 0x11, 0xd2, 
	0x98, 0x4e, 0x52, 0x54, 0x00, 0xdc, 0x9e, 0x09 };

/*
 * OBEX client
 */

int
obexapp_client(obex_t *handle)
{
	context_p	context = (context_p) OBEX_GetUserData(handle);
	char		s1[PATH_MAX], s2[PATH_MAX], *s0, *s, *p1, *p2;
	int		done, error;

	if (OBEX_SetTransportMTU(handle, context->mtu, context->mtu) < 0) {
		log_err("%s(): OBEX_SetTransportMTU failed", __func__);
		return (-1);
	}

	if (OBEX_TransportConnect(handle, (void *) 1, 0) < 0) {
		log_err("%s(): OBEX_TransportConnect failed", __func__);
		return (-1);
	}

	if (context->fbs)
		error = obexapp_client_request_connect(handle,
					FolderBrowsingServiceUUID,
					sizeof(FolderBrowsingServiceUUID));
	else
		error = obexapp_client_request_connect(handle, NULL, 0);

	if (error < 0) {
		printf("Could not connect to the remote OBEX server. " \
			"Response: %s (%#x). Service: %s\n",
			OBEX_ResponseToString(context->rsp), context->rsp,
			context->fbs? "File Browsing" : "None");

		return (-1);
	}

	for (done = 0; !done; ) {
		if ((s0 = obexapp_util_gets("obex> ", s1, sizeof(s1))) == NULL) {
			done = 1;
			continue;
		}

		if ((s = obexapp_util_strsep(&s0, " \t")) == NULL)
			continue;

		error = 0;

		switch (tolower((int)s[0])) {
		case 'c': /* chdir/capability */
			switch (tolower((int)s[1])) {
			case 'a': /* capability */
				error = obexapp_client_request_get(handle,
						NULL, NULL,
						OBEXAPP_GET_CAPABILITY);
				break;

			case 'd': /* chdir */
				p1 = obexapp_util_strsep(&s0, " \t");
				if (p1 == NULL) {
					p1 = obexapp_util_gets("cd: remote directory> ", s1, sizeof(s1));
					if (p1 == NULL) {
						done = 1;
						continue;
					}
					if (p1[0] == '\0') {
						printf("Cancelled\n");
						continue;
					}
				}

				if (strcmp(p1, "/") == 0)
					error = obexapp_client_request_setpath(
							handle, NULL,
							OBEXAPP_SETPATH_ROOT);
				else if (strcmp(p1, "..") == 0)
					error = obexapp_client_request_setpath(
							handle, NULL,
							OBEXAPP_SETPATH_PARENT);
				else
					error = obexapp_client_request_setpath(
							handle, p1,
							OBEXAPP_SETPATH);
				break;

			default:
				printf("CApability, CD?\n");
				break;
			}
			break;

		case 'd': /* disconnect and exit */
			switch (tolower((int)s[1])) {
			case 'e': /* delete */
				p1 = obexapp_util_strsep(&s0, " \t");
				if (p1 == NULL) {
					p1 = obexapp_util_gets("delete: remote file> ", s1, sizeof(s1));
					if (p1 == NULL) {
						done = 1;
						continue;
					}
					if (p1[0] == '\0') {
						printf("Cancelled\n");
						continue;
					}
				}

				error = obexapp_client_request_put(handle, NULL,
						p1, OBEXAPP_PUT_DELETE);
				break;

			case 'i': /* disconnect */
				error = obexapp_client_request_disconnect(handle);
			 	done = 1;
				break;

			default:
				printf("DElete, DIsconnect?\n");
				break;
			}
			break;

		case 'e': /* empty */
			p1 = obexapp_util_strsep(&s0, " \t");
			if (p1 == NULL) {
				p1 = obexapp_util_gets("empty: remote file> ", s1, sizeof(s1));
				if (p1 == NULL) {
					done = 1;
					continue;
				}
				if (p1[0] == '\0') {
					printf("Cancelled\n");
					continue;
				}
			}

			error = obexapp_client_request_put(handle, NULL, p1,
					OBEXAPP_PUT_CREATE_EMPTY);
			break;

		case 'g': /* get */
			if (tolower((int)s[1]) != 'e' || tolower((int)s[2]) != 't') {
				printf("GET, GETDefault?\n");
				continue;
			}

			switch (s[3]) {
			case '\0':
				p1 = obexapp_util_strsep(&s0, " \t");
				p2 = obexapp_util_strsep(&s0, " \t");

				if (p1 == NULL) {
					p1 = obexapp_util_gets("get: remote file> ", s1, sizeof(s1));
					if (p1 == NULL) {
						done = 1;
						continue;
					}
					if (p1[0] == '\0') {
						printf("Cancelled\n");
						continue;
					}

					p2 = obexapp_util_gets("get: local file> ", s2, sizeof(s2));
					if (p2 == NULL) {
						done = 1;
						continue;
					}
					if (p2[0] == '\0') {
						printf("Cancelled\n");
						continue;
					}
				} else if (p2 == NULL) {
					p2 = basename(p1);
					if (p2 == NULL) {
						printf("Cancelled. %s\n",
							strerror(errno));
						continue;
					}
				}

				error = obexapp_client_request_get(handle,
						p2, p1, OBEXAPP_GET);
				break;

			case 'd':
				p1 = obexapp_util_strsep(&s0, " \t");
				if (p1 == NULL) {
					p1 = obexapp_util_gets("getdefault: local file> ", s1, sizeof(s1));
					if (p1 == NULL) {
						done = 1;
						continue;
					}
					if (p1[0] == '\0') {
						printf("Cancelled\n");
						continue;
					}
				}

				error = obexapp_client_request_get(handle,
						p1, NULL, OBEXAPP_GET);
				break;

			default:
				printf("GET, GETDefault?\n");
				break;
			}
			break;

		case 'l':
			switch (tolower((int)s1[1])) {
			case 'c': /* lcd */
				p1 = obexapp_util_strsep(&s0, " \t");
				if (p1 == NULL) {
					p1 = obexapp_util_gets("lcd: path> ", s1, sizeof(s1));
					if (p1 == NULL) {
						done = 1;
						continue;
					}
					if (p1[0] == '\0') {
						printf("Cancelled\n");
						continue;
					}
				}

				if (chdir(p1) < 0)
					printf("Could not chdir(%s): %s\n",
						p1, strerror(errno));
				break;

			case 'p': /* lpwd */
				p1 = getcwd(s1, sizeof(s1));
				if (p1 != NULL)
					printf("Local CWD: %s\n", p1);
				else
					printf("Could not get local CWD: %s\n",
						strerror(errno));
				break;

			case 's': /* ls */
				error = obexapp_client_request_get(handle,
						NULL, NULL,
						OBEXAPP_GET_FOLDER_LISTING);
				break;

			default:
				printf("LCd, LPwd, LS?\n");
				break;
			}
			break;

		case 'm': /* mkdir */
			p1 = obexapp_util_strsep(&s0, " \t");
			if (p1 == NULL) {
				p1 = obexapp_util_gets("mkdir: remote directory> ", s1, sizeof(s1));
				if (p1 == NULL) {
					done = 1;
					continue;
				}	
				if (p1[0] == '\0') {
					printf("Cancelled\n");
					continue;
				}
			}

			error = obexapp_client_request_setpath(handle, p1,
					OBEXAPP_SETPATH_MKDIR);
			break;

		case 'p': /* put */
			p1 = obexapp_util_strsep(&s0, " \t");
			p2 = obexapp_util_strsep(&s0, " \t");

			if (p1 == NULL) {
				p1 = obexapp_util_gets("put: local file> ", s1, sizeof(s1));
				if (p1 == NULL) {
					done = 1;
					continue;
				}
				if (p1[0] == '\0') {
					printf("Cancelled\n");
					continue;
				}

				p2 = obexapp_util_gets("put: remote file> ", s2, sizeof(s2));
				if (p2 == NULL) {
					done = 1;
					continue;
				}
				if (p2[0] == '\0') {
					printf("Cancelled\n");
					continue;
				}
			} else if (p2 == NULL) {
				p2 = basename(p1);
				if (p2 == NULL) {
					printf("Cancelled. %s\n",
						strerror(errno));
					continue;
				}
			}

			error = obexapp_client_request_put(handle, p1, p2,
					OBEXAPP_PUT);
			break;

		default:
			printf("CApability, CD, DElete, DIsconnect, Empty, " \
				"GET, GETDefault, Ls, Mkdir, Put?\n");
			continue;
		}

		printf("%s, response: %s (%#x)\n",
			(error < 0)? "Failure" : "Success",
			OBEX_ResponseToString(context->rsp), context->rsp);
	}

	return (0);
} /* obexapp_client */

/*
 * OBEX non-interactive client
 */

static void
obexapp_non_interactive_client_usage(void)
{
	fprintf(stderr,
"Supported non-interactive commands:\n" \
"get remote-file [local-file]   get remote-file and save it as local-file\n" \
"getdefault local-file          get default vCard and save it as local-file\n" \
"put local-file [remote-file]   put local-file as remote-file\n");
}

int
obexapp_non_interactive_client(obex_t *handle, int argc, char **argv)
{
	context_p	 context = (context_p) OBEX_GetUserData(handle);
	int		 (*handler)(obex_t *, char const *, char const *, int);
	char		*command = NULL, *local = NULL, *remote = NULL;
	int		 opcode, error;

	if (argc == 0) {
		obexapp_non_interactive_client_usage();
		return (-1);
	}

	command = argv[0];
	argc --;
	argv ++;

	if (strcasecmp(command, "get") == 0) {
		handler = obexapp_client_request_get;
		opcode = OBEXAPP_GET;

		switch (argc) {
		case 2:
			remote = argv[0];
			local = argv[1];
			break;

		case 1:
			remote = argv[0];
			local = basename(remote);
			if (local == NULL) {
				fprintf(stderr, "Unable to construct local " \
					"name for the GET command. %s (%d)",
	                                strerror(errno), errno);
				return (-1);
			}

			if (context->root[0] == '\0') {
				context->root[0] = '.';
				context->root[1] = '\0';
			}

			strlcat(context->root, "/", PATH_MAX);
			strlcat(context->root, local, PATH_MAX);

			local = context->root;
			break;

		default:
			obexapp_non_interactive_client_usage();
			return (-1);
			/* NOT REACHED */
		}
	} else if (strcasecmp(command, "getdefault") == 0) {
		handler = obexapp_client_request_get;
		opcode = OBEXAPP_GET;

		switch (argc) {
		case 1:
			local = argv[0];
			remote = NULL;
			break;

		default:
			obexapp_non_interactive_client_usage();
			return (-1);
			/* NOT REACHED */
		}
	} else if (strcasecmp(command, "put") == 0) {
		handler = obexapp_client_request_put;
		opcode = OBEXAPP_PUT;

		switch (argc) {
		case 2:
			local = argv[0];
			remote = argv[1];
			break;

		case 1:
			local = argv[0];
			remote = basename(local);
			if (remote == NULL) {
				fprintf(stderr, "Unable to construct remote " \
					"name for the PUT command. %s (%d)",
					strerror(errno), errno); 
				return (-1);
			}
			break;

		default:
			obexapp_non_interactive_client_usage();
			return (-1);
			/* NOT REACHED */
		}
	} else {
		obexapp_non_interactive_client_usage();
		return (-1);
	}

	/* CONNECT */
	if (OBEX_SetTransportMTU(handle, context->mtu, context->mtu) < 0) {
		fprintf(stderr, "OBEX_SetTransportMTU failed\n");
		return (-1);
	}

	if (OBEX_TransportConnect(handle, (void *) 1, 0) < 0) {
		fprintf(stderr, "OBEX_TransportConnect failed\n");
		return (-1);
	}

	if (context->fbs)
		error = obexapp_client_request_connect(handle,
					FolderBrowsingServiceUUID,
					sizeof(FolderBrowsingServiceUUID));
	else
		error = obexapp_client_request_connect(handle, NULL, 0);

	if (error < 0) {
		fprintf(stderr, "Could not connect to the remote OBEX server. "\
			"Response: %s (%#x). Service: %s\n",
			OBEX_ResponseToString(context->rsp), context->rsp,
			context->fbs? "File Browsing" : "None");

		return (-1);
	}

	/* COMMAND */
	error = (*handler)(handle, local, remote, opcode);
	if (error < 0)
		fprintf(stderr, "Cound not perform non-interactive %s(%s %s) " \
			"command. Response: %s (%#x).\n",
			command, (local != NULL)? local : "NULL",
			(remote != NULL)? remote : "NULL",
			OBEX_ResponseToString(context->rsp), context->rsp);

	/* DISCONNECT */
	error = obexapp_client_request_disconnect(handle);
	if (error < 0)
		fprintf(stderr, "Could not disconnect from the remote OBEX " \
			"server. Response: %s (%#x)\n",
			OBEX_ResponseToString(context->rsp), context->rsp);

	return (0);
}

/*
 * Submit OBEX_CMD_CONNECT 
 */

static int
obexapp_client_request_connect(obex_t *handle, uint8_t const *target, int target_size)
{
	context_p		 context = (context_p) OBEX_GetUserData(handle);
	obex_object_t		*object = NULL;
	obex_headerdata_t	 hv;
	int			 error = -1;

	context->rsp = OBEX_RSP_BAD_REQUEST; /* XXX */
	context->connection_id = OBEXAPP_INVALID_CONNECTION_ID;

	if ((object = OBEX_ObjectNew(handle, OBEX_CMD_CONNECT)) == NULL) {
		log_err("%s(): OBEX_ObjectNew failed", __func__);
		goto done;
	}

	if (target != NULL) {
		hv.bs = target;
		if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_TARGET,
				hv, target_size, 0) < 0) {
			log_err("%s(): Could not add HDR_TARGET", __func__);
			goto done;
		}
	}

	if ((error = OBEX_Request(handle, object)) != 0) {
		log_err("%s(): OBEX_Request failed", __func__);
		goto done;
	} else
		object = NULL; /* Now we gave object to the library */

	obexapp_client_handle_input(handle);

	if (target != NULL &&
	    context->connection_id == OBEXAPP_INVALID_CONNECTION_ID)
		log_warning("%s(): Server did not return connection ID!",
			__func__);

	if (context->rsp != OBEX_RSP_SUCCESS)
		error = -1;
	else
		error = 0;
done:
	if (object != NULL) {
		OBEX_ObjectDelete(handle, object);
		object = NULL;
	}

	return (error);
}

/*
 * Submit OBEX_CMD_DISCONNECT 
 */

static int
obexapp_client_request_disconnect(obex_t *handle)
{
	context_p		 context = (context_p) OBEX_GetUserData(handle);
	obex_object_t		*object = NULL;
	obex_headerdata_t	 hv;
	int			 error = -1;

	context->rsp = OBEX_RSP_BAD_REQUEST; /* XXX */

	if ((object = OBEX_ObjectNew(handle, OBEX_CMD_DISCONNECT)) == NULL) {
		log_err("%s(): OBEX_ObjectNew failed", __func__);
		goto done;
	}

	if (context->connection_id != OBEXAPP_INVALID_CONNECTION_ID) {
		hv.bq4 = context->connection_id;
		if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_CONNECTION,
				hv, sizeof(hv.bq4), 0) < 0) {
			log_err("%s(): Could not add HDR_CONNECTION", __func__);
			goto done;
		}
	}

	if ((error = OBEX_Request(handle, object)) != 0) {
		log_err("%s(): OBEX_Request failed", __func__);
		goto done;
	} else
		object = NULL; /* Now we gave object to the library */

	obexapp_client_handle_input(handle);
	OBEX_TransportDisconnect(handle);

	error = 0;
	context->rsp = OBEX_RSP_SUCCESS;
done:
	if (object != NULL) {
		OBEX_ObjectDelete(handle, object);
		object = NULL;
	}
	
	return (error);
}

/*
 * Submit OBEX_CMD_PUT
 */

static int
obexapp_client_request_put(obex_t *handle, char const *local, 
		char const *remote, int opcode)
{
	context_p		 context = (context_p) OBEX_GetUserData(handle);
	obex_object_t		*object = NULL;
	obex_headerdata_t	 hv;
	struct stat		 st;
	uint8_t			*ucname = NULL;
	int			 ucname_len, error;

	context->rsp = OBEX_RSP_BAD_REQUEST; /* XXX */
	context->opcode = opcode;
	context->sfd = -1;
	error = -1;

	if (remote == NULL || strchr(remote, '/') != NULL) {
		log_err("%s(): Invalid remote name %s",
			__func__, (remote != NULL)? remote : "NULL");
		goto done;
	}

	if ((object = OBEX_ObjectNew(handle, OBEX_CMD_PUT)) == NULL) {
		log_err("%s(): OBEX_ObjectNew failed", __func__);
		goto done;
	}

	if (context->connection_id != OBEXAPP_INVALID_CONNECTION_ID) {
		hv.bq4 = context->connection_id;
		if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_CONNECTION,
				hv, sizeof(hv.bq4), 0) < 0) {
			log_err("%s(): Could not add HDR_CONNECTION", __func__);
			goto done;
		}
	}

	ucname_len = (strlen(remote) + 1) * 2;
	if ((ucname = (uint8_t *) malloc(ucname_len)) == NULL) {
		log_crit("%s(): Could not allocate ucname", __func__);
		goto done;
	}

	if (obexapp_util_locale_to_utf16be(remote, strlen(remote),
						ucname, ucname_len) < 0) {
		log_err("%s(): Could not convert to UTF-16BE", __func__);
		goto done;
	}

	hv.bs = ucname;
	if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_NAME,
			hv, ucname_len, 0) < 0) {
		log_err("%s(): Could not add HDR_NAME", __func__);
		goto done;
	}

	switch (opcode) {
	case OBEXAPP_PUT:
		if (local == NULL) {
			log_err("%s(): Invalid local name", __func__);
			goto done;
		}

		if ((context->sfd = open(local, O_RDONLY)) < 0) {
			log_err("%s(): Could not open(%s). %s (%d)",
				__func__, local, strerror(errno), errno);
			goto done;
		}

		if (fstat(context->sfd, &st) < 0) {
			log_err("%s(): Could not fstat(%d). %s (%d)",
				__func__, context->sfd, strerror(errno),
				errno);
			goto done;
		}

		if (!S_ISREG(st.st_mode)) {
			log_err("%s(): Object %s is not a regular file",
				__func__, local);
			goto done;
		}

		log_info("%s(): Putting object %s as %s, size %" PRId64,
			__func__, local, remote, st.st_size);

		hv.bq4 = st.st_size;
		if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_LENGTH, 
				hv, sizeof(hv.bq4), 0) < 0) {
			log_err("%s(): Could not add HDR_LENGTH", __func__);
			goto done;
		}

		hv.bs = NULL;
		if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_BODY,
				hv, 0, OBEX_FL_STREAM_START) < 0) {
			log_err("%s(): Could not add HDR_BODY", __func__);
			goto done;
		}

		obexapp_stream_stats_reset(context);
		break;

	case OBEXAPP_PUT_CREATE_EMPTY:
		log_info("%s(): Creating empty object %s", __func__, remote);

		hv.bs = NULL;
		if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_BODY_END,
				hv, 0, 0) < 0) {
			log_err("%s(): Could not add HDR_BODY_END", __func__);
			goto done;
		}
		break;

	case OBEXAPP_PUT_DELETE:
		log_info("%s(): Deleting object %s", __func__, remote);
		break;

	default:
		log_err("%s(): Unknown opcode %d for OBEX_PUT_CMD",
			__func__, opcode);
		goto done;
		/* NOT REACHED */
	}

	if ((error = OBEX_Request(handle, object)) != 0) {
		log_err("%s(): OBEX_Request failed", __func__);
		goto done;
	} else
		object = NULL; /* Now we gave object to the library */

	obexapp_client_handle_input(handle);

	if (context->rsp != OBEX_RSP_SUCCESS)
		error = -1;
	else {
		error = 0;
		if (opcode == OBEXAPP_PUT)
			obexapp_stream_stats_print(context);
	}

done:
	if (object != NULL) {
		OBEX_ObjectDelete(handle, object);
		object = NULL;
	}

	if (ucname != NULL) {
		free(ucname);
		ucname = NULL;
	}

	if (context->sfd >= 0) {
		close(context->sfd);
		context->sfd = -1;
	}

	return (error);
} /* obexapp_client_request_put */

/*
 * Submit OBEX_CMD_GET
 */

static int
obexapp_client_request_get(obex_t *handle, char const *local, 
		char const *remote, int opcode)
{
	context_p		 context = (context_p) OBEX_GetUserData(handle);
	obex_object_t		*object = NULL;
	obex_headerdata_t	 hv;
	uint8_t			*ucname = NULL;
	int			 ucname_len, error;

	context->rsp = OBEX_RSP_BAD_REQUEST; /* XXX */
	context->opcode = opcode;
	context->temp[0] = '\0';
	context->sfd = -1;
	context->ls[0] = '\0';
	error = -1;

	if ((object = OBEX_ObjectNew(handle, OBEX_CMD_GET)) == NULL) {
		log_err("%s(): OBEX_ObjectNew failed", __func__);
		goto done;
	}

	if (context->connection_id != OBEXAPP_INVALID_CONNECTION_ID) {
		hv.bq4 = context->connection_id;
		if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_CONNECTION,
				hv, sizeof(hv.bq4), 0) < 0) {
			log_err("%s(): Could not add HDR_CONNECTION", __func__);
			goto done;
		}
	}

	switch (opcode) {
	case OBEXAPP_GET:
		if (local == NULL) {
			log_err("%s(): Invalid local name", __func__);
			goto done;
		}

		log_info("%s(): Getting object %s, local name %s",
			__func__, (remote == NULL)? "default" : remote, local);

		if (remote != NULL) {
			ucname_len = (strlen(remote) + 1) * 2;
			if ((ucname = (uint8_t *) malloc(ucname_len)) == NULL) {
				log_crit("%s(): Could not allocate ucname",
					__func__);
				goto done;
			}

			if (obexapp_util_locale_to_utf16be(
					remote, strlen(remote),
					ucname, ucname_len) < 0) {
				log_err("%s(): Could not convert to UTF-16BE",
					__func__);
				goto done;
			}

			hv.bs = ucname;
			if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_NAME,
					hv, ucname_len, 0) < 0) {
				log_err("%s(): Could not add HDR_NAME",
					__func__);
				goto done;
			}
		} else {
			hv.bs = (uint8_t const *) "text/x-vCard";
			if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_TYPE,
					hv, strlen((char const *) hv.bs) + 1, 0) < 0) {
				log_err("%s(): Could not add HDR_TYPE",
					__func__);
				goto done;
			}
		}

		context->sfd = obexapp_util_mkstemp(local, context->temp, PATH_MAX);
		if (context->sfd < 0) {
			log_err("%s(): Could not obexapp_util_mkstemp. %s (%d)",
				__func__, strerror(errno), errno);
			goto done;
		}

		obexapp_stream_stats_reset(context);
		OBEX_ObjectReadStream(handle, object, NULL);
		break;

	case OBEXAPP_GET_FOLDER_LISTING:
		log_info("%s(): Getting folder listing for %s",
			__func__, (remote == NULL)? "current folder" : remote);

		if (remote == NULL)
			remote = ""; /* Empty Name header */

		ucname_len = (strlen(remote) + 1) * 2;
		if ((ucname = (uint8_t *) malloc(ucname_len)) == NULL) {
			log_crit("%s(): Could not allocate ucname", __func__);
			goto done;
		}

		if (obexapp_util_locale_to_utf16be(remote, strlen(remote),
						ucname, ucname_len) < 0) {
			log_err("%s(): Could not convert to UTF-16BE",
				__func__);
			goto done;
		}

		hv.bs = ucname;
		if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_NAME,
				hv, ucname_len, 0) < 0) {
			log_err("%s(): Could not add HDR_NAME", __func__);
			goto done;
		}

		hv.bs = (uint8_t const *) "x-obex/folder-listing";
		if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_TYPE,
				hv, strlen((char const *) hv.bs) + 1, 0) < 0) {
			log_err("%s(): Could not add HDR_TYPE", __func__);
			goto done;
		}
		break;

	case OBEXAPP_GET_CAPABILITY:
		log_info("%s(): Getting OBEX capability object", __func__);

		hv.bs = (uint8_t const *) "x-obex/capability";
		if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_TYPE,
				hv, strlen((char const *) hv.bs) + 1, 0) < 0) {
			log_err("%s(): Could not add HDR_TYPE", __func__);
			goto done;
		}
		break;

	default:
		log_err("%s(): Unknown opcode %d for OBEX_GET_CMD",
			__func__, opcode);
		goto done;
		/* NOT REACHED */
	}

	if ((error = OBEX_Request(handle, object)) != 0) {
		log_err("%s(): OBEX_Request failed", __func__);
		goto done;
	} else
		object = NULL; /* Now we gave object to the library */
		
	obexapp_client_handle_input(handle);

	if (context->rsp == OBEX_RSP_SUCCESS) {
		switch (context->opcode) {
		case OBEXAPP_GET_FOLDER_LISTING:
			obexapp_util_display_folder_listing(context->ls);
			break;

		case OBEXAPP_GET_CAPABILITY:
			printf("%s\n", context->ls);
			error = 0;
			break;

		default:
			if ((error = rename(context->temp, local)) < 0)
				log_err("%s(): Could not rename(%s, %s). " \
					"%s (%d)", __func__,
					context->temp, local, strerror(errno),
					errno);
			else
				obexapp_stream_stats_print(context);
			break;
		}
	} else
		error = -1;
done:
	if (object != NULL) {
		OBEX_ObjectDelete(handle, object);
		object = NULL;
	}

	if (ucname != NULL) {
		free(ucname);
		ucname = NULL;
	}

	if (context->sfd != -1) {
		close(context->sfd);
		context->sfd = -1;
	}

	if (context->opcode == OBEXAPP_GET)
		unlink(context->temp);

	return (error);
} /* obexapp_client_request_get */

/*
 * Submit OBEX_CMD_SETPATH
 */

static int
obexapp_client_request_setpath(obex_t *handle, char *path, int opcode)
{
	context_p		 context = (context_p) OBEX_GetUserData(handle);
	obex_object_t		*object = NULL;
	obex_headerdata_t	 hv;
	obex_setpath_hdr_t	 hdr;
	uint8_t			*ucname = NULL;
	int			 ucname_len = 0, error;

	hdr.constants = 0;
	context->rsp = OBEX_RSP_BAD_REQUEST; /* XXX */
	context->opcode = opcode;
	error = -1;	

	if ((object = OBEX_ObjectNew(handle, OBEX_CMD_SETPATH)) == NULL) {
		log_err("%s(): OBEX_ObjectNew failed", __func__);
		goto done;
	}

	switch (opcode) {
	case OBEXAPP_SETPATH:
	case OBEXAPP_SETPATH_MKDIR:
		if (path == NULL) {
			log_err("%s(): Invalid path", __func__);
			goto done;
		}

		log_info("%s(): %sdir %s", __func__, 
			(opcode == OBEXAPP_SETPATH)? "ch" : "mk", path);

		hdr.flags = (opcode == OBEXAPP_SETPATH)? 0x02 : 0x00;

		ucname_len = (strlen(path) + 1) * 2;
		if ((ucname = (uint8_t *) malloc(ucname_len)) == NULL) {
			log_crit("%s(): Could not allocate ucname", __func__);
			goto done;
		}

		if (obexapp_util_locale_to_utf16be(path, strlen(path),
						ucname, ucname_len) < 0) {
			log_err("%s(): Could not convert to UTF-16BE",
				__func__);
			goto done;
		}
		break;

	case OBEXAPP_SETPATH_PARENT:
		log_info("%s(): chdir ..", __func__);
		hdr.flags = 0x03;
		break;

	case OBEXAPP_SETPATH_ROOT:
		log_info("%s(): chdir /", __func__);
		hdr.flags = 0x2;

		ucname_len = 2;	/* Unicode - 2 bytes NULL */
		if ((ucname = (uint8_t *) malloc(ucname_len)) == NULL) {
			log_crit("%s(): Could not allocate ucname", __func__);
			goto done;
		}

		ucname[0] = ucname[1] = '\0';
		break;

	default:
		log_err("%s(): Unknown opcode %d for OBEX_SETPATH_CMD",
			__func__, opcode);
		goto done;
		/* NOT REACHED */
	}

	if (OBEX_ObjectSetNonHdrData(object, (uint8_t *)&hdr, sizeof(hdr)) < 0) {
		log_err("%s(): Could not set non-header data", __func__);
		goto done;
	}

	if (context->connection_id != OBEXAPP_INVALID_CONNECTION_ID) {
		hv.bq4 = context->connection_id;
		if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_CONNECTION,
				hv, sizeof(hv.bq4), 0) < 0) {
			log_err("%s(): Could not add HDR_CONNECTION", __func__);
			goto done;
		}
	}

	if (ucname != NULL) {
		hv.bs = ucname;
		if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_NAME,
				hv, ucname_len, 0) < 0) {
			log_err("%s(): Could not add HDR_NAME", __func__);
			goto done;
		}
	}

	if ((error = OBEX_Request(handle, object)) != 0) {
		log_err("%s(): OBEX_Request failed", __func__);
		goto done;
	} else
		object = NULL; /* Now we gave object to the library */
		
	obexapp_client_handle_input(handle);

	if (context->rsp != OBEX_RSP_SUCCESS)
		error = -1;
	else
		error = 0;
done:
	if (object != NULL) {
		OBEX_ObjectDelete(handle, object);
		object = NULL;
	}

	if (ucname != NULL) {
		free(ucname);
		ucname = NULL;
	}

	return (error);
}

/*
 * Wait until OBEX request finished
 */

static void
obexapp_client_handle_input(obex_t *handle)
{
	context_p	context = (context_p) OBEX_GetUserData(handle);
	int		error;

	log_debug("%s(): Entering event processing loop...", __func__);

	for (context->done = 0; !context->done; ) {
		error = OBEX_HandleInput(context->handle,
					OBEXAPP_TRANSPORT_TIMEOUT);
		if (error < 0) {
			log_err("%s(): OBEX_HandleInput failed, error=%d",
				__func__, error);
			break;
		}
	}

	log_debug("%s(): Event processing loop completed", __func__);
} /* obexapp_client_handle_input */

/*
 * Process OBEX_EV_REQDONE event
 */

void
obexapp_client_request_done(obex_t *handle, obex_object_t *object, 
			int obex_cmd, int obex_rsp)
{
	log_debug("%s()", __func__);

	switch (obex_cmd) {
	case OBEX_CMD_CONNECT:
		obexapp_client_request_connect_done(handle, object, obex_rsp);
		break;

	case OBEX_CMD_DISCONNECT:
		obexapp_client_request_disconnect_done(handle, object, obex_rsp);
		break;

	case OBEX_CMD_PUT:
		obexapp_client_request_put_done(handle, object, obex_rsp);
		break;

	case OBEX_CMD_GET:
		obexapp_client_request_get_done(handle, object, obex_rsp);
		break;

	case OBEX_CMD_SETPATH:
		obexapp_client_request_setpath_done(handle, object, obex_rsp);
		break;

#if 0 /* XXX FIXME - what to do with these? */
	case OBEX_CMD_COMMAND:
	case OBEX_CMD_ABORT:
	case OBEX_FINAL:
#endif

	default:
		log_debug("%s(): Command %#x has finished, response %#x",
			__func__, obex_cmd, obex_rsp);
		break;
	}
} /* obexapp_client_request_done */

/*
 * Process OBEX_CMD_CONNECT response
 */

static int
obexapp_client_request_connect_done(obex_t *handle, obex_object_t *object,
		int obex_rsp)
{
	context_p		 context = (context_p) OBEX_GetUserData(handle);
	obex_headerdata_t	 hv;
	uint8_t			 hi, *data = NULL;
	uint32_t		 hlen;

	log_debug("%s(): Connect completed, response %#x", __func__, obex_rsp);

	context->done = 1;
	context->rsp = obex_rsp;

	if (obex_rsp != OBEX_RSP_SUCCESS)
		return (obex_rsp);

	if (OBEX_ObjectGetNonHdrData(object, &data) == sizeof(obex_connect_hdr_t))
		log_debug("%s(): OBEX connect header: " \
			"version=%#x, flags=%#x, mtu=%d", __func__,
			((obex_connect_hdr_t *) data)->version,
			((obex_connect_hdr_t *) data)->flags,
			ntohs(((obex_connect_hdr_t *) data)->mtu));
	else
		log_err("%s(): Invalid OBEX connect header?!", __func__);

	while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen)) {
		switch (hi) {
		case OBEX_HDR_WHO:
			log_debug("%s(): WHO length %d", __func__, hlen);
			break;

		case OBEX_HDR_CONNECTION:
			log_debug("%s(): Connection ID - %#x",
				__func__, hv.bq4);

			context->connection_id = hv.bq4;
			break;

		default:
			log_debug("%s(): Skipping header, hi=%#x, hlen=%d",
				__func__, hi, hlen);
			break;
		}
	}

	return (obex_rsp);
} /* obexapp_client_request_connect_done */

/*
 * Process OBEX_CMD_DISCONNECT response
 */

static int
obexapp_client_request_disconnect_done(obex_t *handle,
		__unused obex_object_t *object, int obex_rsp)
{
	context_p	context = (context_p) OBEX_GetUserData(handle);

	log_debug("%s(): Disconnect completed, response %#x",
		__func__, obex_rsp);

	context->done = 1;
	context->rsp = obex_rsp;

	return (obex_rsp);
} /* obexapp_client_request_disconnect_done */

/*
 * Process OBEX_CMD_PUT response
 */

static int
obexapp_client_request_put_done(obex_t *handle, obex_object_t *object,
		int obex_rsp)
{
	context_p		context = (context_p) OBEX_GetUserData(handle);
	obex_headerdata_t	hv;
	uint8_t			hi;
	uint32_t		hlen;

	log_debug("%s(): Put completed, response %#x", __func__, obex_rsp);

	context->done = 1;
	context->rsp = obex_rsp;

	if (obex_rsp != OBEX_RSP_SUCCESS)
		return (obex_rsp);

	while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen)) {
		switch (hi) {
		case OBEX_HDR_LENGTH:
			log_debug("%s(): Length - %d", __func__, hv.bq4);
			break;

		case OBEX_HDR_CONNECTION:
			log_debug("%s(): Connection ID - %#x",
				__func__, hv.bq4);
			break;

		case OBEX_HDR_NAME:
			if (hlen == 0) {
				log_debug("%s(): Got no Name", __func__);
				break;
			}

			if (obexapp_util_locale_from_utf16be(hv.bs, hlen,
						context->file, PATH_MAX) < 0)
				log_err("%s(): Could not convert from UTF-16BE",
					__func__);
			else
				log_debug("%s(): Name - %s, hlen=%d",
					__func__, context->file, hlen);
			break;

		default:
			log_debug("%s(): Skipping header, hi=%#x, hlen=%d",
				__func__, hi, hlen);
			break;
		}
	}

	return (obex_rsp);
} /* obexapp_client_request_put_done */

/*
 * Process OBEX_CMD_GET response
 */

static int
obexapp_client_request_get_done(obex_t *handle, obex_object_t *object,
		int obex_rsp)
{
	context_p		 context = (context_p) OBEX_GetUserData(handle);
	obex_headerdata_t	 hv;
	uint8_t			 hi;
	uint32_t		 hlen;
	int			 body_len = 0;
	uint8_t const		*body = NULL;

	log_debug("%s(): Get completed, response %#x", __func__, obex_rsp);

	context->done = 1;
	context->rsp = obex_rsp;

	if (obex_rsp != OBEX_RSP_SUCCESS)
		return (obex_rsp);

	while(OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen)) {
		switch (hi) {
		case OBEX_HDR_LENGTH:
			log_debug("%s(): Length - %d", __func__, hv.bq4);
			break;

		case OBEX_HDR_BODY:
		case OBEX_HDR_BODY_END:
			body = hv.bs;
			body_len = hlen;

			log_debug("%s(): Body%s, length %d",
				__func__, (hi == OBEX_HDR_BODY)? "" : " end",
				body_len);
			break;

                case OBEX_HDR_CONNECTION:
			log_debug("%s(): Connection ID - %#x",
				__func__, hv.bq4);
			break;

		case OBEX_HDR_NAME:
			if (hlen == 0) {
				log_debug("%s(): Got no Name", __func__);
				break;
			}

			if (obexapp_util_locale_from_utf16be(hv.bs, hlen,
						context->file, PATH_MAX) < 0)
				log_err("%s(): Could not convert from UTF-16BE",
					__func__);
			else
				log_debug("%s(): Name - %s, hlen=%d",
					__func__, context->file, hlen);
			break;

		default:
			log_debug("%s(): Skipping header, hi=%#x, hlen=%d",
				__func__, hi, hlen);
			break;
		}
	}

	switch (context->opcode) {
	case OBEXAPP_GET_FOLDER_LISTING:
	case OBEXAPP_GET_CAPABILITY:
		if (body_len + 1 > context->ls_size) {
			char	*ls = (char *) realloc(context->ls, body_len + 1);

			if (ls == NULL) {
				log_crit("%s(): Could not reallocate context->ls",
					__func__);
				break;
			}

			context->ls = ls;
			context->ls_size = body_len + 1;
		}

		memcpy(context->ls, body, body_len);
		context->ls[body_len] = '\0';
		break;

	case OBEXAPP_GET:
		break;

	default:
		log_err("%s(): Unknown opcode %d for OBEX_GET_CMD",
			__func__, context->opcode);
		break;
	}

	return (obex_rsp);
} /* obexapp_client_request_get_done */

/*
 * Process OBEX_CMD_SETPATH response
 */

static int
obexapp_client_request_setpath_done(obex_t *handle, obex_object_t *object,
		int obex_rsp)
{
	context_p		context = (context_p) OBEX_GetUserData(handle);
	obex_headerdata_t	hv;
	uint8_t			hi;
	uint32_t		hlen;

	log_debug("%s(): SetPath completed, response %#x", __func__, obex_rsp);

	context->done = 1;
	context->rsp = obex_rsp;

	if (obex_rsp != OBEX_RSP_SUCCESS)
		return (obex_rsp);

	while(OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen)) {
		switch (hi) {
                case OBEX_HDR_CONNECTION:
			log_debug("%s(): Connection ID - %#x",
				__func__, hv.bq4);
			break;

		case OBEX_HDR_NAME:
			if (hlen == 0) {
				log_debug("%s(): Got no Name", __func__);
				break;
			}

			if (obexapp_util_locale_from_utf16be(hv.bs, hlen,
						context->file, PATH_MAX) < 0)
				log_err("%s(): Could not convert from UTF-16BE",
					__func__);
			else
				log_debug("%s(): Name - %s, hlen=%d",
					__func__, context->file, hlen);
			break;

		default:
			log_debug("%s(): Skipping header, hi=%#x, hlen=%d",
				__func__, hi, hlen);
			break;
		}
	}

	return (obex_rsp);
} /* obexapp_client_request_setpath_done */

