//
// nono
// Copyright (C) 2023 nono project
// Licensed under nono-license.txt
//

//
// Windrv (相当) ホストドライバ
//

#pragma once

#include "windrv.h"
#include <map>
#include <vector>

// Windrv のホスト実装。
class HostWindrv : public Device
{
	using inherited = Device;

	// dirs の最大ディレクトリ数。
	static const int MaxVDirs = 20;

 public:
	explicit HostWindrv(WindrvDevice *parent_);
	~HostWindrv() override;

	// ルートパスの設定。
	bool InitRootPath(const std::string& rootpath_);

	uint32 Initialize();

	uint32 GetCapacity(Human68k::capacity *);

	// ゲストパスのディレクトリをオープンする。
	VDir *OpenVDir(const std::string& path_);
	// vdir をクローズする。
	void CloseVDir(VDir *);

	// path のファイルエントリについて FILES の以下を埋めて返す。
	//  .attr Human68k のファイル属性
	//  .date Human68k 形式の日付
	//  .time Human68k 形式の時刻
	//  .size ファイルサイズ
	bool GetFileStat(Human68k::FILES *dst, const std::string& path) const;

	uint32 MakeDir(const std::string& path_);
	uint32 RemoveDir(const std::string& path_);
	uint32 Rename(const std::string& oldpath, const std::string& newpath);
	uint32 Delete(const std::string& path_);
	uint32 GetAttribute(const std::string& path_);
	uint32 SetAttribute(const std::string& path_, uint8 attr_);
	uint32 CreateFile(Windrv::FCB *, const std::string& path_, uint8 attr);
	uint32 Open(Windrv::FCB *, const std::string& path_);
	void   Close(Windrv::FCB *);
	uint32 Read(Windrv::FCB *, void *dst, uint32 len);
	uint32 Write(Windrv::FCB *, const void *src, uint32 len);
	uint32 Seek(Windrv::FCB *, int whence, int32 offset);
	uint32 GetFileTime(Windrv::FCB *);
	uint32 SetFileTime(Windrv::FCB *, uint32 datetime);
	uint32 Flush();

 private:
	// 仮想ディレクトリ構造

	// ゲストパス path に対応する VDir を (必要なら作成して) 返す。
	// 見付からなければ NULL を返す。
	VDir *GetVDirByGuest(const std::string& path);

	// ホスト相対パス path に対応する VDir があれば返す。
	// 見付からなければ NULL を返す。
	VDir *FindVDir(const std::string& path) const;

	// vdir を必要なら更新。
	bool UpdateVDir(VDir *vdir);

	// ホスト相対パス path に対応する VDir を作成して追加。
	VDir *AddVDir(const std::string& path);

	// vdir のディレクトリエントリを(再)構成する。
	bool FillVDir(VDir *, const char *caller, bool force);

	// path を分離する。
	static std::vector<std::string> SplitPath(const std::string& path);

	// path を正規化する。
	static std::string NormalizePath(const std::string& path);

	// name が Human68k ファイル名の制約を満たしていれば true を返す。
	bool IsHuman68kFilename(const char *name) const;

	// ホストパス path の属性を返す。
	uint32 GetAttributeInternal(const std::string& path, struct stat *) const;

#if defined(HAVE_STAT_ST_TIMESPEC)
	static uint64 timespec2nsec(const struct timespec& ts);
#endif

	// 文字コード(未対応)
	std::string ToHostCharset(const std::string& src) const;
	std::string ToGuestCharset(const std::string& src) const;

	// ルートディレクトリのホスト側パス。末尾 '/'。
	std::string rootpath {};

	// ホスト相対パスと対応する仮想ディレクトリの連想配列。
	std::map<std::string, VDir *> vdirs {};
};
