//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// LUNA ビットマッププレーン
//

#pragma once

#include "planevram.h"

// VRAM の ROP (定数)
struct ROP {
	static const uint ZERO	= 0;	// 0
	static const uint AND1	= 1;	// vram & data
	static const uint AND2	= 2;	// vram & ~data
	static const uint cmd	= 3;
	static const uint AND3	= 4;	// ~vram & data
	static const uint THRU	= 5;	// data
	static const uint EOR	= 6;	// vram ^ data
	static const uint OR1	= 7;	// vram | data
	static const uint NOR	= 8;	// ~vram | ~data
	static const uint ENOR	= 9;	// (vram & data) | (~vram & ~data)
	static const uint INV1	= 10;	// ~data
	static const uint OR2	= 11;	// vram | ~data
	static const uint INV2	= 12;	// ~vram
	static const uint OR3	= 13;	// ~vram | data
	static const uint NAND	= 14;	// ~vram | ~data
	static const uint ONE	= 15;	// 1
};

class LunafbDevice : public PlaneVRAMDevice
{
	using inherited = PlaneVRAMDevice;

	static const uint32 plane0base = 0xb10c0000;
	static const uint32 plane7end  = 0xb12c0000;
	static const uint32 fcset0base = 0xb1300000;
	static const uint32 fcset7end  = 0xb1500000;

	// プレーン数の上限
	static const uint MAX_PLANES = 8;

 public:
	LunafbDevice();
	~LunafbDevice() override;

	bool Init() override;
	void ResetHard(bool poweron) override;

	busdata Read(busaddr addr) override;
	busdata Write(busaddr addr, uint32 data) override;
	busdata Peek1(uint32 addr) override;
	bool Poke1(uint32 addr, uint32 data) override;

	void UpdateInfo(TextScreen&, int x, int y) const override;

	// エミュレータによる出力
	void EmuInitScreen();
	void EmuFill(uint x, uint y, uint w, uint h, uint32 data);
	void EmuPutc(uint x, uint y, uint c);
	void EmuPutc(uint x, uint y, uint c, uint rop);
	void EmuScrollY(uint dy, uint sy, uint h);
	void SetBMSEL(uint8 bmsel_) { bmsel = bmsel_; }

 private:
	// offset の VRAM が変更されたことを記録する
	inline void SetDirty(uint32 offset);

	// X 方向にはみ出したときに対応する仮想画面の Y 座標を返す。
	int GetWrapY(int y) const override;

	busdata WriteReg(busaddr addr, uint32 data);
	void WriteRFCNT(uint32);
	void WriteCommonBitmap8(uint32, uint32);
	void WriteCommonBitmap16(uint32, uint32);
	void WriteCommonBitmap32(uint32, uint32);
	void WriteCommonFCSet(uint rop, uint32 data);
	void WritePlane8(uint, uint32, uint32);
	void WritePlane16(uint, uint32, uint32);
	void WritePlane32(uint, uint32, uint32);

	DECLARE_MONITOR_CALLBACK(MonitorUpdate);

	// 演算
	static uint32 RasterOp(uint rop, uint32 vram, uint32 data);

	// ROP 文字列を返す
	static const char *RopStr(uint rop);

	// 現在のプレーン数でのプレーンの終了アドレス
	uint32 plane_end {};
	// 現在のプレーン数での FCSet の(ブロック単位での)最終アドレス
	uint32 fcset_end {};

	// RFCNT
	//
	//    3                   2                   1                   0
	//  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
	// +---------------+---------------+-----------+---+---------------+
	// |      無効     |  水平カウンタ |    無効   |  ラスタカウンタ   |
	// +---------------+---------------+-----------+---+---------------+
	uint32 rfcnt {};
	// 親クラスにある
	// int xscroll {};
	// int yscroll {};

	// プレーンセレクタ
	//
	//    3                   2                   1                   0
	//  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
	// +---------------+---------------+---------------+-------+-+-+-+-+
	// |                           無効                        |3|2|1|0|
	// +---------------+---------------+---------------+-------+-+-+-+-+
	//                                                          | | | |
	//  プレーンN (%1 = 同時設定/書き込み有効、%0=無効) --------+-+-+-+

	uint32 bmsel {};

	// ファンクションセットとアクセスマスク
	uint fcset[MAX_PLANES] {};
	uint32 fcmask[MAX_PLANES] {};

	// ウェイト [nsec]
	busdata wait {};

	Monitor *monitor {};
};

static inline LunafbDevice *GetLunafbDevice() {
	return Object::GetObject<LunafbDevice>(OBJ_PLANEVRAM);
}
