//
// nono
// Copyright (C) 2024 nono project
// Licensed under nono-license.txt
//

//
// メイン画面をコンソールにする
//

#include "console.h"
#include "bitmap.h"
#include "builtinrom.h"
#include "comdriver_cons.h"
#include "mainapp.h"
#include "scheduler.h"
#include "wxcolor.h"
#include <algorithm>

// エスケープシーケンス等。
#define ESC		'\x1b'
#define CSI		'['

#define PRINTLOG(fmt...)	do {	\
	if (log) {	\
		fprintf(log, fmt);	\
		fflush(log);	\
	}	\
} while (0)

// v[] がいずれかでも true なら true を返す。
static bool
AnyOf(const std::vector<bool>& v)
{
	return std::any_of(v.begin(), v.end(), [](bool x) { return x; });
}

// コンストラクタ
ConsoleDevice::ConsoleDevice()
	: inherited(OBJ_CONSOLE)
{
	screen.resize(width * height);
	dirty.resize(height);
	pending.resize(height);
	SetPalette(false);
}

// デストラクタ
ConsoleDevice::~ConsoleDevice()
{
	if (log) {
		fclose(log);
		log = NULL;
	}
}

// 初期化
bool
ConsoleDevice::Init()
{
	renderer = GetRenderer();

	// ログファイル。
	if (gMainApp.console_logfile) {
		log = fopen(gMainApp.console_logfile, "w");
		if (log == NULL) {
			warn("console logfile '%s'", gMainApp.console_logfile);
			return false;
		}
	}

	vsync_event.func = ToEventCallback(&ConsoleDevice::VSyncCallback);
	vsync_event.time = 16666.667_usec;
	vsync_event.SetName("Console V-Sync");
	scheduler->RegistEvent(vsync_event);

	return true;
}

// COM ドライバを接続。
// (HostCOM の cons ドライバが呼ぶ)
void
ConsoleDevice::Attach(COMDriver *comdriver_)
{
	comdriver = comdriver_;

	SetPalette((comdriver != NULL));
}

void
ConsoleDevice::ResetHard(bool poweron)
{
	if (poweron) {
		// 最初に一回フチを含めて描画する。
		init_screen = true;

		// シリアルコンソールなのでリセットには反応しない。
		Clear();
		Locate(0, 0);
	}

	scheduler->RestartEvent(vsync_event);

	ResetTerminal();
}

void
ConsoleDevice::VSyncCallback(Event& ev)
{
	bool update_needed;

	// 更新 (dirty[]) があれば pending[] に重ねる。
	{
		std::lock_guard<std::mutex> lock(mtx);
		for (int y = 0; y < height; y++) {
			// std::vector<bool> には operator|=() がない。
			if (__predict_false(dirty[y])) {
				pending[y] = true;
				dirty[y] = false;
			}
		}

		update_needed = AnyOf(pending);
	}

	// 更新の必要があれば Render に作画指示。
	if (update_needed) {
		renderer->NotifyRender();
	}

	scheduler->StartEvent(ev);
}

// パレットの状態を変更する。
void
ConsoleDevice::SetPalette(bool console_is_active)
{
	if (console_is_active) {
		palette[0] = UD_LIGHT_GREY;	// BG
		palette[1] = UD_BLACK;		// FG
	} else {
		palette[0] = UD_LIGHT_GREY;	// BG
		palette[1] = UD_GREY;		// FG
	}
	// 反転色。
	palette[2] = palette[0];

	// 画面に即反映させる。
	{
		std::lock_guard<std::mutex> lock(mtx);
		std::fill(pending.begin(), pending.end(), true);
	}
}

// 端末の状態をリセットする。
// (画面のクリアはここではしない?)
void
ConsoleDevice::ResetTerminal()
{
	ResetState();
	SetAttr(0);
	scroll_top = 0;
	scroll_btm = height - 1;
	save_curx = 0;
	save_cury = 0;
}

// エスケープシーケンスの状態を初期化する。
void
ConsoleDevice::ResetState()
{
	state = 0;
	seq.clear();
	arg.clear();
}

// ch をログ出力する。
void
ConsoleDevice::PutcharDebug(uint32 ch)
{
	if (ch == '\n') {
		// 改行が来たらログも改行する。
		fputs("\\n\n", log);
		log_newline = true;
	} else {
		if (ch == ESC) {
			// 読みやすさのため ESC の前で改行したい。
			if (log_newline == false) {
				fputc('\n', log);
			}
			fputs("ESC", log);
		} else if (ch == '\r') {
			fputs("\\r", log);
		} else if (ch == '\t') {
			fputs("\\t", log);
		} else if (ch < 0x20 || ch >= 0x7f) {
			fprintf(log, "\\x%02x", ch);
		} else {
			fputc(ch, log);
		}
		log_newline = false;
	}
	fflush(log);
}

// シリアルコンソールから来る出力。
// ホストドライバスレッドから呼ばれる。
void
ConsoleDevice::Putchar(uint32 ch)
{
	if (__predict_false(log)) {
		PutcharDebug(ch);
	}

	switch (state) {
	 case ESC:
		if (PutcharESC(ch)) {
			return;
		}
		break;

	 case CSI:
		if (PutcharCSI(ch)) {
			return;
		}
		break;

	 case '(':	// G0
		switch (ch) {
		 case '0':		// DEC 特殊記号
			SetAttr(cur_attr | ATTR_DECSG);
			break;
		 case 'B':		// ASCII
		 default:
			SetAttr(cur_attr & ~ATTR_DECSG);
			break;
		}
		ResetState();
		return;

	 case ')':	// G1
	 case '$':	// 94 multibyte
	 case '#':
		// とりあえず何もしない。
		ResetState();
		return;

	 case '?':	// CSI '?'
		if (PutcharCSIQ(ch)) {
			return;
		}
		break;

	 default:
		PutcharPlain(ch);
		return;
	}

	// シーケンスを破棄。ここまでのを表示して平文に戻る。
	putlog(0, "Unknown Sequence:%s", Dump(ch).c_str());
	for (auto c : seq) {
		Putc(c);
	}
	Putc(ch);
	ResetState();
}

// 何のシーケンスでもない1文字を表示する。
void
ConsoleDevice::PutcharPlain(uint32 ch)
{
	switch (ch) {
	 case '\x00':	// NUL
	 case '\x01':	// SOH
	 case '\x02':	// STX
	 case '\x03':	// ETX
	 case '\x04':	// EOT
	 case '\x05':	// ENQ
	 case '\x06':	// ACK
		break;
	 case '\x07':	// BEL
		// とりあえず無視。
		break;

	 case '\x08':	// BS
		LocateX(cur_x - 1);
		return;

	 case '\t':		// TAB
		// XXX 右端どうなる?
		LocateX(roundup(cur_x, 8));
		return;

	 case '\n':		// LF
	 case '\x0b':	// VT
	 case '\x0c':	// FF
		LF();
		return;

	 case '\r':	// CR
		CR();
		return;

	 case '\x0e':	// SO
	 case '\x0f':	// SI
		// Change GL
		break;

	 case '\x10':	// DLE
	 case '\x11':	// DC1/XON
	 case '\x12':	// DC2
	 case '\x13':	// DC3/XOFF
	 case '\x14':	// DC4
	 case '\x15':	// NAK
	 case '\x16':	// SYN
	 case '\x17':	// ETB
		break;

	 case '\x18':	// CAN
		// Cancel
		ResetState();
		return;

	 case '\x19':	// EM
		break;

	 case '\x1a':	// SUB
		ResetState();
		return;

	 case ESC:
		seq.push_back(ch);
		state = ch;
		return;

	 case '\x1c':	// FS
	 case '\x1d':	// GS
	 case '\x1e':	// RS
	 case '\x1f':	// US
		break;

	 default:
		if (ch < 0x80) {
			// ここが通常の1文字出力。
			Putc(ch);
			return;
		}
		// 8ビットシーケンスは無視?
		break;
	}
	putlog(2, "Ignore: \\x%02x", ch);
}

// ESC の次の文字を処理する。
// 処理すれば true、知らないシーケンスなので中止する場合は false を返す。
bool
ConsoleDevice::PutcharESC(uint32 ch)
{
	switch (ch) {
	 case '[':		// CSI
	 case '(':		// G0
	 case ')':		// G1
	 case '$':		// 94 multibyte
	 case '#':		// Font width/height control
		seq.push_back(ch);
		state = ch;
		return true;

	 case '\\':		// ST
		break;

	 case '7':		// カーソル位置保存
		save_curx = cur_x;
		save_cury = cur_y;
		break;

	 case '8':		// カーソル位置復帰
		Locate(save_curx, save_cury);
		break;

	 case 'c':		// Hard terminal reset
		Clear();
		ResetTerminal();
		Locate(0, 0);
		break;

	 case 'M':		// 1行上に移動 (画面を1行下にスクロール)
		ScrollDown();
		Locate(0, 0);
		break;

	 case '*':		// G2
	 case '+':		// G3
	 case 'B':		// ASCII
	 case 'A':		// ISO Latin 1
	 case '<':		//
	 case '0':		// DEC special graphics
	 case '-':		// G1 (96)
	 case '.':		// G2
	 case '/':		// G3
	 case '4':		// Dutch
	 case '5':
	 case '6':		// Norwegian/Danish
	 case 'C':		// Finnish
	 case 'R':		// French
	 case 'Q':		// French canadian
	 case 'K':		// German
	 case 'Y':		// Italian
	 // "%5" とかの2文字のやつは NetBSD/x68k ite も未対応。
	 case '`':
	 case 'n':
	 case '}':
	 case 'o':
	 case '|':
	 case '~':
	 case '=':
	 case '>':
		putlog(2, "Ignore:%s", Dump(ch).c_str());
		break;

	 case 'Z':		// 端末属性を応答する。
	 default:
		return false;
	}

	ResetState();
	return true;
}

// CSI の次の文字を処理する。
// 処理すれば true、知らないシーケンスなので中止する場合は false を返す。
bool
ConsoleDevice::PutcharCSI(uint32 ch)
{
	switch (ch) {
	 case '0':
	 case '1':
	 case '2':
	 case '3':
	 case '4':
	 case '5':
	 case '6':
	 case '7':
	 case '8':
	 case '9':
	 case ';':
	 case '\"':
	 case '$':
	 case '>':
		seq.push_back(ch);
		arg.push_back(ch);
		return true;

	 case 'A':
	 {
		int n = atoi(arg.c_str());
		if (n == 0) {
			n = 1;
		}
		LocateY(cur_y - n);
		ResetState();
		return true;
	 }

	 case 'B':
	 {
		int n = atoi(arg.c_str());
		if (n == 0) {
			n = 1;
		}
		LocateY(cur_y + n);
		ResetState();
		return true;
	 }

	 case 'C':
	 {
		int n = atoi(arg.c_str());
		if (n == 0) {
			n = 1;
		}
		LocateX(cur_x + n);
		ResetState();
		return true;
	 }

	 case 'D':
	 {
		int n = atoi(arg.c_str());
		if (n == 0) {
			n = 1;
		}
		LocateX(cur_x - n);
		ResetState();
		return true;
	 }

	 case 'H':
	 {
		// CSI <y> ; <x> H … (x-1, y-1) にカーソルを移動。
		int y = atoi(arg.c_str());
		int x = 0;
		const char *p = strchr(arg.c_str(), ';');
		if (p) {
			x = atoi(p + 1);
		}
		if (x > 0) {
			x--;
		}
		if (y > 0) {
			y--;
		}
		Locate(x, y);
		ResetState();
		return true;
	 }

	 case 'J':
	 {
		// CSI [<n>] J
		// <n> が 0 か省略なら、カーソル位置から右下までを消去。
		// <n> が 1 なら、カーソル位置から左上までを消去。
		// <n> が 2 なら、全画面消去。
		// XXX SPA と ERM は未対応。
		int n = atoi(arg.c_str());
		switch (n) {
		 default:	// 知らんけど
		 case 0:
			for (int y = cur_y; y < height - 1; y++) {
				int x = (y == cur_y) ? cur_x : 0;
				for (; x < width; x++) {
					Setc(x, y, ' ' | cur_attr);
				}
			}
			break;
		 case 1:
			for (int y = 0; y <= cur_y; y++) {
				int xend = (y == cur_y) ? cur_x : (width - 1);
				for (int x = 0; x <= xend; x++) {
					Setc(x, y, ' ' | cur_attr);
				}
			}
			break;
		 case 2:
			// カーソル位置は移動しないまま?
			Clear();
			break;
		}
		ResetState();
		return true;
	 }

	 case 'K':
	 {
		// CSI [<n>] K
		// <n> が 0 か省略なら、カーソル位置の右側を消去。
		// <n> が 1 なら、カーソル位置の左側を消去。
		// <n> が 2 なら、カーソル行を消去。
		int n = atoi(arg.c_str());
		int x0;
		int x1;
		switch (n) {
		 default:	// 知らんけど
		 case 0:
			x0 = cur_x;
			x1 = width - 1;
			break;
		 case 1:
			x0 = 0;
			x1 = cur_x;
			break;
		 case 2:
			x0 = 0;
			x1 = width - 1;
			break;
		}
		for (int x = x0; x <= x1; x++) {
			Setc(x, cur_y, ' ' | cur_attr);
		}
		ResetState();
		return true;
	 }

	 case 'm':
	 {
		// CSI 7 ; 37 m みたいにセミコロンで区切ってパラメータを複数書ける。
		// パラメータには 38:2:rr:gg:bb みたいにコロンで区切ったひとかたまり
		// もある。
		const char *ap = arg.c_str();
		for (;;) {
			int n = atoi(ap);
			switch (n) {
			 case 0:			// すべての属性を解除
				SetAttr(0);
				break;
			 case 1:			// ボールドをセット
				SetAttr(cur_attr | ATTR_BOLD);
				break;
			 case 2:			// 低輝度をセット (対応予定なし)
				break;
			 case 3:			// イタリックをセット
				SetAttr(cur_attr | ATTR_ITALIC);
				break;
			 case 4:			// 下線をセット
				SetAttr(cur_attr | ATTR_UNDERLINE);
				break;
			 case 5:			// 低速点滅をセット (対応予定なし)
				break;
			 case 6:			// 高速点滅をセット (対応予定なし)
				break;
			 case 7:			// 反転をセット
				SetAttr(cur_attr | ATTR_REVERSE);
				break;
			 case 8:			// 文字色を背景色と同じにする
				break;
			 case 9:			// 打ち消し線をセット
				SetAttr(cur_attr | ATTR_STRIKE);
				break;
			 case 10 ... 20:	// フォントセットの設定?
				break;
			 case 21:			// 二重下線をセット (対応予定なし)
				break;
			 case 22:			// ボールド(と低輝度)を解除
				SetAttr(cur_attr & ~ATTR_BOLD);
				break;
			 case 23:			// イタリックを解除
				SetAttr(cur_attr & ~ATTR_ITALIC);
				break;
			 case 24:			// 下線を解除
				SetAttr(cur_attr & ~ATTR_UNDERLINE);
				break;
			 case 25:			// 点滅を解除
				break;
			 case 26:			// ?
				break;
			 case 27:			// 反転を解除
				SetAttr(cur_attr & ~ATTR_REVERSE);
				break;
			 case 28:			// シークレットを解除
				break;
			 case 29:			// 打ち消し線を解除
				SetAttr(cur_attr & ~ATTR_STRIKE);
				break;
			 case 30 ... 37:	// 文字色の設定
				break;
			 case 38:			// 文字色(拡張)を設定
				break;
			 case 39:			// 文字色をデフォルトに戻す
				break;
			 case 40 ... 47:	// 背景色を設定
				break;
			 case 48:			// 背景色(拡張)を設定
				break;
			 case 49:			// 背景色をデフォルトに戻す
				break;
			 case 51:			// □を重ねる (対応予定なし)
				break;
			 case 52:			// ○を重ねる (対応予定なし)
				break;
			 case 53:			// 上線をセット
				SetAttr(cur_attr | ATTR_OVERLINE);
				break;
			 case 54:			// □○を取り消し
				break;
			 case 55:			// 上線を取り消し
				SetAttr(cur_attr & ~ATTR_OVERLINE);
				break;
			 case 60 ... 63:	// 縦線 (対応予定なし)
				break;
			 case 64:			// 二重打ち消し線 (対応予定なし)
				break;
			 case 65:			// 60-64 を取り消し
				break;
			 case 90 ... 97:	// 文字色(高輝度)を設定
				break;
			 case 100 ... 107:	// 背景色(高輝度)を設定
				break;
			 default:
				// 無視
				break;
			}

			if (*ap == '\0') {
				break;
			}
			ap = strchr(ap + 1, ';');
			if (ap == NULL) {
				break;
			}
			ap++;
		}

		ResetState();
		return true;
	 }

	 case 'r':
	 {
		// CSI <t> ; <b> r。<t> 省略時は 1。<b> 省略時は画面最下行。
		int top = atoi(arg.c_str());
		int btm = height;
		const char *p = strchr(arg.c_str(), ';');
		if (p) {
			btm = atoi(p + 1);
		}
		if (top > 0) {
			top--;
		}
		if (btm > 0) {
			btm--;
		}
		scroll_top = top;
		scroll_btm = btm;
		ResetState();
		return true;
	 }

	 case '?':
		seq.push_back(ch);
		state = ch;
		return true;

	 default:
		return false;
	}
}

// CSI '?' の次の文字を処理する。
// 処理すれば true、知らないシーケンスなので中止する場合は false を返す。
bool
ConsoleDevice::PutcharCSIQ(uint32 ch)
{
	switch (ch) {
	 case '0':
	 case '1':
	 case '2':
	 case '3':
	 case '4':
	 case '5':
	 case '6':
	 case '7':
	 case '8':
	 case '9':
	 case ';':
	 case '\"':
	 case '$':
	 case '>':
		seq.push_back(ch);
		arg.push_back(ch);
		return true;

	 case 'h':	// 拡張オプションを設定
	 case 'l':	// 拡張オプションを解除
		// CSI '?' <n> h
		// CSI '?' <n> l
		// 全部未対応。
		putlog(2, "Ignored:%s", Dump(ch).c_str());
		ResetState();
		return true;

	 default:
		return false;
	}
}

// 現在の seq を表示用文字列にして返す。
std::string
ConsoleDevice::Dump() const
{
	return Dump(seq);
}

// 現在の seq と ch を表示用文字列にして返す。
// 大抵の場合最後の1文字はまだ seq に入っていないのでこの形式があると便利。
std::string
ConsoleDevice::Dump(uint32 ch) const
{
	std::vector<uint8> tmp = seq;
	tmp.push_back(ch);
	return Dump(tmp);
}

// エスケープシーケンスを表示用文字列にして返す。
// 戻り文字列には先頭に空白が入っている。
/*static*/ std::string
ConsoleDevice::Dump(const std::vector<uint8>& src)
{
	std::string dst;

	for (auto ch : src) {
		if (ch == ESC) {
			dst += " ESC";
		} else if (ch < ' ' || ch >= 0x7f) {
			dst += string_format(" \\x%02x", ch);
		} else if (ch == ' ') {
			dst += " ' '";
		} else {
			dst += ' ';
			dst += (char)ch;
		}
	}
	return dst;
}

// 画面を消去する。
// カーソル位置は変わらない。属性もクリアされない。
void
ConsoleDevice::Clear()
{
	std::fill(screen.begin(), screen.end(), ' ' | cur_attr);
	std::fill(dirty.begin(), dirty.end(), true);
}

// カーソルの X 座標を移動。(画面をはみ出さないようにクリップされる)
void
ConsoleDevice::LocateX(int x)
{
	if (__predict_false(x < 0)) {
		x = 0;
	}
	if (__predict_false(x > width - 1)) {
		x = width - 1;
	}
	cur_x = x;
	dirty[cur_y] = true;
}

// カーソルの Y 座標を移動。(画面をはみ出さないようにクリップされる)
void
ConsoleDevice::LocateY(int y)
{
	if (__predict_false(y < 0)) {
		y = 0;
	}
	if (__predict_false(y > height - 1)) {
		y = height - 1;
	}
	dirty[cur_y] = true;
	cur_y = y;
	dirty[cur_y] = true;
}

// 文字出力後のカーソル移動。
void
ConsoleDevice::Ascend()
{
	cur_x++;
	dirty[cur_y] = true;
	if (cur_x >= width) {
		CRLF();
	}
}

// 復帰。
void
ConsoleDevice::CR()
{
	LocateX(0);
}

// 改行。(DECSTBM の影響を受ける)
void
ConsoleDevice::LF()
{
	dirty[cur_y] = true;
	cur_y++;
	if (cur_y >= scroll_btm) {
		ScrollUp();
		cur_y = scroll_btm;
	}
	dirty[cur_y] = true;
}

// 全画面を1行上にスクロール。(DECSTBM の影響を受ける)
// カーソル位置は移動しない。
void
ConsoleDevice::ScrollUp()
{
	int dst = scroll_top * width;
	int lines = scroll_btm - scroll_top;
	memmove(&screen[dst], &screen[dst + width],
		lines * width * sizeof(screen[0]));
	uint32 ch = ' ' | cur_attr;
	for (int x = 0; x < width; x++) {
		Setc(x, scroll_btm, ch);
	}
	std::fill(dirty.begin(), dirty.end(), true);
}

// 全画面を1行下にスクロール。(DECSTBM の影響を受ける)
// カーソル位置は移動しない。
void
ConsoleDevice::ScrollDown()
{
	int src = scroll_top * width;
	int lines = scroll_btm - scroll_top;
	memmove(&screen[src + width], &screen[src],
		lines * width * sizeof(screen[0]));
	uint32 ch = ' ' | cur_attr;
	for (int x = 0; x < width; x++) {
		Setc(x, scroll_top, ch);
	}
	std::fill(dirty.begin(), dirty.end(), true);
}

// 現在位置に、現在の属性で文字 ch を出力する。
// 出力後カーソルは移動する。
void
ConsoleDevice::Putc(uint32 ch)
{
	if (ch == '\n') {
		CR();
		LF();
		return;
	}

	Setc(cur_x, cur_y, ch | cur_attr);
	Ascend();
}

// (x, y) の位置に chattr を出力する。
// 出力後カーソルは移動しない。
// 現在の attr は参照しない (呼び出す側が指定すること)。
void
ConsoleDevice::Setc(int x, int y, uint32 chattr)
{
	int pos = y * width + x;
	assert(pos >= 0);
	assert(pos < width * height);
	screen[pos] = chattr;
}

// 現在の属性を設定する。
// ほぼログ出力のため。
void
ConsoleDevice::SetAttr(uint32 new_attr)
{
	if (cur_attr == new_attr) {
		return;
	}
	cur_attr = new_attr;

	if (log) {
		std::string buf;
		if ((cur_attr & ATTR_BOLD)) {
			buf += ",ATTR_BOLD";
		}
		if ((cur_attr & ATTR_ITALIC)) {
			buf += ",ATTR_ITALIC";
		}
		if ((cur_attr & ATTR_UNDERLINE)) {
			buf += ",ATTR_UNDERLINE";
		}
		if ((cur_attr & ATTR_STRIKE)) {
			buf += ",ATTR_STRIKE";
		}
		if ((cur_attr & ATTR_OVERLINE)) {
			buf += ",ATTR_OVERLINE";
		}
		if ((cur_attr & ATTR_DECSG)) {
			buf += ",ATTR_DECSG";
		}
		if ((cur_attr & ATTR_REVERSE)) {
			buf += ",ATTR_REVERSE";
		}

		if (buf.empty()) {
			buf = "<ATTR_NORMAL>";
		} else {
			buf[0] = '<';
			buf += '>';
		}
		fputs(buf.c_str(), log);
		fflush(log);
	}
}

// 画面合成。
// レンダリングスレッドから呼ばれる。
bool
ConsoleDevice::Render(BitmapRGBX& dst)
{
	bool updated = false;

	// フチも含めて再描画。
	if (__predict_false(init_screen)) {
		init_screen = false;
		dst.Fill(palette[0]);
		updated = true;
	}

	// 行ごとに更新があるか調べる。
	std::vector<bool> modified(height);
	{
		std::lock_guard<std::mutex> lock(mtx);
		modified = pending;
		std::fill(pending.begin(), pending.end(), false);
	}
	updated |= AnyOf(modified);

	const uint8 *cgrom8x16 = Builtin::CGROM8x16(0);
	for (uint y = 0; y < height; y++) {
		if (__predict_false(modified[y])) {
			uint py = Padding + y * font_height;
			const uint32 *s = &screen[y * width];
			for (uint x = 0; x < width; x++) {
				uint px = Padding + x * font_width;
				uint32 attr = *s++;
				// とりあえず上半分(G1)も無視。
				uint32 ch = attr & 0x7f;
				// 属性による反転。
				int paloff = 0;
				if ((attr & ATTR_REVERSE) != 0) {
					paloff ^= 1;
				}
				// カーソル位置は更に反転。
				if (x == cur_x && y == cur_y) {
					paloff ^= 1;
				}

				const uint8 *font;
				if ((attr & ATTR_DECSG) && ('j' <= ch && ch <= 'x')) {
					font = &decsg8x16[(ch - 'j') * font_height];
				} else {
					font = &cgrom8x16[ch * font_height];
				}
				BitmapI1 bc1(font, font_width, font_height);
				if (__predict_false((attr & ATTR_ITALIC))) {
					bc1 = bc1.ConvertToItalic();
				}
				if (__predict_false((attr & ATTR_BOLD))) {
					bc1 = bc1.ConvertToBold();
				}

				dst.DrawBitmapI1(px, py, bc1, &palette[paloff]);

				// 下線、上線、取り消し線
				if (__predict_false((attr & ATTR_LINEMASK))) {
					Color fg = palette[paloff ^ 1];
					if ((attr & ATTR_OVERLINE)) {
						dst.DrawLineH(fg, px, py + 0, px + font_width);
					}
					if ((attr & ATTR_STRIKE)) {
						dst.DrawLineH(fg, px, py + 9, px + font_width);
					}
					if ((attr & ATTR_UNDERLINE)) {
						dst.DrawLineH(fg, px, py + 14, px + font_width);
					}
				}
			}
		}
	}

	return updated;
}

#define B(x)	(0b##x)
#define B2(x)	(B(x) >> 8), (B(x) & 0xff)

// DEC Special Graphics Character Set フォント。(罫線のみ)
/*static*/ const uint8 ConsoleDevice::decsg8x16[15 * font_height] = {
	B(00010000),	// [0] j:┘
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(11110000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),

	B(00000000),	// [1] k:┐
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(11110000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),

	B(00000000),	// [2] l:┌
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00011111),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),

	B(00010000),	// [3] m:└
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00011111),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),

	B(00010000),	// [4] n:┼
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(11111111),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),

	B(00000000),	// [5] o:─(Scan1)
	B(11111111),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),

	B(00000000),	// [6] p:─(Scan3)
	B(00000000),
	B(00000000),
	B(00000000),
	B(11111111),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),

	B(00000000),	// [7] q:─(Scan5)
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(11111111),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),

	B(00000000),	// [8] r:─(Scan7)
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(11111111),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),

	B(00000000),	// [9] s:─(Scan9)
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(11111111),
	B(00000000),
	B(00000000),

	B(00010000),	// [10] t:├
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00011111),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),

	B(00010000),	// [11] u:┤
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(11110000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),

	B(00010000),	// [12] v:┴
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(11111111),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),

	B(00000000),	// [13] w:┬
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(00000000),
	B(11111111),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),

	B(00010000),	// [14] x:│
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
	B(00010000),
};


//
// 入力担当デバイス
//

// コンストラクタ
ConsoleKeyboard::ConsoleKeyboard()
	: inherited()
{
}

// デストラクタ
ConsoleKeyboard::~ConsoleKeyboard()
{
}

// COM ドライバを接続。
// (HostCOM の cons ドライバが呼ぶ)
void
ConsoleKeyboard::Attach(COMDriverConsole *comdriver_)
{
	comdriver = comdriver_;
}

// 文字入力。
// WXMainView から呼ばれる。
void
ConsoleKeyboard::CharInput(uint charcode)
{
	if (comdriver == NULL) {
		return;
	}

	// charcode は 0x80 以上を独自に使っているので (keyboard.h 参照)、
	// それらをシリアルコンソールのエスケープに置き換えて送り込む。
	switch (charcode) {
	 case CC_NUL:
		comdriver->EnqueueChar(0);
		break;

	 case CC_F1 ... CC_F10:
		break;

	 case CC_up:
		comdriver->EnqueueChar(ESC);
		comdriver->EnqueueChar('[');
		comdriver->EnqueueChar('A');
		break;

	 case CC_down:
		comdriver->EnqueueChar(ESC);
		comdriver->EnqueueChar('[');
		comdriver->EnqueueChar('B');
		break;

	 case CC_right:
		comdriver->EnqueueChar(ESC);
		comdriver->EnqueueChar('[');
		comdriver->EnqueueChar('C');
		break;

	 case CC_left:
		comdriver->EnqueueChar(ESC);
		comdriver->EnqueueChar('[');
		comdriver->EnqueueChar('D');
		break;

	 default:
		comdriver->EnqueueChar(charcode);
		break;
	}
}
