/*
 * $Header: /home/vikas/src/nocol/nsmon/RCS/main.c,v 1.13 1998/08/28 03:20:41 vikas Exp $
 */

/*
 *
 * DESCRIPTION: name-server monitor
 *   This program opens the data file supplied by the user  and then
 *   creates a 'nsmon-output' file for use with netmon.  It then
 *   directly reads and writes from this file.  If it gets SIGHUP, it
 *   rescans the data file before continuing.
 *
 *   Can now query multiple domains.
 *
 * Copyright 1997 Netplex Technologies Inc., info@netplex-tech.com
 * Copyright 1994 Vikas Aggarwal vikas@navya.com
 * Copyright 1992 JvNCnet
 *
 */

/*
 * $Log: main.c,v $
 * Revision 1.13  1998/08/28 03:20:41  vikas
 * Back to displaying the site name instead of the domain name in the
 * 'sitename' field (more flexible)
 *
 * Revision 1.12  1998/07/31 18:23:27  vikas
 * Now handles multiple domains.
 * Displays the domain being queried instead of the 'sitename'
 * since the domain name is more relevant.
 *
 * Revision 1.11  1997/04/12  11:30:31  vikas
 * can now query multiple domains (sent in by Daniel Thorson,
 * dthorson@muon.nrm.minn.seagate.com) in the config file.
 *
 *
 * Revision 1.9  1994/06/20 17:46:02  vikas
 * Added -n option for non-auth answers. Patch sent in by meyer@ns.uoregon.edu
 *
 * Revision 1.8  1994/06/09  21:14:54  vikas
 * Deleted the definition for inet_addr
 * It is defined in arpa/inet.h
 *
 * Revision 1.7  1994/05/16  02:07:07  vikas
 * Major cleanup for using the new 'nocol_startup()' library
 * call. No setjmp() etc. calls.
 *
 * Revision 1.6  1994/01/21  17:32:05  aggarwal
 * Moved a NOT_AUTH answer to Error condition. Also fixed bug in
 * malloc-ing of the pid-filename (needed 2 chars and not 1)
 *
 * Revision 1.5  1994/01/10  19:46:00  aggarwal
 * Added typecasts to malloc. Included poll_sites.c into this main.c file.
 * More response codes from nsmon()
 *
 * Revision 1.4  1993/10/29  22:02:36  aggarwal
 * Added reading poll-interval and the domain-name from the
 * config file.
 *
 * Revision 1.3  1993/10/02  05:36:23  aggarwal
 * Now puts pidfile in the ETC directory. Also deleted the code for
 * handling the signalling (should be using sockets for logging at this
 * time).
 *
 * Revision 1.2  1992/06/14  23:38:16  aggarwal
 * Had to increment 'sender' after 'strrchr'
 *
 * Revision 1.1  1992/06/11  05:02:00  aggarwal
 * Initial revision
 *
 */

/*  */

#ifndef lint
 static char rcsid[] = "$RCSfile: main.c,v $ $Revision: 1.13 $ $Date: 1998/08/28 03:20:41 $" ;
#endif

#include "nocol.h"			/*	common structures	*/

#define GLOBALS				/* for global variables */
# include "nsmon.h"			/* program specific defines	*/
#undef GLOBALS

#include <string.h>			/* For strcat() definitions	*/
#include <sys/file.h>
#include <signal.h>			/* For signal numbers		*/


static char	*configfile;		/* File with the list of nodes	*/
static char	*datafile ;		/* Names of the data file	*/
static char	*querydata;		/* Query data string (domainname) */
static char	*sender;
static time_t	pollinterval ;		/* Time between polls */
static int maxseverity = E_CRITICAL ; 	/* Max severity of events in nocol */


/*
 * For each host being monitored, we keep a linked list of which domain
 * to query for and whether it should be queried for an AUTHORITATIVE
 * answer only. At the start of each polling cycle, we reset to the
 * beginning of the linked list
 */
int ndomain = -1;		/* Index to the no. of domains specified */
struct site_info
{
  char *domain;		/* Domain name */
  int aa_wanted;	/* 1 if Authorative Answer required */
  struct site_info *next;
};

struct site_info *site_info_list = NULL;	/* linked list */
void free_site_list();		/* to free up the linked list */

main (ac, av)
  int ac;
  char **av;
{
  extern char *optarg;
  extern int optind;
  int  fdout = 0;			/* File desc for output data file */
  register int c ;
  char *p;
  time_t starttm, polltime ;

  prognm = av[0] ;				/* Save the program name */

#ifdef SENDER
  sender = SENDER ;
#else						/* delete the directory name */
  if ((sender = (char *)strrchr (prognm , '/')) == NULL)
    sender = prognm ;				/* no path in program name */
  else
    sender++ ;				/* skip leading '/' */
#endif

  while ((c = getopt(ac, av, "do:")) != EOF)
    switch (c)
    {
    case 'd':
      debug++ ;
      break ;

    case 'o':				/* output datafile */
      /* Allocate space */
      if(!(datafile = (char *)malloc(strlen(optarg) + 1))) {
	fprintf(stderr, "nsmon:Out of Memory\n");
	goto Cleanup;
      }	    
      sprintf(datafile, "%s\0", optarg) ;
      break ;
    case '?':
    default:
      fprintf (stderr, "%s: Unknown flag: %c\n", prognm, optarg);
      help() ;
      goto Cleanup ;
    }

    switch (ac - optind) {
      case 0:					/* default input file */
	break ;
      case 1:
	if(!(configfile = (char *)malloc(strlen(av[optind]) + 1))) {
	  fprintf(stderr, "nsmon:Out of Memory\n");
	  goto Cleanup;
	}
	sprintf(configfile, "%s\0", av[optind]) ;
	break ;
      default:
	fprintf (stderr, "%s Error: Too many 'hosts' files\n\n", prognm);
	help() ;
	goto Cleanup;
    }

    querydata = QUERYDATA;

    nocol_startup(&configfile, &datafile);

    if (fdout != 0)			/* Say, recovering from longjmp	*/
      close(fdout);

    if ( (fdout = open(datafile, O_RDWR|O_CREAT|O_TRUNC, DATAFILE_MODE)) < 0) {
      fprintf(stderr, "(%s) ERROR in open datafile ", prognm);
      perror (datafile);
      goto Cleanup ;
    }

    openeventlog() ;			/* Event logging */
    if (init_sites(fdout, configfile) == -1 )
      goto Cleanup ;

    if (!pollinterval)
      pollinterval = POLLINTERVAL ;		/* default value */

    if (debug)
      fprintf(stderr,
	"(debug) %s: CONFIGFILE= '%s', DATAFILE= '%s', QUERYDATA= '%s'\n",
	  prognm, configfile, datafile, querydata) ;

    /* poll_sites makes one complete pass over the list of nodes */
    while (1)			      		/* forever */
    {
	starttm = time((time_t *)NULL) ;	/* time started this cycle */
	if (poll_sites(fdout) == -1)		/* Polling error */
	  break ;

	if ( (polltime = time((time_t *)NULL) - starttm) < pollinterval)
	{
	  
	  if(debug)
	    fprintf(stderr, "(debug)%s: Sleeping for %u seconds.ZZZ...\n",
		    prognm, (unsigned)(pollinterval - polltime));
	 
	  sleep ((unsigned)(pollinterval - polltime));
	}
    }

    /* HERE ONLY IF ERROR */
Cleanup:
    nocol_done();

}	/***************** End of main *******************/
    
/*+ 
 * FUNCTION:
 * 	Brief usage
 */
help ()
{
    static char usage[] = " [-d (debug)] [-o <output file>] <configfile>\n";

    fprintf(stderr, "\nUSAGE: %s %s\n\n", prognm, usage);
    fprintf(stderr, "\tThis program sends queries to nameservers to verify\n");
    fprintf(stderr, "\ttheir operational status.\n");

    fprintf(stderr,"\tBy default, the list of nodes to monitor is in %s\n", 
	    configfile);
    fprintf(stderr,"\tbut can be changed on the command line.\n");

    fprintf(stderr,"\tThe 'node-file' format is:\n");
    fprintf(stderr,"\t\t POLLINTERVAL  <secs>\n\n");
    fprintf(stderr,"\t\t DOMAINNAME  <domain>\n");
    fprintf(stderr,"\t\t <node> <ip-address>  [TEST | AUTH]\n");
    fprintf(stderr,"\tThe program writes its pid in %s.pid and \n",prognm);
    fprintf(stderr,"\tif a new process starts, it kills the earlier one.\n");
    fprintf(stderr,"\tSending a SIGUSR1 signal toggles debugging\n\n");
    return (1);
}


/*
 * FUNCTION
 *
 *	init_sites
 *
 *	This function writes to the LSTFILE. All sites in the CONFIGFILE
 *	file are set to UNINIT status.
 *
 *	Careful while using 'localtime': the calue of the month varies from
 *	0 - 11 and hence has to be incremented for the correct month.
 */

init_sites(fdout, configfile)
     int fdout ;			/* Output file descriptor	*/
     char *configfile;			/* Filename of the configfile 	*/
{
    extern char sigtoprog[];		/* In pingmon.h			*/
    extern char *sender ;
    FILE *p_nodes ;
    EVENT v;				/* Defined in NOCOL.H		*/
    char record[MAXLINE];
    struct tm *loctime ;
    time_t locclock ;			/* Careful, don't use 'long'	*/
    int auth_wanted;			/* Set to 1 if want auth answer */
    struct site_info *lastnode = NULL, *newnode; /* for building linked list */

    if(site_info_list)			/* if not null, free it up */
      free_site_list(&site_info_list);	/* In case rereading confg file */

    if ((p_nodes = fopen(configfile, "r")) == NULL)
    {
	fprintf(stderr, "%s error (init_sites) ", prognm) ;
	perror (configfile);
	return (-1);
    }

    /*
     * Fill in the static data stuff
     */
    bzero (&v, sizeof(v)) ;
    locclock = time((time_t *)0);
    loctime = localtime((long *)&locclock);

    v.mon = loctime->tm_mon + 1;	v.day = loctime->tm_mday;
    v.hour = loctime->tm_hour;		v.min = loctime->tm_min;

    /*
     * in the following strncpy's, the NULL is already appended because
     * of the bzero, and copying one less than size of the arrays.
     */
    strncpy (v.sender, sender, sizeof(v.sender) - 1);

    strncpy (v.var.name, VARNM, sizeof (v.var.name) - 1);
    strncpy (v.var.units, VARUNITS, sizeof (v.var.units) - 1);
    v.var.value = 0 ; v.var.threshold = 0 ;	/* threshold not used */
    v.nocop = SETF_UPDOUN (v.nocop, n_UNKNOWN);	/* Set all to UNKNOWN	*/
    v.severity = E_INFO ;

    while(fgetLine(p_nodes,record,MAXLINE) > 0 ) 
    {
	char w1[MAXLINE], w2[MAXLINE], w3[MAXLINE];	/* Confg words	*/
	int rc;						/* return code	*/

	v.nocop = 0 ;				/* Init options to zero	*/
	*w1 = *w2 = *w3 = '\0' ;
	rc = sscanf(record,"%s %s %s", w1, w2, w3);
	if (rc == 0 || *w1 == '\0' || *w1 == '#')  /* Comment or blank 	*/
	  continue;

	if (strncmp(w1, "POLLINT", 7) == 0 || strncmp(w1, "pollint", 7) == 0)
	{
	    char *p; 					/* for strtol */
	    pollinterval = (u_long)strtol(w2, &p, 0) ;
	    if (p == w2)
	    {
		fprintf(stderr,"(%s): Error in format for POLLINTERVAL '%s'\n",
			prognm, w2) ;
		pollinterval = 0 ;		/* reset to default above */
	    }
	    continue ;
	}

	if (strncasecmp(w1,"DOMAINNAME",10) == 0)
	{
	    querydata = (char *)malloc(strlen(w2) + 2);
	    strcpy (querydata, w2);
	    /* This will be 'attached' to all sites under this domain */

	    if (debug)
	      fprintf(stderr, "(debug): Querydata changed to  '%s'\n", 
		      querydata);
	    
	    continue ;
	}

	/* These are <sitename> <addr> pairs. Some people might want to
	 * display the domain name instead of the site name...
	 */
#ifdef DISPLAY_DOMAIN
	strncpy(v.site.name, querydata, sizeof(v.site.name) - 1);
#else
	strncpy(v.site.name, w1, sizeof(v.site.name) - 1);	/* */
#endif
	strncpy(v.site.addr, w2, sizeof(v.site.addr) - 1);	/* no checks */

	if (inet_addr(w2) == -1)	/* bad address */
	{
	    fprintf(stderr,
		    "(%s): Error in address '%s' for site '%s', ignoring\n",
		    prognm, w2, w1);
	    continue ;
	}

	auth_wanted = 0;	/* Default AA not required */
	if (*w3 != '\0')	/* Some other keyword	*/
	{
	    if (strcmp(w3, "test") == 0 || strcmp(w3, "TEST") == 0)
	      v.nocop = v.nocop | n_TEST ;
	    else
	      /* 
	       * The word 'AUTH' can optionally be specified
	       * at the end of the site line
	       * This means that name query has to be authorative
	       */
	      if(strcasecmp(w3, "AUTH") == 0 )
	      {
		auth_wanted = 1; /* Authorative reply wanted */
		if(debug)
		  fprintf(stderr, "(debug)Authorative Answer for %s\n", w1);
	      }
	      else
		fprintf(stderr,"%s: Ignoring unknown keyword- %s\n",prognm,w3);
	}
	
	/*
	 * Create a site info node and add to the tail of linked list
	 */
	newnode = (struct site_info *)calloc(1, sizeof(struct site_info));
	if(!newnode)
	{
	  fprintf(stderr,"%s: Out of Memory ", prognm);
	  return -1;
	}

	newnode->domain = querydata;
	newnode->aa_wanted = auth_wanted;	
	newnode->next = NULL;
	
	if(!site_info_list)		  /* This is the first node(head) */
	  site_info_list = newnode;
	else
	  lastnode->next = newnode;
	lastnode = newnode;

	write (fdout, (char *)&v, sizeof(v)) ;
    }	/* end: while */

    if(debug)
    {
      fprintf(stderr, "(debug)Sites are :\n");
      for(lastnode = site_info_list; lastnode; lastnode = lastnode->next)
	fprintf(stderr, "(debug)Dom: %s  %s\n",
		lastnode->domain, (lastnode->aa_wanted)? "AA" : "Not AA");
    }
    fclose (p_nodes);			/* Not needed any more		*/    
    return(1);				/* All OK			*/
}		/* end:  init_sites()		*/

/*  */

/*
 * 	adapted from poll_sites.c of pingmon 1.15
 *
 * Makes one pass over all the hosts.
 *
 * Updates: 	Also deleted the wierd  sockad.sin_addr.S_un stuff.
 * 		Now passed the 'querydata' properly.
 *
 */

/* #defines for finish_status */
#define REACHED_EOF 1
#define READ_ERROR 2

poll_sites(fdout)
     int fdout;				/* Descriptors to open files	*/
{
  EVENT v;			    	/* described in nocol.h		*/
  struct tm *ltime ;    
  time_t locclock ;			/* careful: don't use 'long'	*/
  int status;		       		/* site status			*/
  int bufsize;				/* for reading in file */
  int sigpid;				/* PID of program to get signal	*/
  struct site_info *node = NULL;	/* Pointer to the site info LL */


  if ( lseek(fdout, (off_t)0, SEEK_SET) == -1) { /* rewind the file	*/
    perror (prognm);
    return (-1);
  }

  /* 
   * until end of the file or erroneous data... one entire pass
   */
  node = site_info_list;	/* Point to the head of Linked List */

  while ( (bufsize = read (fdout, (char *)&v, sizeof(v))) == sizeof(v) ) 
  {
   
    /*
     * Linked list has to match the list of events in file.
     * /
    if(!list_ptr)
    {
      fprintf(stderr,"%s: Datafile inconsistent with internal info.\n",prognm);
      return;
    }
    /*      switch(nsmon(v.site.addr, querydata, QUERYCLASS, QUERYTYPE,
		   QUERYTIMEOUT, auth_wanted, debug)) */

    switch(nsmon(v.site.addr, node->domain, 
		 QUERYCLASS, QUERYTYPE, QUERYTIMEOUT, node->aa_wanted, 
		 debug)) 
      {
       case ALL_OK:
	  status = 1;			/* '1' for all OK */
	  break;

       case NOT_AUTHORITATIVE:
	  if (!(node->aa_wanted)) 
	    status = 1;		/* Auth answ was not required anyway, so OK */
	  else
	    status = 0;
	  break;

       case NS_ERROR:
       case NO_NSDATA:
       default:
	  status = 0;
	  break;
      } /* switch */

      if (debug)
      {
	  fprintf(stderr, "(debug) %s: Nameserver status is %d/%s\n",
		  prognm, status, status ? "UP" : "DOWN" );
	  fflush(stderr);
      }

      update_event(&v, status, /* VALUE */ (u_long)status, maxseverity);
      
      /* Now rewind to start of present record and write to the file  */
     
      lseek(fdout,-(off_t)sizeof(v), SEEK_CUR);
      write (fdout, (char *)&v, sizeof(v));

      node = node->next;
     
  }	/* end of:    while (read..)	*/
    
  /**** Now determine why we broke out of the above loop *****/
    
  if (bufsize == 0)			/* reached end of the file	*/
    return (1);
  else {				/* error in output data file 	*/
    fprintf (stderr, "%s: Insufficient read data in output file", prognm);
    return (-1);
  }

}	/* end of:  poll_sites		*/

/*
 * free the linked list. Since two or more nodes might point to the same
 * domain name, take care to free up only once
 */

void free_site_list(si_list)
  struct site_info **si_list;
{
  struct site_info *curptr, *nextptr;
  char *prev_domain = NULL;

  for(curptr = *si_list; curptr ; curptr = nextptr)
  {
    if(prev_domain != curptr->domain)
    {
      /* 
       * Need to free the domain names also.
       * This can be tricky, as number of nodes have 'domain' pointing to
       * the same place.Hence, should be freed only once.
       */
      free(curptr->domain);
      prev_domain = curptr->domain;	/* store location of domain name */
    }
    nextptr = curptr->next;		/* store next cell before freeing */
    free(curptr);
  }
  *si_list = NULL;
}	/* end free_site_list() */

