/* $Id: neb-wipe.c,v 1.13 2006/03/10 18:02:01 stoehr Exp $ */

/*
 * Copyright (C) 2006 by Florian Stoehr <ich@florian-stoehr.de>
 * 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms,  with  or
 * without  modification,  are permitted provided that the fol-
 * lowing conditions are met:
 * 
 *  (1)   Redistributions of source code must retain the  above
 *        copyright  notice,  this  list  of conditions and the
 *        following disclaimer.
 * 
 *  (2)   Redistributions in binary  form  must  reproduce  the
 *        above  copyright  notice, this list of conditions and
 *        the following disclaimer in the documentation  and/or
 *        other materials provided with the distribution.
 * 
 *  (3)   All  advertising materials mentioning features or use
 *        of this software must display the following  acknowl-
 *        edgement: This product includes software developed by
 *        Florian Stoehr
 * 
 *  (4)   The name of Florian Stoehr may not be used to endorse
 *        or  promote products derived from this software with-
 *        out specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY FLORIAN STOEHR  ''AS  IS''  AND
 * ANY  EXPRESS  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIM-
 * ITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND  FIT-
 * NESS  FOR  A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT
 * SHALL FLORIAN STOEHR BE LIABLE  FOR  ANY  DIRECT,  INDIRECT,
 * INCIDENTAL,  SPECIAL,  EXEMPLARY,  OR  CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO,  PROCUREMENT  OF  SUBSTITUTE
 * GOODS  OR  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSI-
 * NESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF  LIA-
 * BILITY,  WHETHER  IN  CONTRACT,  STRICT  LIABILITY,  OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY  WAY  OUT
 * OF  THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSI-
 * BILITY OF SUCH DAMAGE.
 */
 
#include <sys/param.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

void PrintUsageExit(void);
void UpCheck(void);
void UpFinished(void);
void PassPrepare(const char *);
void WritePassRandom(const char *);
void WritePass(const char *, unsigned char, unsigned char, unsigned char);

#define BSIZE 64*1024*3
#define VERSION "1.0"

unsigned char buffer[BSIZE];
off_t i_DiskSize, i_CurPos, i_LastPos, i_Step;
time_t t_Start;
int i_FD;

/*
 * Print usage information, then exit program
 */
void
PrintUsageExit(void)
{
	printf("neb-wipe - Erase harddisk partitions in a very secure manner,\n");
	printf("           using the 35-pass Gutmann method\n");
	printf("           Version %s\n", VERSION);
	printf("\n");
	printf("Usage is: neb-wipe [ -r count ] device\n");
	printf("          where device is the desired partition, e.g. \"sd0a\"\n");
	printf("          If the -r option is given, no Gutmann patterns are used;\n");
	printf("          Instead, neb-wipe writes the given amount of random\n");
	printf("          patterns to the disk.\n\n");		   
			   
	exit(EXIT_FAILURE);
}

/*
 * Update counters; Display ETA if needed (every 5 percent)
 */
void
UpCheck(void)
{
	char sbuf[32], tbuf[16];
	time_t t_Diff, t_Left, t_Now;
	double curp;
	int i_Hour, i_Min, i_Sec;
	
	i_CurPos += BSIZE;
	
	if ((i_CurPos - i_LastPos) >= i_Step) {
		/* 5 percent gone. Calculate ETA and give a message */
		curp = ((double)i_CurPos / (double)i_DiskSize) * 100.0;
		i_LastPos = i_CurPos;

		t_Now = time(NULL);
		t_Diff = t_Now - t_Start;
		i_Sec = (int)(((double)t_Diff / curp) * 100.0) - t_Diff;
		i_Hour = i_Sec / 3600;
		i_Sec -= i_Hour * 3600;
		i_Min = i_Sec / 60;
		i_Sec -= i_Min * 60;
		
		memset(sbuf, 0x00, 32);
		
		if (i_Hour < 10)
			strcat(sbuf, "0");
		sprintf(tbuf, "%d", i_Hour);		
		strcat(sbuf, tbuf);
		strcat(sbuf, ":");
		
		if (i_Min < 10)
			strcat(sbuf, "0");
		sprintf(tbuf, "%d", i_Min);		
		strcat(sbuf, tbuf);
		strcat(sbuf, ":");
		
		if (i_Sec < 10)
			strcat(sbuf, "0");
		sprintf(tbuf, "%d", i_Sec);
		strcat(sbuf, tbuf);

		printf("    %2.0f%% done - will finish in about %s\n", curp, sbuf);
		fflush(stdout);
	}		
}

/*
 * Finished a pass. Print statistics.
 */
void
UpFinished(void)
{
	char sbuf[32], tbuf[16];
	int i_Hour, i_Min, i_Sec;
	
	i_Sec = time(NULL) - t_Start;
	
	i_Hour = i_Sec / 3600;
	i_Sec -= i_Hour * 3600;
	i_Min = i_Sec / 60;
	i_Sec -= i_Min * 60;

	memset(sbuf, 0x00, 32);
		
	if (i_Hour < 10)
		strcat(sbuf, "0");
	sprintf(tbuf, "%d", i_Hour);		
	strcat(sbuf, tbuf);
	strcat(sbuf, ":");
		
	if (i_Min < 10)
		strcat(sbuf, "0");
	sprintf(tbuf, "%d", i_Min);		
	strcat(sbuf, tbuf);
	strcat(sbuf, ":");
		
	if (i_Sec < 10)
		strcat(sbuf, "0");
	sprintf(tbuf, "%d", i_Sec);
	strcat(sbuf, tbuf);

	printf("   100%% done - Finished in %s\n", sbuf);
	fflush(stdout);
}

/*
 * Prepare for the next write pass
 */
void
PassPrepare(const char *s)
{
	printf("%s\n", s);
	fflush(stdout);
	
	lseek(i_FD, 0, SEEK_SET);
	
	i_CurPos = 0;
	i_LastPos = 0;
	t_Start = time(NULL);
}

/*
 * Invoke one pass of random writing
 */
void
WritePassRandom(const char *s)
{
	unsigned long *lptr;
	int i;
	
	PassPrepare(s);	
	
	for (;;) {
		/* Fill buffer with random data */
		lptr = (unsigned long*)&buffer;
		
		for (i = 0; i < BSIZE / sizeof(unsigned long); i++)
			 *lptr++ = random();
	
		if (write(i_FD, buffer, BSIZE) != BSIZE)
			break;
			
		UpCheck();
	}
	
	UpFinished();	
}

/*
 * Invoke one pass of pattern writing
 */
void
WritePass(const char *s, unsigned char m1, unsigned char m2, unsigned char m3)
{
	int i;

	PassPrepare(s);
	
	/* Fill with given pattern */
	for (i = 0; i < BSIZE / 3; i++) {
		*(buffer + (i * 3)) = m1;
		*(buffer + (i * 3) + 1) = m2;
		*(buffer + (i * 3) + 2) = m3;
	}
	
	for (;;) {
		if (write(i_FD, buffer, BSIZE) != BSIZE)
			break;
			
		UpCheck();
	}	

	UpFinished();	
}

/*
 * neb-wipe: Programm to wipe off harddisk partitions in a secure manner
 */
int
main(int argc, char **argv)
{
	char s[128], s_Device[MAXPATHLEN+1], tmp[16];
	int ch;
	unsigned long i_GB, seeder;
	long num;
	int i, i_RD;
	char *ep;
	
	setprogname(argv[0]);
	
	num = -1;
	
	/* Process command-line options */	
	while ((ch = getopt(argc, argv, "r:h")) != -1) {
		switch (ch) {
		case 'r':
			errno = 0;
			num = strtol(optarg, &ep, 10);
			if (num <= 0 || *ep != '\0' || (errno == ERANGE &&
			    (num == LONG_MAX || num == LONG_MIN))) {
				printf("ERROR: Illegal repeat count -- %s\n\n", optarg);
				PrintUsageExit();
			}
			break;
		case '?':
		case 'h':
		default:
			PrintUsageExit();
			/* NOTREACHED */
		}
	}
	
	argc -= optind;
	argv += optind;
	
	if (argc < 1)
		PrintUsageExit();

	/* Data setup */
	i_FD = opendisk(argv[0], O_WRONLY, s_Device, MAXPATHLEN, 0);
	
	if (i_FD == -1)	{	
		printf("ERROR opening device. Make sure you are root.\n");

		exit(EXIT_FAILURE);
	}
	
	/* Get disk size to show progress / ETA */
	i_DiskSize = lseek(i_FD, 0, SEEK_END);	
	i_GB = (i_DiskSize / (1024 * 1024 * 1024)) + 1;
	i_Step = (i_DiskSize / 20);
	
	/*
	 * Use an entropy-pool value as random seed; Gathering all
	 * values through /dev/urandom would be a lot nicer but
	 * unfortunately this is way too slow for the purpose of 
	 * filling gigabytes :-(
	 */
	i_RD = open("/dev/urandom", O_RDONLY);
	
	if (i_RD == -1) {
		printf("ERROR opening /dev/urandom. Make sure you have rnd(4)\n \
			enabled and an existing devicenode for urandom.\n");
		close(i_FD);
		
		exit(EXIT_FAILURE);
	}
	
	read(i_RD, &seeder, sizeof(unsigned long));
	close(i_RD);	
	srandom(seeder);

	printf("neb-wipe v%s\n", VERSION);
	printf("Size of %s is %u GB\n\n", s_Device, i_GB);
	
	if (num == -1) {
		/* 
		 * Perform the 35-pass Gutmann method
		 */
		WritePassRandom("01/35: Random seq 1");
		WritePassRandom("02/35: Random seq 2");
		WritePassRandom("03/35: Random seq 3");
		WritePassRandom("04/35: Random seq 4");
		WritePass("05/35: Pattern seq 1", 0x55, 0x55, 0x55);
		WritePass("06/35: Pattern seq 2", 0xAA, 0xAA, 0xAA);
		WritePass("07/35: Pattern seq 3", 0x92, 0x49, 0x24);
		WritePass("08/35: Pattern seq 4", 0x49, 0x24, 0x92);
		WritePass("09/35: Pattern seq 5", 0x24, 0x92, 0x49);
		WritePass("10/35: Pattern seq 6", 0x00, 0x00, 0x00);
		WritePass("11/35: Pattern seq 7", 0x11, 0x11, 0x11);
		WritePass("12/35: Pattern seq 8", 0x22, 0x22, 0x22);
		WritePass("13/35: Pattern seq 9", 0x33, 0x33, 0x33);
		WritePass("14/35: Pattern seq 10", 0x44, 0x44, 0x44);
		WritePass("15/35: Pattern seq 11", 0x55, 0x55, 0x55);
		WritePass("16/35: Pattern seq 12", 0x66, 0x66, 0x66);
		WritePass("17/35: Pattern seq 13", 0x77, 0x77, 0x77);
		WritePass("18/35: Pattern seq 14", 0x88, 0x88, 0x88);
		WritePass("19/35: Pattern seq 15", 0x99, 0x99, 0x99);
		WritePass("20/35: Pattern seq 16", 0xAA, 0xAA, 0xAA);
		WritePass("21/35: Pattern seq 17", 0xBB, 0xBB, 0xBB);
		WritePass("22/35: Pattern seq 18", 0xCC, 0xCC, 0xCC);
		WritePass("23/35: Pattern seq 19", 0xDD, 0xDD, 0xDD);
		WritePass("24/35: Pattern seq 20", 0xEE, 0xEE, 0xEE);
		WritePass("25/35: Pattern seq 21", 0xFF, 0xFF, 0xFF);
		WritePass("26/35: Pattern seq 22", 0x92, 0x49, 0x24);
		WritePass("27/35: Pattern seq 23", 0x49, 0x24, 0x92);
		WritePass("28/35: Pattern seq 24", 0x24, 0x92, 0x49);
		WritePass("29/35: Pattern seq 25", 0x6D, 0xB6, 0xDB);
		WritePass("30/35: Pattern seq 26", 0xB6, 0xDB, 0x6D);
		WritePass("31/35: Pattern seq 27", 0xDB, 0x6D, 0xB6);
		WritePassRandom("32/35: Random seq 5");
		WritePassRandom("33/35: Random seq 6");
		WritePassRandom("34/35: Random seq 7");
		WritePassRandom("35/35: Random seq 8");
	} else {
		/*
		 * Perform the given number of random rewrite passes
		 */
		for (i = 0; i < num; i++) {
			memset(s, 0x00, 32);

			if ((i + 1) < 10)
				strcat(s, "0");
			sprintf(tmp, "%d", i + 1);
			strcat(s, tmp);
			strcat(s, "/");
			
			if (num < 10)
				strcat(s, "0");
			sprintf(tmp, "%d", num);
			strcat(s, tmp);
			strcat(s, ": Random seq ");
			sprintf(tmp, "%d", i + 1);
			strcat(s, tmp);

			WritePassRandom(s);
		}			
	}

	printf("\nDone\n");
	
	close(i_FD);
	
	exit(EXIT_SUCCESS);
}
