/* $NetBSD$ */

/*
 * Copyright (c) 1995 Mark Brinicombe
 * Copyright (c) 1995 Melvin Tang-Richardson
 * Copyright (c) 1995 Ian Biondani
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the RiscBSD kernel teamA
 * 4. The name of the company nor the name of the author may be used to
 *    endorse or promote products derived from this software without specific
 *    prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * RiscBSD kernel project
 *
 * setdisplay.c
 *
 * Sets display parameters for a console
 *
 *    $Id$
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <machine/vidc.h>
#include <machine/vconsole.h>

#include "pathnames.h"

#define ABS(x)	((x) < 0 ? -(x) : (x))

#define BUF_SIZE 320
#define MONDEF_ERR_LEN 40

#define MONITOR_TITLE_LEN 40
#define MODE_TITLE_LEN 20		/* From Display manager */
#define MODE_NAME_LEN MODE_TITLE_LEN
#define PRINT

int verbose = 0;
int xres = -1;
int yres = -1;
int max_pixelrate = 200000;
int pixelrate = 0;
int framerate = 0;
int bitsperpixel = 8;

#ifdef RISCOS
void bzero(char *ptr, int len)
{
	int counter;

	for (counter = 0; counter < len; counter++)
		ptr[coutner]=0;
}
#endif

int file_format;
int DPMS_state;
char monitor_title[40];
char mondef_err[MONDEF_ERR_LEN];
int mondef_defining;

struct amode {
	char mode_name[MODE_TITLE_LEN];
	int x_res, y_res;
	int pixel_rate;
	int hswr, hbsr, hdsr, hder, hber, hcr;
	int vswr, vbsr, vdsr, vder, vber, vcr;
	int sync_pol;
	int dummy;
	int dummy2;
	int framerate;
} amode;

struct amode chosen; /* Stores the current chosen mode */

void
copy_to_vidc_struct(vidc, amode)
	struct vidc_mode *vidc;
	struct amode amode;
{
	vidc->pixel_rate = amode.pixel_rate;
	vidc->hswr = amode.hswr;
	vidc->hbsr = amode.hbsr;
	vidc->hdsr = amode.hdsr;
	vidc->hder = amode.hder;
	vidc->hber = amode.hber;
	vidc->hcr  = amode.hcr;
	vidc->vswr = amode.vswr;
	vidc->vbsr = amode.vbsr;
	vidc->vdsr = amode.vdsr;
	vidc->vder = amode.vder;
	vidc->vber = amode.vber;
	vidc->vcr  = amode.vcr;
	vidc->bitsperpixel = bitsperpixel;
	/*vidc->sync_pol = amode.sync_pol;*/
}


void
mondef_newdef()
{
	file_format = -1;
	DPMS_state = -1;
	monitor_title[0] = 0;
	mondef_defining = 0;
}


int
mondef_parse(fptr)
	FILE *fptr;
{
	char buf[BUF_SIZE];
    
	mondef_newdef();

	/* Parse all the lines in this file */
    
	while (!feof(fptr)) {
		if (fgets(buf, BUF_SIZE, fptr) != NULL)
			if (mondef_parseline(buf) == -1)
				return(-1);
	}
}


int
mondef_startmode(line)
	char *line;
{
	/*
	 * Are we already in a mode definition ?
	 */
     
	if (mondef_defining == 1) {
		strcpy(mondef_err, "Wrong context for keyword startmode");
		return -1;
	} else {
	        int *ptr;
		int counter;
		mondef_defining = 1;
	
		for (counter = 0; counter < sizeof(struct amode); counter++)
			ptr[counter] = -1;
        
		return(0);
	}
}


int
mondef_mode_name(line)
	char *line;
{
	if (strlen (amode.mode_name) > MODE_NAME_LEN)
		strncpy(amode.mode_name, line + 10, MODE_NAME_LEN);
	else
		strcpy(amode.mode_name, line + 10);
	return(0);
}


int
mondef_x_res(line)
	char *line;
{
	amode.x_res = atoi(line+6);
	return(0);
}


int mondef_y_res(line)
	char *line;
{
	amode.y_res = atoi(line+6);
	return(0);
}


int mondef_pixel_rate(line)
	char *line;
{
	amode.pixel_rate = atoi(line+11);
	return(0);
}


int mondef_h_timings(line)
	char *line;
{
	sscanf(line, "h_timings:%i,%i,%i,%i,%i,%i", &amode.hswr, &amode.hbsr,
	    &amode.hdsr, &amode.hder, &amode.hber, &amode.hcr);
	return(0);
}


int
mondef_v_timings(line)
	char *line;
{
	sscanf(line, "v_timings:%i,%i,%i,%i,%i,%i", &amode.vswr, &amode.vbsr,
	    &amode.vdsr, &amode.vder, &amode.vber, &amode.vcr);
	return(0);
}


int
mondef_sync_pol(line)
	char *line;
{
	amode.sync_pol = atoi(line+9);
	return(0);
}


int
mondef_endmode(line)
	char *line;
{
	int *ptr;
	int counter;
	mondef_defining = 1;

	ptr = (int *) &amode;
 
	/*
	 * Check to see if all the parameters have been specified
	 */
    
	for (counter = 0; counter < sizeof(struct amode); counter++) {
		if (ptr[counter] == -1) {
			strncpy(mondef_err, "Incorrect mode definition", MONDEF_ERR_LEN);
			return(-1);
		}
	}

	amode.framerate = (int)(((float)amode.pixel_rate * 1000.0 /
	    (float)(amode.hswr + amode.hbsr + amode.hdsr + amode.hder + amode.hber + amode.hcr) /
	    (float)(amode.vswr + amode.vbsr + amode.vdsr + amode.vder + amode.vber + amode.vcr))
	     + 0.5);

	if ((int)(amode.pixel_rate / 1000) > max_pixelrate)
		return;
	if ((xres != amode.x_res) || (yres != amode.y_res))
		return;

	if (verbose)
		printf("found : pixelrate=%d framerate=%d\n", amode.pixel_rate, amode.framerate);

	if ((chosen.x_res == -1) || (chosen.y_res == -1))
		chosen = amode;
	if (pixelrate && (ABS((amode.pixel_rate/1000) - pixelrate) < ABS((chosen.pixel_rate/1000) - pixelrate)))
		chosen = amode;		
	if (framerate && (ABS((amode.framerate) - framerate) < ABS((chosen.framerate) - framerate)))
		chosen = amode;		

	if (verbose)
		printf("chosen : chosen.pixel_rate=%d framerate=%d\n",
		    chosen.pixel_rate, chosen.framerate);
}


void
mondef_null_crlf(text)
	char *text;
{
	while (*text) {
		if (*text == '\r' || *text == '\n') {
			*text = 0;
			break;
		}
		++text;
	}
}


int
mondef_parseline(line)
	char *line;
{

	while (*line == ' ' || *line == '\t')
		++line;

	mondef_null_crlf(line);

	 /* Blank lines */
	if (strlen(line) == 0)
		return(0);
        
	/* Comments */
    
	if (line[0]=='#')
		return 0;
        
	/*
	 * The Display Manager Application in RiscOS expects the next 3 fields
	 * to occur sequentially
	 */
     
	/* file_format */

	if (file_format == -1) {
		if (strncmp(line, "file_format", 10)) {
			strcpy(mondef_err, "file_format expected");
			return -1;
		} else {
			file_format = atoi(line + 12);
			return 0;
		}
	}

	/* monitor_title */

	if (monitor_title[0] == 0) {
		if (strncmp(line, "monitor_title", 13)) {
			strcpy(mondef_err, "monitor_title expected");
			return(-1);
		} else {
			strncpy(monitor_title, line + 14, MONITOR_TITLE_LEN); 
#ifdef PRINT
			if (verbose)
				printf("monitor_title = %s\n", monitor_title);
#endif
			return 0;
		}
	}

	/* DPMS_state */

	if (DPMS_state == -1) {
		if (strncmp(line, "DPMS_state", 10)) {
			strcpy(mondef_err, "DPMS_state expected");
			return(-1);
		} else {
			DPMS_state = atoi(line+12);
#ifdef PRINT
			if (verbose)
				printf("DPMS_state = %i\n", DPMS_state);
#endif
			return(0);
		}
	}


	if (!strncmp(line, "mode_name", 9))    	return(mondef_mode_name	(line));
	if (!strncmp(line, "x_res", 5))     	return(mondef_x_res	(line));
	if (!strncmp(line, "y_res", 5))     	return(mondef_y_res	(line));
	if (!strncmp(line, "pixel_rate", 10))	return(mondef_pixel_rate(line));
	if (!strncmp(line, "h_timings", 9)) 	return(mondef_h_timings	(line));
	if (!strncmp(line, "v_timings", 9)) 	return(mondef_v_timings	(line));
	if (!strncmp(line, "sync_pol", 8)) 	return(mondef_sync_pol	(line));
	if (!strncmp(line, "endmode", 7)) 	return(mondef_endmode	(line));
}

void
usage()
{
	printf("Usage:\tsetvideo -x xres -y yres [-m monitorfile] [-d device] [-c bitsperpixel]\n");
	printf("or\tsetvideo -x xres -y yres [-m monitorfile] [-d device] [-p pixelrate] [-c bitsperpixel]\n");
	printf("or\tsetvideo -x xres -y yres [-m monitorfile] [-d device] [-f framerate] [-c bitsperpixel]\n");
	printf("or\tsetvideo [-h]\n");
	printf("Default device:\t\t%s\n", DEVICENAME);
	printf("Default monitorfile:\t%s\n", MONITORFILE);
	exit(0);
}

int
main(argc, argv)
	int argc;
	char *argv[];
{
	FILE *file;
	int ch;
	char *modefile;
	char *devicename;
	struct vidc_mode vidc;
	int fd;

	devicename = DEVICENAME;
	modefile = MONITORFILE;

	while ((ch = getopt(argc, argv, "m:x:y:p:d:c:f:vh")) != EOF) {
		switch(ch) {
		case 'x':
			xres = atoi(optarg);
			break;
		case 'y':
			yres = atoi(optarg);
			break;
		case 'p':
			pixelrate = atoi(optarg);
			break;
		case 'c':
			bitsperpixel = atoi(optarg);
			break;
		case 'd':
			devicename = optarg;
			break;
		case 'm':
			modefile = optarg;
			break;
		case 'f':
			framerate = atoi(optarg);
			break;
		case 'v':
		   	verbose = 1;
		   	break;
		case 'h':
		   	usage();
		   	break;
		}
	}

	argc -= optind;
	argv += optind;

	if ((xres < 0) || (yres < 0))
		usage();

	if (pixelrate && framerate) {
		fprintf(stderr, "Cannot specify both pixelrate and framerate\n");
		exit(1);
	}

	chosen.x_res = -1;
	chosen.y_res = -1;
	chosen.pixel_rate = 0;
	chosen.framerate = 0;

	if ((file = fopen(modefile, "r")) == 0) {
		perror("Can't open monitor definition file: ");
		exit(1);
	}

	if (mondef_parse(file) == -1) {
		printf("setdisplay: %s\n", mondef_err);
		fclose(file);
		exit(1);
	}

	fclose(file);

	if ((chosen.x_res == -1) && (chosen.y_res == -1)) {
		fprintf(stderr, "No suitable mode found in %s\n", modefile);
		exit(1);
	}

	if (verbose)
		printf("xres = %i\nyres = %i\nPixel Rate=%i\n",
		    chosen.x_res, chosen.y_res, chosen.pixel_rate);

	copy_to_vidc_struct(&vidc, chosen);

	fd = open(devicename, O_RDWR);
	if (fd < 0) {
		perror("Can't open device");
		exit(0);
	}

	if (ioctl(fd, CONSOLE_MODE, &vidc) != 0)
		perror("setdisplay: could not change mode");
	close(fd);
}
