/* $NetBSD$ */

/*
 * Copyright (c) 1994 Mark Brinicombe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Mark Brinicombe.
 * 4. The name of the company nor the name of the author may be used to
 *    endorse or promote products derived from this software without specific
 *    prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTERS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Generates a waveform for the beep device
 *
 * Modified 28/11/96 to work for 16bit sound by Ale.
 */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <string.h>
#include <math.h>

#include <machine/beep.h>
#include "lin2log.h"

int make_waveform __P((unsigned char *table, int cycles, int volume, int prec));

int
main(argc, argv)
	int argc;
	char *argv[];
{
	int file;
	int loop = 1;
	int cycles = 0;
	int rate = 0;
	int volume = 100;
	int prec = 1;
	unsigned char table[0x1000];
	struct wavebuffer wave;

	while (loop < argc) {
		if (strcmp(argv[loop], "-v") == 0) {
			++loop;
			volume = atoi(argv[loop]);
			if (volume < 0 || volume > 100) {
				fprintf(stderr, "Volume out of range\n");
				exit(1);
			}
		}
		if (strcmp(argv[loop], "-r") == 0) {
			++loop;
			rate = atoi(argv[loop]);
			if (rate < 3 || rate > 255) {
				fprintf(stderr, "Rate out of range\n");
				exit(1);
			}
		}
		if (strcmp(argv[loop], "-c") == 0) {
			++loop;
			cycles = atoi(argv[loop]);
			if (cycles < 16 || cycles > 511) {
				fprintf(stderr, "Cycles out of range\n");
				exit(1);
			}
		}
		if (strcmp(argv[loop], "-l") == 0) {
			prec=2;
		}
		if (strcmp(argv[loop], "-h") == 0) {
			printf("Syntax: mkbeep -v <volume> -r <rate> -c <cycles> [-l]\n\n");
			printf("Volume is specified in %%\n");
			printf("Rate is VIDC sample rate in microseconds\n");
			printf("-l for 16 bit hardware, default is 8 bit\n");
			exit(0);
		}
		++loop; 
	}
        
	file = open("/dev/beep", O_RDWR);
	if (file == -1) {
		perror("open");
		exit(1);
	}

	/* If no args just beep */

	if (argc == 1) {
		ioctl(file, BEEP_GENERATE, NULL);
	} else {
		if (rate != 0)
			ioctl(file, BEEP_SETRATE, (void *)rate);

		if (cycles != 0) {
			make_waveform(table, cycles, volume, prec);
			wave.size = 0x1000;
			wave.addr = (caddr_t)table;
			ioctl(file, BEEP_SET, &wave);
		} else if (volume != 0) {
			make_waveform(table, 128, volume, prec);
			wave.size = 0x1000;
			wave.addr = (caddr_t)table;
			ioctl(file, BEEP_SET, &wave);
		}
	}

	close(file);
	printf("\n");
}
  

int
make_waveform(table, cycles, volume, prec)
	unsigned char *table;
	int cycles;
	int volume;
	int prec;
{
	int loop;
	int value;
	double sine;
	int level;

	printf("Building new waveform ");
	fflush(stdout);
	for (loop = 0; loop < 0x1000; loop += prec) {
		if ((loop & 0xff) == 0) {
			putchar('.');
			fflush(stdout);
		}
		if (loop < 0x100)
			level = loop * 16;
		else if (loop < 0x800)
			level = 0xfff;
		else
			level = 0xfff - (loop - 0x800) * 2;
		sine = sin(((float)loop * 360.0 * prec * cycles / 4096.0) / 180.0 * M_PI);
		value = (int)((double)level * (double)volume / 100.0 * sine);
		table[loop] = lin_to_log[value & 0x1fff];
		if (prec == 1) 
			table[loop] = lin_to_log[value & 0x1fff];
		else {
		  	table[loop] = (value & 0xff); 
		  	table[loop+1] = (value & 0xff00) >> 8; 
		}
	}
}

/* End of mkbeep.c */
