use material_colors::{
    blend::harmonize,
    color::Argb,
    dynamic_color::{DynamicScheme, MaterialDynamicColors},
    hct::{Cam16, Hct},
    image::{FilterType, ImageReader},
    quantize::{Quantizer, QuantizerCelebi},
    scheme::variant::{
        SchemeContent, SchemeExpressive, SchemeFidelity, SchemeFruitSalad, SchemeMonochrome,
        SchemeNeutral, SchemeRainbow, SchemeTonalSpot, SchemeVibrant,
    },
    score::Score,
    theme::{ColorGroup, CustomColor, CustomColorGroup},
};

use color_eyre::{eyre::WrapErr, Report};
use colorsys::{Hsl, Rgb};
use std::str::FromStr;

use crate::FilterType as OwnFilterType;
use crate::{color::math::get_color_distance_lab, scheme::SchemeTypes};

use material_colors::image::AsPixels;

#[derive(clap::Parser, Debug, Clone)]
pub enum ColorFormat {
    Hex { string: String },
    Rgb { string: String },
    Hsl { string: String },
}

impl ColorFormat {
    pub fn get_string(&self) -> &String {
        match self {
            ColorFormat::Hex { string } => string,
            ColorFormat::Rgb { string } => string,
            ColorFormat::Hsl { string } => string,
        }
    }
}

#[derive(clap::Subcommand, Debug, Clone)]
pub enum Source {
    /// The image to use for generating a color scheme
    Image { path: String },

    #[cfg(feature = "web-image")]
    /// The image to fetch from web and use for generating a color scheme
    WebImage { url: String },

    /// The source color to use for generating a color scheme
    #[clap(subcommand)]
    Color(crate::color::color::ColorFormat),

    /// The json file to use and import for templates
    Json { path: String },
}

impl Source {
    pub fn is_image(&self) -> bool {
        match self {
            Source::Image { path: _ } => true,
            _ => false,
        }
    }

    pub fn is_json(&self) -> bool {
        match self {
            Source::Json { path: _ } => true,
            _ => false,
        }
    }
}

#[derive(serde::Serialize, serde::Deserialize, Debug, Clone)]
pub struct ColorDefinition {
    pub name: String,
    pub color: String,
}

#[derive(serde::Serialize, serde::Deserialize, Debug, Clone)]
#[serde(untagged)]
pub enum OwnCustomColor {
    Color(String),
    Options { color: String, blend: bool },
}

impl OwnCustomColor {
    pub fn to_custom_color(
        &self,
        name: String,
    ) -> Result<material_colors::theme::CustomColor, material_colors::error::Error> {
        Ok(match self {
            OwnCustomColor::Color(color) => material_colors::theme::CustomColor {
                value: Argb::from_str(color)?,
                blend: true,
                name,
            },
            OwnCustomColor::Options { color, blend } => material_colors::theme::CustomColor {
                value: Argb::from_str(color)?,
                blend: *blend,
                name,
            },
        })
    }
}

pub fn get_source_color(
    source: &Source,
    resize_filter: &Option<OwnFilterType>,
) -> Result<Argb, Report> {
    use crate::color::color;

    let filter: FilterType = match resize_filter {
        Some(v) => FilterType::from(v),
        None => FilterType::from(&OwnFilterType::Triangle),
    };

    let source_color: Argb = match source {
        Source::Image { path } => {
            info!("Opening image in <d><u>{}</>", path);
            color::get_source_color_from_image(path, filter)
                .wrap_err(format!("Could not get source color from image: {}", path))?
        }
        #[cfg(feature = "web-image")]
        Source::WebImage { url } => {
            info!("Fetching image from <d><u>{}</>", url);
            color::get_source_color_from_web_image(url, filter)
                .expect("Could not get source color from web image")
        }
        Source::Color(color) => color::get_source_color_from_color(color).wrap_err(format!(
            "Could not get source color from color {}",
            color.get_string()
        ))?,
        Source::Json { path: _ } => unreachable!(),
    };
    Ok(source_color)
}

pub fn get_source_color_from_image(path: &str, filter_type: FilterType) -> Result<Argb, Report> {
    let mut original = ImageReader::open(path)?;
    let image = original.resize(112, 112, filter_type);
    let pixels: Vec<Argb> = image
        .as_pixels()
        .iter()
        .copied()
        .filter(|argb| argb.alpha == 255)
        .collect();
    let mut result = QuantizerCelebi::quantize(&pixels, 128);

    result
        .color_to_count
        .retain(|&argb, _| Cam16::from(argb).chroma >= 5.0);

    let ranked = Score::score(&result.color_to_count, None, None, None);

    Ok(ranked[0])
}

#[cfg(feature = "web-image")]
pub fn get_source_color_from_web_image(url: &str, filter_type: FilterType) -> Result<Argb, Report> {
    let bytes = reqwest::blocking::get(url)?.bytes()?;
    Ok(ImageReader::extract_color(
        ImageReader::read(&bytes)?.resize(128, 128, filter_type),
    ))
}

pub fn get_source_color_from_color(color: &ColorFormat) -> Result<Argb, Report> {
    match color {
        ColorFormat::Hex { string } => {
            Ok(Argb::from_str(string).expect("Invalid hex color string provided"))
        }
        ColorFormat::Rgb { string } => {
            let rgb = Rgb::from_str(string).expect("Invalid rgb color string provided");
            Ok(Argb {
                red: rgb.red() as u8,
                green: rgb.green() as u8,
                blue: rgb.blue() as u8,
                alpha: 255,
            })
        }
        ColorFormat::Hsl { string } => {
            let rgb: Rgb = Hsl::from_str(string)
                .expect("Invalid hsl color string provided")
                .into();
            Ok(Argb {
                red: rgb.red() as u8,
                green: rgb.green() as u8,
                blue: rgb.blue() as u8,
                alpha: 255,
            })
        }
    }
}

pub fn generate_dynamic_scheme(
    scheme_type: &Option<SchemeTypes>,
    source_color: Argb,
    is_dark: bool,
    contrast_level: Option<f64>,
) -> DynamicScheme {
    match scheme_type.unwrap_or(SchemeTypes::SchemeContent) {
        SchemeTypes::SchemeContent => {
            SchemeContent::new(Hct::new(source_color), is_dark, contrast_level).scheme
        }
        SchemeTypes::SchemeExpressive => {
            SchemeExpressive::new(Hct::new(source_color), is_dark, contrast_level).scheme
        }
        SchemeTypes::SchemeFidelity => {
            SchemeFidelity::new(Hct::new(source_color), is_dark, contrast_level).scheme
        }
        SchemeTypes::SchemeFruitSalad => {
            SchemeFruitSalad::new(Hct::new(source_color), is_dark, contrast_level).scheme
        }
        SchemeTypes::SchemeMonochrome => {
            SchemeMonochrome::new(Hct::new(source_color), is_dark, contrast_level).scheme
        }
        SchemeTypes::SchemeNeutral => {
            SchemeNeutral::new(Hct::new(source_color), is_dark, contrast_level).scheme
        }
        SchemeTypes::SchemeRainbow => {
            SchemeRainbow::new(Hct::new(source_color), is_dark, contrast_level).scheme
        }
        SchemeTypes::SchemeTonalSpot => {
            SchemeTonalSpot::new(Hct::new(source_color), is_dark, contrast_level).scheme
        }
        SchemeTypes::SchemeVibrant => {
            SchemeVibrant::new(Hct::new(source_color), is_dark, contrast_level).scheme
        }
    }
}

pub fn make_custom_color(
    color: CustomColor,
    scheme_type: &Option<SchemeTypes>,
    source_color: Argb,
    contrast_level: Option<f64>,
) -> CustomColorGroup {
    let value = if color.blend {
        harmonize(color.value, source_color)
    } else {
        color.value
    };

    let light = generate_dynamic_scheme(scheme_type, value, false, contrast_level);
    let dark = generate_dynamic_scheme(scheme_type, value, true, contrast_level);

    // debug!("custom_color: {:#?}", &custom_color);
    CustomColorGroup {
        color,
        value,
        light: ColorGroup {
            color: MaterialDynamicColors::primary().get_argb(&light),
            on_color: MaterialDynamicColors::on_primary().get_argb(&light),
            color_container: MaterialDynamicColors::primary_container().get_argb(&light),
            on_color_container: MaterialDynamicColors::on_primary_container().get_argb(&light),
        },
        dark: ColorGroup {
            color: MaterialDynamicColors::primary().get_argb(&dark),
            on_color: MaterialDynamicColors::on_primary().get_argb(&dark),
            color_container: MaterialDynamicColors::primary_container().get_argb(&dark),
            on_color_container: MaterialDynamicColors::on_primary_container().get_argb(&dark),
        },
    }
}

pub fn get_closest_color(colors_to_compare: &Vec<ColorDefinition>, compare_to: &str) -> String {
    let mut closest_distance: Option<f64> = None;
    let mut closest_color: &str = "";

    for c in colors_to_compare {
        let distance = get_color_distance_lab(&c.color, compare_to);
        if closest_distance.is_none() || closest_distance.unwrap() > distance {
            closest_distance = Some(distance);
            closest_color = &c.name;
        }
        debug!("distance: {}, name: {}", distance, c.name)
    }
    debug!(
        "closest distance: {:?}, closest color: {}",
        closest_distance, closest_color
    );
    closest_color.to_string()
}
