/*
 * makeztxt: Translates ASCII files into zTXT databases for the Palm Handheld
 * Copyright (C) 2001 John Gruenenfelder
 *   johng@as.arizona.edu
 *   http://gutenpalm.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 *      Free Software Foundation, Inc.
 *      59 Temple Place
 *      Suite 330
 *      Boston, MA 02111-1307
 *      USA
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <getopt.h>

#include "libztxt/ztxt.h"


/*  makeztxt program version */
#define VERSION                 "1.20"
/*  makeztxt options file in user home directory */
#define OPTIONS_FILE            ".gutenpalmrc"



/* User definable options for new databases */
typedef struct optsType {
  int           list_bmrks;
  int           adjust_type;
  int           line_length;
  int           title_set;
  char          *output_filename;
} opts;



/*  Process commandline arguments  */
int     process_arguments(ztxt *, int, char **);
/*  Read a line of text from a file and strip off CR  */
char *  fgets_nocr(char *, int, FILE *);
/*  Read option file in home directory for regex  */
FILE *  fopen_in_home_dir(char *, char *);
void    load_options(ztxt *);
/*  Fatal error if condition is true  */
void    errorif(int);
/*  Convert a Palm ID into an int */
u_long  id_to_int(char *);


/* Global options */
opts    options;



int
main(int argc, char *argv[])
{
  int           last_arg;
  char          temptitle[dmDBNameLength];
  int           x;
  char          *y;
  char          *inputbuf = NULL;
  long          insize;
  long          i;
  int           infile;
  int           outfile;
  struct stat   instats;
  u_int         filesize;
  ztxt          *ztxtdb;


  /* Default options for a new DB */
  options.list_bmrks    = 0;
  options.adjust_type   = 0;
  options.line_length   = 0;
  options.title_set     = 0;
  options.output_filename = NULL;

  /* Create a new zTXT database */
  ztxtdb = ztxt_init();
  errorif(ztxtdb == NULL);

  /* Processing... */
  last_arg = process_arguments(ztxtdb, argc, argv);
  load_options(ztxtdb);

  if (last_arg >= argc)
    {
      fprintf(stderr, "no input\n");
      exit(1);
    }

  if (strcmp(argv[last_arg], "-") != 0)
    {
      /* Reading a normal input file */
      if (!options.title_set)
        { 
          strncpy(temptitle, argv[last_arg], dmDBNameLength);
          ztxt_set_title(ztxtdb, temptitle);
        }

      if (options.output_filename == NULL)
        {
          /* Form DB filename */
          options.output_filename = strdup(argv[last_arg]);
          options.output_filename = (char *)realloc(options.output_filename,
                                               (strlen(options.output_filename)
                                                + 10));
          /* First strip off any directory prefixes */
          y = strrchr(options.output_filename, '/');
          if (y)
            memmove(options.output_filename, y+1, strlen(y+1)+1);
          /* Now strip off any existing extension */
          y = strrchr(options.output_filename, '.');
          if (y == NULL)
            y = &options.output_filename[strlen(options.output_filename)];
          strcpy(y,".pdb");
        }

      /* Allocate the input buffer */
      x = stat(argv[last_arg], &instats);
      errorif(x == -1);
      inputbuf = (char *)malloc(instats.st_size + 1);
      errorif(inputbuf == NULL);

      /* Load and process the input file */
      infile = open(argv[last_arg], O_RDONLY);
      errorif(infile == -1);
      filesize = read(infile, inputbuf, instats.st_size);
      errorif(filesize == -1);
      close(infile);

      ztxt_set_data(ztxtdb, inputbuf, filesize);
    }
  else
    {
      /* Reading input from stdin */
      if (options.output_filename == NULL)
        {
          fprintf(stderr, "You must specify an output file with -o when "
                  "reading from stdin\n");
          ztxt_free(ztxtdb);
          exit(1);
        }
      else if (!options.title_set)
        {
          fprintf(stderr, "You must specify a title with -t when reading "
                  "from stdin\n");
          ztxt_free(ztxtdb);
          free(options.output_filename);
          exit(1);
        }

      /* Allocate the input buffer */
      inputbuf = (char *)malloc(10 * 1024);
      errorif(inputbuf == NULL);
      insize = 10 * 1024;
      i = 0;

      /* Read in all the data from stdin */
      while (!feof(stdin))
        {
          if (i >= insize - 1024)
            {
              inputbuf = (char *)realloc(inputbuf, insize + (10*1024));
              insize += 10*1024;
            }
          fgets(&(inputbuf[i]), 512, stdin);
          i += strlen(&(inputbuf[i]));
        }

      if (i == 0)
        {
          fprintf(stderr, "No data read from stdin.  Aborting.\n");
          ztxt_free(ztxtdb);
          free(inputbuf);
          free(options.output_filename);
          exit(1);
        }
      inputbuf[i] = '\0';

      ztxt_set_data(ztxtdb, inputbuf, i);
    }



  /* Make a zTXT */
  x = ztxt_process(ztxtdb, options.adjust_type, options.line_length);
  if (x > 0)
    {
      fprintf(stderr, "An error occured during the compression phase.\n"
              "Exiting...\n");
      ztxt_free(ztxtdb);
      free(inputbuf);
      free(options.output_filename);
      exit(1);
    }

  if (options.list_bmrks)
    ztxt_list_bookmarks(ztxtdb);

  ztxt_generate_db(ztxtdb);


  /* Output database */
  outfile = open(options.output_filename, O_WRONLY|O_CREAT|O_TRUNC,
                 S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH);
  errorif(outfile == -1);

  x = write(outfile, ztxtdb->output, ztxtdb->output_size);
  errorif(x != ztxtdb->output_size);

  close(outfile);


  /* Free memory */
  ztxt_free(ztxtdb);
  free(inputbuf);
  free(options.output_filename);

  return 0;
}


/* Process commandline arguments */
int
process_arguments(ztxt *ztxtdb, int argc, char *argv[])
{
  int   ind;
  int   opt;
  int   w;
  char  *usage = 
    "usage:   makeztxt [-hlV] [-t title] [-a adjust_type] [-b line_length]\n"
    "                  [-c creator] [-u type] [-w wbits] [-r regex]\n"
    "                  [-o outputfile] [-z compression_type] input";
  char  *optstr = "a:b:c:o:r:t:u:w:z:hlV";
  static struct option long_options[] = {
    {"adjust", 1, 0, 'a'},
    {"length", 1, 0, 'b'},
    {"creator", 1, 0, 'c'},
    {"help", 0, 0, 'h'},
    {"list", 0, 0, 'l'},
    {"output", 1, 0, 'o'},
    {"regex", 1, 0, 'r'},
    {"title", 1, 0, 't'},
    {"type", 1, 0, 'u'},
    {"version", 0, 0, 'V'},
    {"wbits", 1, 0, 'w'},
    {"compression", 1, 0, 'z'},
    {0, 0, 0, 0}};
 
  if (argc < 2)
    {
      fprintf(stderr, "%s\n", usage);
      fprintf(stderr, "type 'makeztxt -h' for more help\n");
      exit(1);
    }

  opt = getopt_long(argc, argv, optstr, long_options, &ind);
  while (opt != -1)
    {
      switch (opt)
        {
          case 'a':
            options.adjust_type = atoi(optarg);
            if ((options.adjust_type < 0) || (options.adjust_type > 2))
              {
                fprintf(stderr, "adjustment type out of range.  "
                        "Must be 0, 1, or 2\n");
                options.adjust_type = 0;
              }
            break;
          case 'b':
            options.line_length = atoi(optarg);
            break;
          case 'c':
            if (strlen(optarg) != 4)
              fprintf(stderr, "Creator must be a 4 character string\n");
            else
              ztxt_set_creator(ztxtdb, id_to_int(optarg));
            break;
          case 'h':
            printf("%s\n\n", usage);
            printf(
              "\t-a, --adjust       method for linefeed stripping (0,1,2)\n"
              "\t\t0 = strip linefeeds for lines longer than length\n"
              "\t\t1 = strip linefeed if line has any text\n"
              "\t\t2 = no linefeed stripping\n"
              "\t-b, --length       lines longer than this get linefeed "
                                     "stripped\n"
              "\t                     (default = autodetect)\n"
              "\t-c, --creator      database creator ID\n"
              "\t-h, --help         this help\n"
              "\t-l, --list         list regex generated bookmarks\n"
              "\t-o, --output       output file to be created\n"
              "\t-r, --regex        do a regex search for bookmarks\n"
              "\t-t, --title        title of database\n"
              "\t-u, --type         database type ID\n"
              "\t-V, --version      makeztxt version\n"
              "\t-w, --wbits        controls zLib mem usage (10 - 15)\n"
              "\t-z, --compression  method of compression (1 or 2)\n"
              "\t\t1 = allow random access (default)\n"
              "\t\t2 = one big data stream (10%% more compression)\n\n");
            exit(1);
          case 'l':
            options.list_bmrks =  1;
            break;
          case 'o':
            options.output_filename = strdup(optarg);
            break;
          case 'r':
            ztxt_add_regex(ztxtdb, optarg);
            break;
          case 't':
            options.title_set = 1;
            ztxt_set_title(ztxtdb, optarg);
            break;
          case 'u':
            if (strlen(optarg) != 4)
              fprintf(stderr, "Type must be a 4 character string\n");
            else
              ztxt_set_type(ztxtdb, id_to_int(optarg));
            break;
          case 'V':
            printf("makeztxt %s\n", VERSION);
            exit(0);
            break;
          case 'w':
            w = atoi(optarg);
            if ((w > 15) || (w < 10))
              {
                fprintf(stderr, "wbits out of range.  Must be 10 to 15\n");
                w = MAXWBITS;
              }
            ztxt_set_wbits(ztxtdb, w);
            break;
          case 'z':
            w = atoi(optarg);
            if ((w != 1) && (w != 2))
              {
                fprintf(stderr, "compression_type must be 1 or 2. Setting to"
                        "default of 1.\n");
                w = 1;
              }
            ztxt_set_compressiontype(ztxtdb, w);
            break;
        }

      opt = getopt_long(argc, argv, optstr, long_options, &ind);
    }

  return optind;
}


/* Read a line from the file and remove CR */
char *
fgets_nocr(char *buffer, int bufsize, FILE *infile)
{
  int   len;
  char  *str;

  str = fgets(buffer, bufsize, infile);
  if (str != NULL)
    {
      /* got something, remove trailing CR character */
      len = strlen(str);

      if (len >= 2)
        {
          if (str[len - 2] == '\r')
            {
              str[len - 2] = '\n';
              str[len - 1] = 0;
            }
        }
    }

  return str;
}


/* Open a file in a users home dir or in the current dir */
FILE *
fopen_in_home_dir(char *file, char *mode)
{
  char *str = getenv("HOME");
  FILE *fd = NULL;
  char buff[512];

  if (str != NULL)
    {
      /* try in home directory */
      strcpy(buff, str);
      strcat(buff, "/");
      strcat(buff, file);

      fd = fopen(buff, mode);
    }

  if (fd == NULL)
    fd = fopen(file, mode);

  return fd;
}


/* Load the options file if it exists */
void
load_options(ztxt *ztxtdb)
{
  FILE  *optfd;
  char  buff[256];
  char  *ptr;
  int   len;

  optfd = fopen_in_home_dir(OPTIONS_FILE, "r");
  if (optfd != NULL)
    {
      buff[0] = 0;

      while (fgets_nocr(buff, 255, optfd) != NULL)
        {
          ptr = buff;

          if (!buff[0])
            break;

          while (*ptr == ' ' || *ptr == '\t' || *ptr == '\n')
            ++ptr;

          if (*ptr != '#' && *ptr)
            {
              /* not a comment line and not blank - process */
              len = strlen(ptr);

              if (ptr[len - 1] == '\n')
                ptr[len - 1] = 0;

              ztxt_add_regex(ztxtdb, strdup(ptr));
            }
          buff[0] = 0;
        }
      fclose(optfd);
    }
}


/*  Converts a 4 byte string Palm ID into a 4 byte integer */
u_long
id_to_int(char *idstr)
{
  if (strlen(idstr) != 4)
    return 0;

  return (0x1000000*(u_long)idstr[0] +
          0x10000*(u_long)idstr[1] +
          0x100*(u_long)idstr[2] +
          (u_long)idstr[3]);
}


/*  Print error message and exit if 'error' is true */
void
errorif(int error)
{
  if (error)
    {
      perror("makeztxt");
      exit(1);
    }
}
