/*
 * libztxt:  A library for creating zTXT databases
 * Copyright (C) 2001 John Gruenenfelder
 *   johng@as.arizona.edu
 *   http://gutenpalm.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 *      Free Software Foundation, Inc.
 *      59 Temple Place
 *      Suite 330
 *      Boston, MA 02111-1307
 *      USA
 *
 */

#ifndef ZTXT_H
#define ZTXT_H 1


#include <regex.h>
#include "databaseHdrs.h"
#include <gutenpalm_common.h>


/*  Current version of the zTXT format (v1.40) */
#define ZTXT_VERSION            0x0128
/*  Palm's have an epoch of Jan. 1, 1904 */
#define PALM_CTIME_OFFSET       0x7C25B080
/*  Max size of the header.  Should be fine except for huge docs */
#define MAX_HEADER_SIZE         8192



/* A linked list of regular expressions */
typedef struct regex_nodeType {
  char          *pattern;
  regex_t       *prog;
  int           bad;
  struct regex_nodeType *next;
} regex_node;


/* A simple struct for bookmarks generated from the regex */
typedef struct bmrk_nodeType {
  UInt32        offset;
  char          title[MAX_BMRK_LENGTH];
  struct bmrk_nodeType *next;
} bmrk_node;


/* This structure defines all the data and attributes of a zTXT database */
typedef struct ztxtType {
  /* Internal portions of a Palm DB */
  DatabaseHdrType *dbHeader;
  RecordEntryType *dbRecordEntries;
  zTXT_record0  record0;
  /* Used to control the compression */
  int           compression_type;
  int           wbits;
  /* zTXT bookmarks generated from regular expressions */
  short         num_regex;
  regex_node    *regex_list;
  short         num_bookmarks;
  bmrk_node     *bookmarks;
  /* input text data */
  long          input_size;
  char          *input;
  /* Intermediate data used during processing */
  long          tmpsize;
  char          *tmp;
  /* Compressed buffer created by zlib */
  long          comp_size;
  char          *compressed_data;
  int           num_records;
  int           *record_offsets;
  /* Complete zTXT database */
  long          output_size;
  char          *output;
} ztxt;



/* Functions to manipulate a zTXT structure */
extern ztxt *   ztxt_init(void);
extern void     ztxt_free(ztxt *db);
extern void     ztxt_add_regex(ztxt *db, char *regex);
extern int      ztxt_process(ztxt *db, int method, int line_length);
extern void     ztxt_generate_db(ztxt *db);

/* Print a list of bookmarks to stdout */
extern void     ztxt_list_bookmarks(ztxt *db);

/*
 * Use these to change the defaults.
 * You must call ztxt_set_data() to do any work.
 * Calling ztxt_set_title() is also a good idea.
 */
extern void     ztxt_set_title(ztxt *db, char *new_title);
extern void     ztxt_set_data(ztxt *db, char *new_data, long datasize);
extern void     ztxt_set_creator(ztxt *db, long new_creator);
extern void     ztxt_set_type(ztxt *db, long new_type);
extern void     ztxt_set_wbits(ztxt *db, int new_wbits);
extern void     ztxt_set_compressiontype(ztxt *db, int new_comptype);

#endif
