/* tuplespace.c -- implementation of Tuplespace 
   Copyright (C) 1997, 1998, 1999, 2000, 2001, 2002 Wong Weng Fai <wongwf@comp.nus.edu.sg>

   This file is part of linda.

   linda is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   linda is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with linda; see the file COPYING.  If not, write to
   the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "tuplespace.h"
#include "glibextra.h"
#include <string.h>

typedef union  _TupleItemData  TupleItemData;

typedef struct _TupleItem            TupleItem;
typedef struct _TupleSlot            TupleSlot;
typedef struct _TupleSlotForeachData TupleSlotForeachData;

typedef enum   _TupleItemType   TupleItemType;
typedef enum   _TupleItemField  TupleItemField;

union _TupleItemData
{
  gchar c, *cp;
  int   i, *ip;
  gchar  *s, **sp;
  float f, *fp;
};

enum _TupleItemType {
  TUPLE_CHAR   = 'c',
  TUPLE_INT    = 'd',
  TUPLE_STRING = 's',
  TUPLE_FLOAT  = 'f'
};


enum _TupleItemField {
  TUPLE_DEFINE = '%', 
  TUPLE_QUERY  = '?'
};

struct _TupleItem
{
  TupleItemField field;
  TupleItemType  type;
  TupleItemData  data;
};

struct _Tuple
{
  TupleType type;
  GCond*    cv;
  GSList*   items;
  gint      hash;
};

struct _TupleSlot
{
  GMutex * mutex;
  GSList * tuples;
};

struct _TupleSlotForeachData
{
  Tuple * new_tuple;
  Tuple * matched_tuple;
  TupleResult result;
};

#define tuplespace_size 128
static  TupleSlot tuplespace[tuplespace_size];

static GMemChunk * tuples_chunk = NULL;
static GMemChunk * tuple_itmes_chunk = NULL;

static void tuple_init(Tuple * tuple, TupleType type, gint hash_init);

static TupleItem * tuple_item_new (TupleItemField field,
				   TupleItemType  type);
static void tuple_item_free (TupleItem * item);
static void tuple_free_item_cb (gpointer data, gpointer user_data);

static void tuple_slot_init    (TupleSlot * slot);
static void tuple_slot_end    (TupleSlot * slot);
static gint tuple_slot_slot    (TupleSlot * slot, Tuple * new_tuple, GTimeVal * end_time);
static gint tuple_slot_out     (gpointer data, gpointer user_data);
static gint tuple_slot_in      (gpointer data, gpointer user_data);

static gboolean tuple_match    (Tuple * tuple1, Tuple * tuple2);
static void     tuple_bind_val     (Tuple * tuple1, Tuple * tuple2);

void
tuplespace_init(void)
{
  gint i;
  if (!tuples_chunk)
    {
      tuples_chunk = g_mem_chunk_create(Tuple,
					32,
					G_ALLOC_AND_FREE);
      tuple_itmes_chunk = g_mem_chunk_create (TupleItem,
					      128,
					      G_ALLOC_AND_FREE);
      for (i = 0; i < tuplespace_size; i++)
	tuple_slot_init(&tuplespace[i]);
    }
}

void
tuplespace_end(void)
{
  gint i;
  if (tuples_chunk)
    {
      for (i = 0; i < tuplespace_size; i++)
	tuple_slot_end(&tuplespace[i]);
      g_mem_chunk_destroy(tuples_chunk);
      tuples_chunk = NULL;
      g_mem_chunk_destroy(tuple_itmes_chunk);
      tuple_itmes_chunk = NULL;
    }
}

Tuple *
tuple_new (TupleType type, const gchar * mask, va_list list)
{
  Tuple * tuple;
  TupleItem * item;
  
  const gchar *mask_ptr;
  gint len = 0, i;

  len  = strlen(mask);
  if (1 == (len % 2))
    return NULL;
  else
    len /= 2;
  
  tuple = g_mem_chunk_alloc(tuples_chunk);
  tuple_init(tuple, type, len);

  for (mask_ptr = mask, i = 1; 
       *mask_ptr != '\0'; 
       mask_ptr += 2, i++) 
    {
      item = tuple_item_new (mask_ptr[0], mask_ptr[1]);
      if (item->field == TUPLE_DEFINE) 
	{
	  switch (item->type) 
	    { 
	    case TUPLE_STRING:
	      item->data.s = va_arg(list, char *);
	      item->data.s = g_strdup(item->data.s);
	      tuple->hash += 3 * i;
	      break;
	    case TUPLE_INT:
	      item->data.i = va_arg(list, int);
	      tuple->hash += 5 * i;
	      break;
	    case TUPLE_FLOAT: 
	      item->data.f = (float)va_arg(list, double);
	      tuple->hash += 7 * i;
	      break;
	    case TUPLE_CHAR: 
	      item->data.c = (char)va_arg(list, int);
	      tuple->hash += 11 * i;
	      break;
	    default:
	      g_assert(item);
	      g_assert(tuple_itmes_chunk);
	      g_mem_chunk_free(tuple_itmes_chunk, item);
	      item = NULL;
	    } 
	}
      else if (item->field == TUPLE_QUERY && type != TUPLE_OUT)
	{
	  switch (item->type) 
	    { 
	    case TUPLE_STRING:
	      item->data.sp = va_arg(list, char **); 
	      tuple->hash += 3 * i;
	      break;
	    case TUPLE_INT:
	      item->data.ip = va_arg(list, int *); 
	      tuple->hash += 5 * i;
	      break;
	    case TUPLE_FLOAT:
	      item->data.fp = va_arg(list, float *); 
	      tuple->hash += 7 * i;
	      break;
	    case TUPLE_CHAR:
	      item->data.cp = va_arg(list, char *); 
	      tuple->hash += 11 * i;
	      break;
	    default:
	      g_assert(item);
	      g_assert(tuple_itmes_chunk);
	      g_mem_chunk_free(tuple_itmes_chunk, item);
	      item = NULL;
	    }
	}
      else
	{
	  g_assert(item);
	  g_assert(tuple_itmes_chunk);
	  g_mem_chunk_free(tuple_itmes_chunk, item);
	  item = NULL;
	}
      
      if (item)
	tuple->items = g_slist_prepend(tuple->items, item);
      else
	{
	  tuple_free(tuple);
	  tuple = NULL;
	  break;
	}
    }
  if (tuple)
    tuple->hash %= tuplespace_size;
  return tuple;
}

static void
tuple_init(Tuple * tuple, TupleType type, gint hash_init)
{
  tuple->type  = type;
  tuple->items = NULL;
  tuple->cv    = g_cond_new ();
  tuple->hash  = hash_init;
}

void
tuple_free(Tuple * tuple)
{
  g_slist_foreach(tuple->items, tuple_free_item_cb, NULL);
  g_slist_free(tuple->items);
  g_cond_free(tuple->cv);
}

static void
tuple_free_item_cb (gpointer data, gpointer user_data)
{
  tuple_item_free(data);
}

static TupleItem *
tuple_item_new(TupleItemField field, TupleItemType  data)
{
  TupleItem * item;

  item  = g_mem_chunk_alloc(tuple_itmes_chunk);
  item->field = field;
  item->type  = data;
  
  return item;
}

static void
tuple_item_free (TupleItem * item)
{
  if (item->field == TUPLE_DEFINE
      && item->type == TUPLE_STRING)
    g_free(item->data.s);

  g_assert(item);
  g_assert(tuple_itmes_chunk);
  g_mem_chunk_free(tuple_itmes_chunk,
		   item);
}

TupleResult
tuplespace_enter(Tuple * tuple, GTimeVal * end_time)
{
  TupleSlot * slot;
  g_return_if_fail (tuple);
  
  slot = &(tuplespace[tuple->hash]);
  return (TupleResult)tuple_slot_slot(slot, tuple, end_time);
}


static void
tuple_slot_init(TupleSlot * slot)
{
  slot->mutex  = g_mutex_new();
  slot->tuples = NULL;
}

static void
tuple_slot_end    (TupleSlot * slot)
{
  g_mutex_lock(slot->mutex);
  g_slist_foreach(slot->tuples, g_func_func1_to_func0, tuple_free);
  g_slist_free(slot->tuples);
  slot->tuples = NULL;
  g_mutex_unlock(slot->mutex);

  g_mutex_free(slot->mutex);
  slot->mutex = NULL;
}

static gint
tuple_slot_slot(TupleSlot * slot, Tuple * new_tuple, GTimeVal * end_time)
{
  GIntFunc func;
  TupleSlotForeachData foreachdata;
  gboolean intime;
  
  foreachdata.new_tuple = new_tuple;
  foreachdata.matched_tuple = NULL;
  
  if (TUPLE_OUT == new_tuple->type)
    {
      foreachdata.result = TUPLE_HOLD;
      g_mutex_lock(slot->mutex);
      g_slist_foreach_with_break (slot->tuples,
				  tuple_slot_out,
				  &foreachdata);
      if (foreachdata.result == TUPLE_HOLD)
	slot->tuples = g_slist_prepend(slot->tuples, new_tuple);
      g_mutex_unlock(slot->mutex);
      return foreachdata.result;
    }
  else if (TUPLE_RD & new_tuple->type
	   || TUPLE_IN & new_tuple->type)
    {
      if (TUPLE_NOBLOCK & new_tuple->type)
	foreachdata.result = TUPLE_FREE ;
      else
	foreachdata.result = TUPLE_HOLD ;
      
      g_mutex_lock(slot->mutex);
      g_slist_foreach_with_break (slot->tuples, 
				  tuple_slot_in, 
				  &foreachdata);
      if (foreachdata.result == TUPLE_HOLD)
	{
	  /* Cannot find matched tuple --- stay here */
	  slot->tuples = g_slist_prepend(slot->tuples, 
					 new_tuple);
	  intime = g_cond_timed_wait(new_tuple->cv, 
				     slot->mutex,
				     end_time);
	  slot->tuples = g_slist_remove(slot->tuples, 
					new_tuple);
	  foreachdata.result = TUPLE_FREE;
	  if (intime)
	    foreachdata.result |= TUPLE_BOUND_VAL;
	  /* Matched out tuple is freed in out-side thread */	  
	}
      else if (TUPLE_IN & new_tuple->type
	       && foreachdata.result & TUPLE_BOUND_VAL)
	{
	  /* Matched out tuple is freed HERE */	  
	  g_assert(foreachdata.matched_tuple);
	  slot->tuples = g_slist_remove(slot->tuples, 
					foreachdata.matched_tuple);
	  tuple_free (foreachdata.matched_tuple);
	}
      g_mutex_unlock(slot->mutex);
      return foreachdata.result;
    }
  else
    g_assert_not_reached();
}

static gint
tuple_slot_out (gpointer data, gpointer user_data)
{
  Tuple       * slotted_tuple = data;
  TupleSlotForeachData * foreachdata = user_data;

  if (tuple_match(foreachdata->new_tuple, slotted_tuple)) 
    {
      if (slotted_tuple->type & TUPLE_IN)
	{
	  tuple_bind_val (foreachdata->new_tuple, slotted_tuple);
	  g_cond_signal(slotted_tuple->cv);
	  foreachdata->result = TUPLE_FREE;
	  return 1;
	}  
      else if (slotted_tuple->type & TUPLE_RD)
	{
	  tuple_bind_val (foreachdata->new_tuple, slotted_tuple);
	  g_cond_signal(slotted_tuple->cv);
	}
    }
  return 0;
}

static gint
tuple_slot_in (gpointer data, gpointer user_data)
{
  Tuple       * slotted_tuple = data;
  TupleSlotForeachData * foreachdata = user_data;
  
  if (tuple_match(foreachdata->new_tuple, slotted_tuple)) 
    {
      if (slotted_tuple->type == TUPLE_OUT)
	{
	  tuple_bind_val (foreachdata->new_tuple, slotted_tuple);
	  foreachdata->matched_tuple = slotted_tuple;
	  foreachdata->result = TUPLE_FREE | TUPLE_BOUND_VAL;
	  return 1;
	}
    }
  return 0;
}

static void
tuple_bind_val (Tuple * tuple1, Tuple * tuple2)
{
  GSList * items1, * items2;
  TupleItem * item1, * item2;
  TupleItem * item_define, * item_query;
  
  if ((tuple1 == NULL) || (tuple2 == NULL)) 
    g_assert_not_reached();

  for (items1 = tuple1->items, items2 = tuple2->items;
       (items1 != NULL) && (items2 != NULL);
       items1 = g_slist_next(items1), items2 = g_slist_next(items2))
    {
      item1 = items1->data;
      item2 = items2->data;

      if (item1->field == TUPLE_DEFINE
	  && item2->field == TUPLE_QUERY)
	{
	  item_define = item1;
	  item_query  = item2;
	}
      else if (item1->field == TUPLE_QUERY
	       && item2->field == TUPLE_DEFINE)
	{
	  item_define = item2;
	  item_query  = item1;
	}
      else
	continue;

      switch (item_define->type)
	{
	case TUPLE_STRING:
	  *(item_query->data.sp) = g_strdup(item_define->data.s);
	  break;
	case TUPLE_INT:
	  *(item_query->data.ip) = item_define->data.i;
	  break;
	case TUPLE_FLOAT:
	  *(item_query->data.fp) = item_define->data.f;
	  break;
	case TUPLE_CHAR:
	  *(item_query->data.cp) = item_define->data.c;
	  break;
	default:
	  g_assert_not_reached();
	}
    }
}

static gboolean
tuple_match (Tuple * tuple1, Tuple * tuple2)
{
  GSList * items1, * items2;
  TupleItem * item1, * item2;
  
  if ((tuple1 == NULL) || (tuple2 == NULL))
    return FALSE;
  
  for (items1 = tuple1->items, items2 = tuple2->items;
       (items1 != NULL) && (items2 != NULL);
       items1 = g_slist_next(items1), items2 = g_slist_next(items2))
    {
      item1 = items1->data;
      item2 = items2->data;

      g_assert(items1 && items2);
      
      if (item1->field == item2->field)
	{
	  if (item1->field == TUPLE_QUERY)
	    return 0;
	  
	  if (item2->type != item1->type)
	    return 0;
	  
	  switch (item1->type) 
	    {
	    case TUPLE_CHAR:
	      if (item1->data.c != item2->data.c)
		return 0;
	      break;
	    case TUPLE_INT:
	      if (item1->data.i != item2->data.i)
		return 0;
	      break;	      
	    case TUPLE_STRING:
	      if (0 != strcmp(item1->data.s, item2->data.s))
		return 0;
	      break;
	    case TUPLE_FLOAT:
	      if (item1->data.i != item2->data.i)
		return 0;
	      break;
	    default:
	      g_assert_not_reached();
	      break;
	    }
	}
      else
	{
	  if (item1->type != item2->type)
	    return 0;
	  g_assert (item1->type == TUPLE_STRING
		    || item1->type == TUPLE_INT
		    || item1->type == TUPLE_CHAR
		    || item1->type == TUPLE_FLOAT);
	}
	  
    }

  if ((items1 == NULL) 
      && (items2 == NULL)) 
    return 1;
  else 
    return 0;
}

