/* linda.c -- linda parallel programming library
   Copyright (C) 1997, 1998, 1999, 2000, 2001, 2002 Wong Weng Fai <wongwf@comp.nus.edu.sg>

   This file is part of linda.

   linda is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   linda is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with linda; see the file COPYING.  If not, write to
   the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "linda.h"
#include "tuplespace.h"

G_LOCK_DEFINE_STATIC (linda_threads_slist);
static GSList * linda_threads_slist = NULL;

static gint linda_spawn_full (GVoidFunc f, gboolean joinable, GError **error);
static void     linda_thread_join       (gpointer data, gpointer user_data);
static gpointer linda_void_func_wrapper (gpointer f);
static gint linda_do_any (TupleType type, GTimeVal * end_time, const gchar * mask, va_list list);

void
linda_init(GThreadFunctions       *vtable)
{
  int i, j;
  
  g_thread_init  (vtable); 
  tuplespace_init ();
} 

void
linda_end(void)
{
  G_LOCK(linda_threads_slist);
  linda_threads_slist = g_slist_reverse(linda_threads_slist);
  g_slist_foreach (linda_threads_slist, linda_thread_join, NULL);
  g_slist_free(linda_threads_slist);
  linda_threads_slist = NULL;
  G_UNLOCK(linda_threads_slist);
  tuplespace_end();
}

static void
linda_thread_join       (gpointer data, gpointer user_data)
{
  g_thread_join((GThread *)data);
}

gint
linda_spawn (GVoidFunc f)
{
  return linda_spawn_full(f, TRUE, NULL);
}

static gint
linda_spawn_full (GVoidFunc f, gboolean joinable, GError **error)
{
  GThread * thr;
  thr = g_thread_create(linda_void_func_wrapper, f, joinable, error);
  if (thr)
    {
      if (joinable)
	{
	  G_LOCK(linda_threads_slist);
	  linda_threads_slist = g_slist_prepend(linda_threads_slist,
						thr);
	  G_UNLOCK(linda_threads_slist);
	}
      return 0;
    }
  else
    return -1;
}

static gpointer
linda_void_func_wrapper(gpointer f)
{
  GVoidFunc f0 = (GVoidFunc)f;
  f0 ();
  return NULL;
}

static gint
linda_do_any (TupleType type, GTimeVal * end_time, const gchar * mask, va_list list)
{ 
  Tuple * tuple;
  gint stat;

  tuple = tuple_new(type, mask, list);  
  if (NULL == tuple)
    return -1;

  stat = tuplespace_enter(tuple, end_time);
  if (stat & TUPLE_FREE)
    tuple_free(tuple);
  return stat & TUPLE_BOUND_VAL;
}

#define LINDA_DO(type, end_time)				\
  va_list tuple_list;						\
  gint stat;							\
								\
  va_start(tuple_list, tuple_mask);				\
  stat = linda_do_any(type, end_time, tuple_mask, tuple_list);	\
  va_end(tuple_list);				    


void
linda_out (const gchar *tuple_mask, ...) 
{
  LINDA_DO(TUPLE_OUT, NULL);
}

void
linda_rd (const gchar *tuple_mask, ...) 
{
  LINDA_DO(TUPLE_RD, NULL);
}

gint
linda_rdp (const gchar *tuple_mask, ...) 
{
  LINDA_DO(TUPLE_RDP, NULL);
  return stat;
}

void
linda_in (const gchar *tuple_mask, ...) 
{
  LINDA_DO(TUPLE_IN, NULL);
}

gint
linda_inp (const gchar *tuple_mask, ...) 
{
  LINDA_DO(TUPLE_INP, NULL);
  return stat;
}

gint
linda_inp_timed (GTimeVal * end_time, const gchar *tuple_mask, ...)
{
  LINDA_DO(TUPLE_IN, end_time);
  return stat;
}

gint
linda_rdp_timed (GTimeVal * end_time, const gchar *tuple_mask, ...)
{
  LINDA_DO(TUPLE_RD, end_time);
  return stat;
}

const
gchar * linda_version(void)
{
  return VERSION;
}
