/**
 * @file header.c Zip header functions
 * 
 * $Id: extra.c,v 1.2 2003/01/01 06:22:34 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libcomprex/internal.h>
#include "zip.h"
#include "extra.h"


/* The PKWARE and Info-Zip (type 1, 2) Unix extras are
 * similar enough that we can use the same function to
 * parse all of them.  This is to avoid redundancy and 
 * oversided spoons.
 */

	
static void
__initUnixExtra(ZipUnixExtra *tmp)
{
	tmp->atime = tmp->mtime = tmp->ctime = -1;
	tmp->uid = tmp->gid = -1;
}

static void
__parseUnixextra(char *buffer, int counter, unsigned short blockLength, ZipUnixExtra **new_extra, int type)
{
	int remaining;
	if((*new_extra) == NULL)
	{
		MEM_CHECK((*new_extra) = (ZipUnixExtra *)malloc(sizeof(ZipUnixExtra)));
		__initUnixExtra((*new_extra));
	}
	switch(type)
	{
		case ZIP_PKWARE_UNIX_EXTRA:
			(*new_extra)->atime    =  cxZipGet32(buffer, &counter);
			(*new_extra)->mtime    =  cxZipGet32(buffer, &counter);
			(*new_extra)->uid      =  cxZipGet16(buffer, &counter);
			(*new_extra)->gid      =  cxZipGet16(buffer, &counter);
			
			remaining = blockLength - counter;
			if (remaining != 0)
			{
				MEM_CHECK((*new_extra)->link = (char *)malloc(remaining + 1));
				strncpy((*new_extra)->link, buffer, remaining);
				(*new_extra)->link[remaining + 1] = '\0';
			}
			else
				(*new_extra)->link = NULL;
			break;
		case ZIP_UNIX1_IZ_EXTRA:
			(*new_extra)->atime    =  cxZipGet32(buffer, &counter);
			(*new_extra)->mtime    =  cxZipGet32(buffer, &counter);
			remaining = blockLength - counter;
			if (remaining == 4) /* UID and GID present */
			{
				(*new_extra)->uid = cxZipGet16(buffer, &counter);
				(*new_extra)->gid = cxZipGet16(buffer, &counter);
			}
			(*new_extra)->link = NULL;
			break;
		case ZIP_UNIX2_IZ_EXTRA:
			(*new_extra)->uid = cxZipGet16(buffer, &counter); 
			(*new_extra)->gid = cxZipGet16(buffer, &counter);

			(*new_extra)->link = NULL;
			break;
	}
}

static void
__parseTStampExtra(char *buffer, int counter, ZipUnixExtra **new_extra)
{
	int flag;
	if (*(new_extra) == NULL)
	{
		MEM_CHECK(((*new_extra) = (ZipUnixExtra *)malloc(sizeof(ZipUnixExtra))));
		__initUnixExtra((*new_extra));
	}
	flag = cxZipGet8(buffer, &counter);
	
	(*new_extra)->mtime = (flag & 1) ? cxZipGet32(buffer, &counter): -1;
	(*new_extra)->atime = (flag & 2) ? cxZipGet32(buffer, &counter): -1;
	(*new_extra)->ctime = (flag & 4) ? cxZipGet32(buffer, &counter): -1;

	(*new_extra)->link = NULL;
}

ZipStatus
cxZipFindExtra(ZipExtra *extra, int type)
{
	/* So we'll parse the extraField as long as we can
	 * and pull out the different [headerID1][data1] + ...
	 */
	unsigned short  headerID;
	char *buffer = extra->field;
	unsigned short  extraLength = extra->length;
	unsigned short  blockLength;
	int counter = 0;	
	unsigned short  currentPos;
	
	if ((extraLength == 0)  || buffer == NULL)
		return ZIP_ERROR;
	

	while (counter < extraLength)
	{
		headerID = cxZipGet16(buffer, &counter);
		blockLength = cxZipGet16(buffer, &counter);
		if (headerID == type)
		{
		   	switch(type)
		   	{
				case ZIP_UNIX1_IZ_EXTRA:
					__parseUnixextra(buffer, counter, blockLength, 
							 &(extra->data), headerID);
					break;
				case ZIP_UNIX2_IZ_EXTRA:
					__parseUnixextra(buffer, counter, blockLength, 
							 &(extra->data), headerID);
					break;
				case ZIP_PKWARE_UNIX_EXTRA:
					__parseUnixextra(buffer, counter, blockLength, 
							 &(extra->data), headerID);
					break;
				case ZIP_TSTAMP_EXTRA:
					__parseTStampExtra(buffer, counter, &extra->data);
					break;
				default:
					/* Don't support other extra types now */
					extra->data = NULL;
					break;
		   	}
			return ZIP_SUCCESS;
		}
		/* Guess this wasn't one we wanted, on to the next; */
		counter = counter + blockLength;
	}	
	extra = NULL;
	return ZIP_ERROR;
}

