/**
 * @file deflate.c Zip deflation-compressed functions
 * 
 * $Id: deflate.c,v 1.4 2003/01/01 06:22:33 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libcomprex/internal.h>
#include "deflate.h"
#include "zip.h"


CxStatus
__cxZipInflateInit2(CxFile *file)
{
	ZipFile *fileData;

	if(file == NULL)
		return CX_ERROR;
	
	fileData = (ZipFile *)file->moduleData;

	fileData->cx_stream.next_in = Z_NULL;
	fileData->cx_stream.avail_in = Z_NULL;
	fileData->cx_stream.zalloc = Z_NULL;
	fileData->cx_stream.zfree = Z_NULL;
	fileData->cx_stream.opaque = Z_NULL;

	if(inflateInit2(&fileData->cx_stream, -MAX_WBITS) != Z_OK)
		return CX_ERROR;
	else
		return CX_SUCCESS;
}

size_t
__inflateReadFunc(void *ptr, size_t size, size_t nmemb, CxFP *fp)
{
	CxFP *parentFp;
	CxFile *file;
	ZipFile *fileData;
	
	char *buffer = (char *)malloc(CX_ZIP_BUF);
	int requestedSize;
	int remainingSize;

	int csize;
	int status;
	file = fp->file;
	parentFp = (CxFP *)cxGetFileArchive(file)->moduleData;

	fileData = (ZipFile *)file->moduleData;
	remainingSize = cxGetFileCompressedSize(file) - (fileData->curOffset - fileData->startOffset);
	requestedSize = size * nmemb;
	fileData->cx_stream.next_out = ptr;
	
	/* Assume that the calling function has allocated enough 
	 * memory, since that's what they requested
	 */
	fileData->cx_stream.avail_out = requestedSize;
 	
	/* Now we'll call inflate until we've read the requested amount of
	 * data.  Multiple calls are needed because the input buffer could
	 * run out  before all of the requested data is read.  We don't predict
	 * here how much a given amount of uncompressed data will take up when
	 * uncompressed. That's all I have to say about that
	 */
	 while ((fileData->cx_stream.avail_out > 0)  && (remainingSize > 0) )
	 {
		csize = (remainingSize > CX_ZIP_BUF) ? CX_ZIP_BUF : remainingSize;
		
		/* Read in the compressed data from the parent archive */
		csize = cxRead(buffer, 1, csize, parentFp);
		if (csize == 0)  /* Somehow, read less than we were supposed to */
			return 0;
		
		remainingSize -= csize;

		fileData->cx_stream.next_in  = buffer;
		fileData->cx_stream.avail_in = csize;

		status = inflate(&fileData->cx_stream, Z_NO_FLUSH);

		if ((status != Z_OK) || (status != Z_STREAM_END))
			break;

	}
	
	fileData->curOffset = fileData->startOffset + (cxGetFileCompressedSize(file) - remainingSize);

	/* XXX Should probably do something if status is Z_ERROR */
	
	free(buffer);
	
	return requestedSize - fileData->cx_stream.avail_out;
}
