/**
 * @file header.h Tar header
 * 
 * $Id: header.h,v 1.4 2003/01/01 06:22:33 chipx86 Exp $
 *
 * @Copyright (C) 1999-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _TAR_HEADER_H_
#define _TAR_HEADER_H_

#include "gnuext.h"

/** Tar files are made in basic blocks of this size. */
#define TAR_BLOCK_SIZE 512

/* Static values. */
#define TAR_MAGIC       "ustar"  /**< Tar magic (with a trailing null)     */
#define TAR_MAGIC_LEN   6        /**< Length of magic.                     */
#define TAR_VERSION     "00"     /**< Tar version (witout a trailing null) */
#define TAR_VERSION_LEN 2        /**< Length of version.                   */

/* Bits used in the mode field. */
#define TAR_SUID         04000  /**< Set UID on execution. */
#define TAR_SGID         02000  /**< Set GID on execution. */
#define TAR_VRX          01000  /**< Reserved.             */

/* File permissions */
#define TAR_USER_READ    00400  /**< Read by owner.            */
#define TAR_USER_WRITE   00200  /**< Write by owner.           */
#define TAR_USER_EXEC    00100  /**< Execute/search by owner.  */
#define TAR_GROUP_READ   00040  /**< Read by group.            */
#define TAR_GROUP_WRITE  00020  /**< Write by group.           */
#define TAR_GROUP_EXEC   00010  /**< Execute/search by group.  */
#define TAR_OTHER_READ   00004  /**< Read by other.            */
#define TAR_OTHER_WRITE  00002  /**< Write by other.           */
#define TAR_OTHER_EXEC   00001  /**< Execute/search by other.  */

typedef enum
{
	BLOCK_STILL_UNREAD,  /**< For when tarReadHeader has not been called */
	BLOCK_SUCCESS,       /**< Header successfully read and checksummed.  */
	BLOCK_ZERO_BLOCK,    /**< Zero block where header expected.          */
	BLOCK_END_OF_FILE,   /**< True end of file while header expected.    */
	BLOCK_FAILED         /**< Ill-formed header, or bad checksum.        */

} TarReadStatus;

typedef enum
{
	NORMAL_FILE_0 = '\0',  /**< For compatibility with decades-old bug. */
	NORMAL_FILE_1 = '0',   /**< Regular file.                           */
	HARDLINK      = '1',   /**< Hard link.                              */
	SYMLINK       = '2',   /**< Symbolic link (hard if not supported).  */
	CHAR_DEVICE   = '3',   /**< Character device.                       */
	BLOCK_DEVICE  = '4',   /**< Block device.                           */
	DIRECTORY     = '5',   /**< Directory.                              */
	FIFO          = '6',   /**< Named pipe.                             */
	CONTIGUOUS    = '7',   /**< Contiguous file.                        */
	GNU_LONGLINK  = 'K',   /**< GNU long link extension.                */
	GNU_LONGNAME  = 'L'    /**< GNU long name extension.                */

} TarFileType;

typedef struct
{
	char name[100];      /**< File name.                          */
	char mode[8];        /**< File mode.                          */
	char uid[8];         /**< User ID (UID).                      */
	char gid[8];         /**< Group ID (GID).                     */
	char size[12];       /**< File size.                          */
	char mtime[12];      /**< Last modified timestamp.            */
	char checksum[8];    /**< MD5 checksum.                       */
	char typeFlag;       /**< Flag for the symbolic or hard link. */
	char linkName[100];  /**< Name for symbolic or hard link.     */
	char magic[6];       /**< Tar magic number.                   */
	char version[2];     /**< Tar version.                        */
	char userName[32];   /**< User name.                          */
	char groupName[32];  /**< Group name.                         */
	char majorDevice[8]; /**< Major device number.                */
	char minorDevice[8]; /**< Minor device number.                */
	char prefix[155];    /**< Prefix.                             */
	char padding[12];    /**< Extra padding.                      */

	char *gnu_longname;
	char *gnu_longlink;

} UstarHeader;

typedef union
{
	char         buffer[TAR_BLOCK_SIZE];
	UstarHeader  header;
	GnuHeader    extraHeader;
	GnuOldHeader oldGnuHeader;
	SparseHeader sparseHeader;

} TarBlock;

TarReadStatus tarReadBlock(FILE *fp, TarBlock *block);

#endif /* _TAR_HEADER_H_ */

