/**
 * @file libcomprex/io_internal.c Internal input/output functions
 * 
 * $Id: io_internal.c,v 1.21 2003/02/16 05:32:19 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libcomprex/internal.h>

typedef struct
{
	FILE *fp;
	long startOffset;

} LocalFileData;

static size_t
__localRead(void *ptr, size_t size, size_t nmemb, CxFP *fp)
{
	return fread(ptr, size, nmemb, ((LocalFileData *)fp->moduleData)->fp);
}

static size_t
__localWrite(const void *ptr, size_t size, size_t nmemb, CxFP *fp)
{
	return fwrite(ptr, size, nmemb, ((LocalFileData *)fp->moduleData)->fp);
}

void
__localSeek(CxFP *fp, long offset, int whence)
{
	LocalFileData *data = (LocalFileData *)fp->moduleData;

	if (whence == SEEK_SET)
		offset += data->startOffset;

	fseek(data->fp, offset, whence);
}

void
__localClose(CxFP *fp)
{
	fclose(((LocalFileData *)fp->moduleData)->fp);

	free(fp->moduleData);
	fp->moduleData = NULL;
}

static size_t
__bufferRead(void *ptr, size_t size, size_t nmemb, CxFP *fp)
{
	size_t len, bufferSize, result;
	long offset;

	bufferSize = cxGetFileSize(fp->file);
	
	len = size * nmemb;
	offset = cxTell(fp);

	if (offset > bufferSize)
	{
		fp->eof = 1;
		return 0;
	}
	else if (offset + len > bufferSize)
	{
		len = bufferSize - offset;
		result = (len / size);
	}
	else
		result = nmemb;

	memcpy(ptr, fp->moduleData + offset, len);

	return result;
}

static size_t
__bufferWrite(const void *ptr, size_t size, size_t nmemb, CxFP *fp)
{
	size_t len, bufferSize, result;
	long offset;

	bufferSize = cxGetFileSize(fp->file);

	len = size * nmemb;
	offset = cxTell(fp);

	if (offset + len > bufferSize)
	{
		len = bufferSize - offset;
		result = (len / size);
	}
	else
		result = nmemb;

	memcpy(fp->moduleData + offset, ptr, len);

	return result;
}

void
__bufferSeek(CxFP *fp, long offset, int whence)
{
	/* NOTE: cxSeek() will handle this. */
}

void
__bufferClose(CxFP *fp)
{
	fp->moduleData = NULL;
}

static CxFP *
__openLocalFile(const char *path, CxAccessMode mode)
{
	LocalFileData *data;
	CxFile *file;
	CxFP   *fp;
	FILE   *_fp;

	_fp = fopen(path, (CX_IS_MODE_READ_WRITE(mode) ? "wb" : "rb"));

	if (_fp == NULL)
		return NULL;

	if (CX_IS_MODE_READ_WRITE(mode))
	{
		char *basename;
		file = cxNewFile();

		basename = cxGetBaseName(path);
		cxSetFileName(file, path);
		free(basename);

		cxSetFilePhysicalPath(file, path);
	}
	else
		file = cxMakeFile(path);

	/* Create the file pointer structure. */
	fp = cxNewFp();

	fp->file = file;

	cxSetFpAccessMode(fp, mode);
	cxSetReadFunc(fp,  __localRead);
	cxSetWriteFunc(fp, __localWrite);
	cxSetSeekFunc(fp,  __localSeek);
	cxSetCloseFunc(fp, __localClose);

	MEM_CHECK(data = (LocalFileData *)malloc(sizeof(LocalFileData)));

	data->fp          = _fp;
	data->startOffset = 0;

	fp->moduleData = data;

	return fp;
}

static CxArchive *
__getNestedArchive(CxArchive *parent)
{
	CxFile *file;
	CxFP   *fp;
	CxArchive *destArchive;

	if (parent == NULL)
		return NULL;

	file = cxGetFirstFile(cxGetArchiveRoot(parent));

	if (file == NULL)
	{
		return parent;
	}

	fp = parent->module->ops.archive->openFile(file, CX_MODE_READ_ONLY);

	if (fp != NULL)
	{
		CxArchive *newArchive;
		CxModule  *module;

		fp->file = file;
		CX_LINK(file);

		newArchive = cxNewArchive();

		cxSetArchiveLocal(newArchive,    0);
		cxSetArchiveFileName(newArchive, cxGetFileName(file));
		cxSetArchivePath(newArchive,     cxGetFilePath(file));

		newArchive->fp = fp;
		fp->archive = newArchive;

		module = cxFindOwnerModule(newArchive, fp);

		if (module == NULL)
		{
			fp->archive = NULL;

			cxDestroyArchive(newArchive);

			return parent;
		}

 		newArchive->parent = parent;

		if (cxGetArchiveType(newArchive) == CX_ARCHIVE_SINGLE)
			destArchive = __getNestedArchive(newArchive);
		else
			destArchive = newArchive;
	}
	else
		destArchive = parent;

	return destArchive;
}

/* javabsp */

static CxArchive *
__getNestedArchiveWithPath(const char *path, CxArchive *parent,
						   char **returnedPath, char raw)
{
	CxFile     *file = NULL;
	CxFP       *fp;
	CxArchive  *newArchive, *destArchive;
	CxModule   *module;
	const char *remainingPath;

	if (path == NULL || parent == NULL)
	{
		/* This should never happen, but better safe than sorry. */
		return NULL;
	}

	if (returnedPath != NULL)
		*returnedPath = NULL;

	if (cxGetArchiveType(parent) == CX_ARCHIVE_SINGLE)
	{
		file = cxGetFirstFile(cxGetArchiveRoot(parent));

		remainingPath = path;
	}
	else
	{
		char *tempPath, *s;

		tempPath = strdup(path);

		/* TODO: Account for other directory separators! */
		for (s = tempPath + strlen(tempPath);
			 s != NULL && file == NULL;
			 s = strrchr(tempPath, '/'))
		{
			*s = '\0';
			file = cxGetFile(cxGetArchiveRoot(parent), tempPath);
		}

		if (file != NULL)
			remainingPath = path + strlen(tempPath);
		else
			remainingPath = path;

		free(tempPath);
	}

	if (file == NULL)
	{
		cxDestroyArchive(parent);

		return NULL;
	}

	fp = parent->module->ops.archive->openFile(file, CX_MODE_READ_ONLY);

	if (fp == NULL)
	{
		cxDestroyArchive(parent);

		return NULL;
	}

	fp->file = file;
	CX_LINK(file);

	newArchive = cxNewArchive();

	cxSetArchiveFileName(newArchive, cxGetFileName(file));
	cxSetArchivePath(newArchive,     cxGetFilePath(file));

	newArchive->fp = fp;
	fp->archive = newArchive;

	/* Find the module that supports this. */
	module = cxFindOwnerModule(newArchive, fp);

	if (module == NULL)
	{
		/*
		 * No supported module could be found. Let's return what
		 * we have and let the calling function handle the
		 * remaining path.
		 */
		if (returnedPath != NULL)
			*returnedPath = strdup(remainingPath);

		cxDestroyArchive(newArchive);

		return NULL;
	}

	cxSetArchiveLocal(newArchive,    0);
	cxSetArchiveFileName(newArchive, cxGetFileName(file));
	cxSetArchivePath(newArchive,     cxGetFilePath(file));

	newArchive->parent = parent;

	if (remainingPath == NULL || *remainingPath == '\0')
	{
		if (cxGetArchiveType(newArchive) == CX_ARCHIVE_SINGLE && !raw)
			destArchive = __getNestedArchive(newArchive);
		else
			destArchive = newArchive;
	}
	else
	{
		destArchive = __getNestedArchiveWithPath(remainingPath, newArchive,
												 returnedPath, raw);
	}

	return destArchive;
}

char
cxOpenArchiveOrFile(const char *path, CxAccessMode mode,
					CxFP **destFp, CxArchive **destArchive)
{
	CxArchive *archive = NULL;
	CxFP      *fp      = NULL;
	struct stat sb;
	char *scheme, *newPath;
	char *temp, *path2, *s;
	char isFullPath = 0;
	int   endPos;

	if (path == NULL || mode == CX_MODE_ERROR)
		return 0;

	if (mode == CX_MODE_RAW)
		mode = CX_MODE_RAW | CX_MODE_READ_ONLY;

	if (destFp      != NULL) *destFp      = NULL;
	if (destArchive != NULL) *destArchive = NULL;
	
	path2 = cxGetFullFilePath(path);

	cxProcessUri(path2, &scheme, &newPath);
	
	if (strcmp(scheme, "file"))
	{
		/* We don't support anything besides file yet. */
		free(path2);

		return 0;
	}
	
	endPos = strlen(newPath);
	
	/* XXX */
	if (CX_IS_MODE_READ_WRITE(mode))
	{
		s = newPath;
		isFullPath = 1;
	}
	else
	{
		/* TODO: Account for other directory separators! */
		for (s = newPath + strlen(newPath);
			 s != NULL;
			 s = strrchr(newPath, '/'))
		{
			*s = '\0';

			if (stat(newPath, &sb) == 0)
			{
				/* This file exists on the filesystem. */

				if (!CX_IS_MODE_READ_WRITE(mode) && S_ISDIR(sb.st_mode))
				{
					/*
					 * This is a directory. If this is not the first element,
					 * it means the file is not found. Either way, return NULL.
					 */
					free(scheme);
					free(newPath);
					free(path2);

					return 0;
				}
				else
				{
					/* This is not a directory! :) */

					if (strlen(newPath) == endPos)
					{
						isFullPath = 1;
					}

					break;
				}
			}

			/* The file was not found. Continue. */
		}
	}

	if (s == NULL)
	{
		/* File not found. Return. */
		free(scheme);
		free(newPath);
		free(path2);

		return 0;
	}

	if (isFullPath == 0 || !CX_IS_MODE_RAW(mode))
	{
		CxModule *module;

		/* Build the archive. */
		archive = cxNewArchive();

		cxSetArchiveLocal(archive, 1);

		temp = cxGetFullFilePath(newPath);
		cxSetArchivePath(archive, temp);            /* XXX */
		free(temp);

		temp = cxGetBaseName(newPath);
		cxSetArchiveFileName(archive, temp);
		free(temp);

		fp = __openLocalFile(cxGetArchivePath(archive), mode);

		if (fp == NULL)
		{
			free(scheme);
			free(newPath);
			free(path2);

			cxDestroyArchive(archive);

			return 0;
		}

		archive->fp = fp;
		fp->archive = archive;

		module = cxFindOwnerModule(archive, fp);

		if (module == NULL)
		{
			CX_LINK(fp); /* Save the file pointer. */

			fp->archive = NULL;

			free(scheme);
			free(newPath);
			free(path2);
			cxDestroyArchive(archive);

			if (destFp != NULL)
			{
				cxRewind(fp);

				*destFp = fp;

				return 1;
			}

			cxClose(fp);

			return 0;
		}

		CX_LINK(fp);

		/* File was found. Let's get the remaining part, if there's any. */
		if (isFullPath == 0 && strlen(path2) > strlen(newPath))
		{
			archive = __getNestedArchiveWithPath(path2 + strlen(newPath) + 1,
												 archive, NULL,
												 CX_IS_MODE_RAW(mode));
		}
		else if (cxGetArchiveType(archive) == CX_ARCHIVE_SINGLE)
		{
			archive = __getNestedArchive(archive);
		}

		if (archive == NULL && isFullPath == 0)
		{
			free(scheme);
			free(newPath);
			free(path2);

			return 0;
		}
	}

	/* Okay, create the file structure. */
	if (CX_IS_MODE_RAW(mode) && destFp != NULL)
	{
		if (isFullPath == 1 || archive == NULL)
		{
			fp = __openLocalFile(newPath, mode);
		}
		else
		{
			fp = archive->fp;
			CX_LINK(fp);

			cxDestroyArchive(archive);
			archive = NULL;
		}

		*destFp = fp;
	}
	else if (destFp != NULL)
	{
		if (cxGetArchiveType(archive) == CX_ARCHIVE_SINGLE)
		{
			CxFile *file = cxGetFirstFile(cxGetArchiveRoot(archive));

			if (destFp != NULL) /* ? */
				cxClose(fp);

			fp = archive->module->ops.archive->openFile(file, mode);
			fp->file = file;
			fp->archive = archive;

			CX_LINK(archive);
		}
	}

	/* Clean up. */
	free(scheme);
	free(newPath);
	free(path2);

	if (destFp != NULL)
		*destFp = fp;
	else if (fp != NULL)
		cxClose(fp);

	if (destArchive != NULL)
		*destArchive = archive;
	else
		cxCloseArchive(archive);

	return 1;
}

char
cxOpenArchiveOrFile2(CxFP *fp, CxAccessMode mode, CxFP **destFp,
					 CxArchive **destArchive)
{
	if (fp == NULL)
		return 0;

	if (destFp      != NULL) *destFp      = NULL;
	if (destArchive != NULL) *destArchive = NULL;

	if (CX_IS_MODE_RAW(mode))
	{
		if (destFp != NULL)
			*destFp = fp;

		return 1;
	}
	else
	{
		CxArchive *archive;
		CxModule  *module;

		/* Build the archive. */
		archive = cxNewArchive();

		cxSetArchiveLocal(archive, 1);

		archive->fp = fp;
		fp->archive = archive;

		/* Try to find the module that owns this. */
		module = cxFindOwnerModule(archive, fp);

		if (module == NULL)
		{
			CX_LINK(fp); /* Save the file pointer. */

			fp->archive = NULL;

			cxDestroyArchive(archive);

			if (destFp != NULL)
			{
				cxRewind(fp);

				*destFp = fp;

				return 1;
			}

			cxClose(fp);

			return 0;
		}

		CX_LINK(archive->fp);

		if (cxGetArchiveType(archive) == CX_ARCHIVE_SINGLE)
		{
			archive = __getNestedArchive(archive);
		}

		if (destFp != NULL)
		{
			if (cxGetArchiveType(archive) == CX_ARCHIVE_SINGLE)
			{
				CxFile *file = cxGetFirstFile(cxGetArchiveRoot(archive));

				fp = archive->module->ops.archive->openFile(file, mode);
				fp->file = file;
				fp->archive = archive;
			}
		}

		if (destFp != NULL)
			*destFp = fp;
		else
			cxClose(fp);

		if (destArchive != NULL)
			*destArchive = archive;
		else
			cxDestroyArchive(archive);

		return 1;
	}

	cxClose(fp);

	return 0;
}

char
cxInternalOpenStream(FILE *stream, CxAccessMode mode, CxFP **destFp,
					 CxArchive **destArchive)
{
	LocalFileData *data;
	struct stat sb;
	CxFile *file;
	CxFP *fp;
	
	if (stream == NULL)
		return 0;

	if (fstat(fileno(stream), &sb) != 0)
		return 0;

	file = cxNewFile();

	cxSetFileSize(file, sb.st_size);
	cxSetFileMode(file, sb.st_mode);
	cxSetFileUid(file,  sb.st_uid);
	cxSetFileGid(file,  sb.st_gid);
	cxSetFileDate(file, sb.st_mtime);

	cxSetFileLocal(file, 1);

	/* Create the file pointer. */
	fp = cxNewFp();

	fp->file = file;

	cxSetReadFunc(fp,  __localRead);
	cxSetWriteFunc(fp, __localWrite);
	cxSetSeekFunc(fp,  __localSeek);
	cxSetCloseFunc(fp, __localClose);

	MEM_CHECK(data    = (LocalFileData *)malloc(sizeof(LocalFileData)));
	data->fp          = stream;
	data->startOffset = ftell(stream);

	fp->moduleData = data;

	return cxOpenArchiveOrFile2(fp, mode, destFp, destArchive);
}

char
cxInternalOpenBuffer(char *buffer, size_t size, CxAccessMode mode,
					 CxFP **destFp, CxArchive **destArchive)
{
	CxFP *fp;
	CxFile *file;

	if (buffer == NULL || size == 0)
		return 0;

	/* Create the file. */
	file = cxNewFile();

	cxSetFileSize(file, size);
	cxSetFileCompressedSize(file, size);

	/* Create the file pointer. */
	fp = cxNewFp();

	fp->file = file;

	cxSetFpAccessMode(fp, mode);
	cxSetReadFunc(fp,  __bufferRead);
	cxSetWriteFunc(fp, __bufferWrite);
	cxSetSeekFunc(fp,  __bufferSeek);
	cxSetCloseFunc(fp, __bufferClose);

	fp->moduleData = buffer;

	return cxOpenArchiveOrFile2(fp, mode, destFp, destArchive);
}

CxStatus
cxInternalExtractFile(CxFile *file, const char *destPath)
{

	size_t s;
	char buffer[4096];
	CxFP *inFp;
	FILE *outFp;
	char *basePath;

	if (cxGetFsNodeType(file) != CX_FSNODETYPE_FILE)
		return CX_NOT_SUPPORTED;

	/* Open the input file. */
	inFp = cxOpenFileHandle(file, CX_MODE_READ_ONLY);

	if (inFp == NULL)
		return CX_ERROR;

	/* Open the output file. */
	if (destPath == NULL)
		destPath = cxGetFileName(file);

	basePath = cxGetBasePath(destPath);
	cxMakePhysDirs(basePath, cxGetFileArchive(file));
	free(basePath);

	outFp = fopen(destPath, "w");

	if (outFp == NULL)
	{
		cxClose(inFp);

		return CX_ERROR; /* XXX Support file not found? */
	}

	/* Write out the contents. */
	while ((s = cxRead(buffer, sizeof(char), 4096, inFp)) > 0)
		fwrite(buffer, sizeof(char), s, outFp);

	/* Close up. */
	fclose(outFp);
	cxClose(inFp);

	cxApplyFsNodeInfo(file, destPath);

	return CX_SUCCESS;
}
