/**
 * @file libcomprex/archive.c Archive API
 * 
 * $Id: archive.c,v 1.33 2003/01/02 02:49:38 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libcomprex/internal.h>
#include <libcomprex/archive.h>
#include <libcomprex/utils.h>
#include <comprex.h>

CxArchive *
cxNewArchive(void)
{
	CxArchive *archive;

	MEM_CHECK(archive = (CxArchive*)malloc(sizeof(CxArchive)));
	memset(archive, 0, sizeof(CxArchive));

	cxSetArchiveType(archive,       CX_ARCHIVE_UNKNOWN);
	cxSetArchiveAccessMode(archive, CX_MODE_READ_WRITE);

	archive->root = cxNewDirectory();
	cxSetDirArchive(archive->root, archive);

#if 0
	cxSetDirName(archive->root, "/");
	cxSetDirPath(archive->root, "/");
#endif

	CX_LINK(archive);

	return archive;
}

void
cxDestroyArchive(CxArchive *archive)
{
	CxArchive *parent;
	CxFP *fp;

	if (archive == NULL || archive->refCount == 0)
		return;

	CX_UNLINK(archive);

	if (archive->refCount > 0)
		return;

	parent = archive->parent;
	fp     = archive->fp;

	if (archive->module != NULL &&
		archive->module->ops.archive->closeArchive != NULL)
	{
		archive->module->ops.archive->closeArchive(archive);
	}

	if (!cxIsArchiveLocal(archive) &&
	     cxGetArchivePhysicalPath(archive) != NULL)
	{
		unlink(cxGetArchivePhysicalPath(archive));
	}

	if (archive->name     != NULL) free(archive->name);
	if (archive->path     != NULL) free(archive->path);
	if (archive->physPath != NULL) free(archive->physPath);

	/* XXX This shouldn't be NULL, but check anyway. */
	if (archive->moduleData != NULL)
		free(archive->moduleData);

	/* Close the associated file pointer. */
	cxClose(fp);

	cxDestroyDirectory(archive->root);

	free(archive);

	/* Destroy the parent, if it's not in use. */
	if (parent != NULL)
		cxDestroyArchive(parent);
}

void
cxSetArchiveModule(CxArchive *archive, CxModule *module)
{
	if (archive == NULL)
		return;

	archive->module = module;
}

void
cxSetArchiveType(CxArchive *archive, CxArchiveType type)
{
	if (archive == NULL)
		return;

	archive->type = type;
}

void
cxSetArchiveAccessMode(CxArchive *archive, CxAccessMode mode)
{
	if (archive == NULL)
		return;

	archive->accessMode = mode;
}

void
cxSetArchiveFileName(CxArchive *archive, const char *filename)
{
	if (archive == NULL)
		return;

	if (archive->name != NULL)
		free(archive->name);
	
	archive->name = (filename == NULL ? NULL : strdup(filename));
}

void
cxSetArchivePath(CxArchive *archive, const char *path)
{
	if (archive == NULL)
		return;

	if (archive->path != NULL)
		free(archive->path);

	archive->path = (path == NULL ? NULL : strdup(path));
}

void
cxSetArchivePhysicalPath(CxArchive *archive, const char *path)
{
	if (archive == NULL)
		return;

	if (archive->physPath != NULL)
		free(archive->physPath);

	archive->physPath = (path == NULL ? NULL : strdup(path));
}

void
cxSetArchiveFileSize(CxArchive *archive, unsigned int fileSize)
{
	if (archive == NULL)
		return;

	archive->fileSize = fileSize;
}

void
cxSetArchiveSize(CxArchive *archive, unsigned int archiveSize)
{
	if (archive == NULL)
		return;

	archive->archiveSize = archiveSize;
}

void
cxSetArchiveLocal(CxArchive *archive, char isLocal)
{
	if (archive == NULL)
		return;

	archive->isLocal = (isLocal == 0 ? 0 : 1);
}

void
cxSetArchiveExtractCallback(CxArchive *archive,
							void (*callback) (CxArchive *, CxFile *,
											  size_t, size_t))
{
	if (archive == NULL) 
		return;

	archive->ex_callback = callback;
}

CxModule *
cxGetArchiveModule(CxArchive *archive)
{
	if (archive == NULL)
		return NULL;

	return archive->module;
}

CxArchiveType
cxGetArchiveType(CxArchive *archive)
{
	if (archive == NULL)
		return CX_ARCHIVE_UNKNOWN;

	return archive->type;
}

CxAccessMode
cxGetArchiveAccessMode(CxArchive *archive)
{
	if (archive == NULL)
		return CX_MODE_ERROR;

	return archive->accessMode;
}

const char *
cxGetArchiveFileName(CxArchive *archive)
{
	if (archive == NULL)
		return NULL;

	return archive->name;
}

const char *
cxGetArchivePath(CxArchive *archive)
{
	if (archive == NULL)
		return NULL;

	return archive->path;
}

const char *
cxGetArchivePhysicalPath(CxArchive *archive)
{
	if (archive == NULL)
		return 0;

	return archive->physPath;
}

size_t
cxGetArchiveFileCount(CxArchive *archive)
{
	if (archive == NULL)
		return 0;

	return archive->fileCount;
}

unsigned int
cxGetArchiveFileSize(CxArchive *archive)
{
	if (archive == NULL)
		return 0;

	return archive->fileSize;
}

unsigned int
cxGetArchiveSize(CxArchive *archive)
{
	if (archive == NULL)
		return 0;

	return archive->archiveSize;
}

char
cxIsArchiveLocal(CxArchive *archive)
{
	if (archive == NULL)
		return 0;

	return archive->isLocal;
}

CxDirectory *
cxGetArchiveRoot(CxArchive *archive)
{
	if (archive == NULL)
		return NULL;

	return archive->root;
}

