from glob import glob

import os

from typing import AnyStr

from e3.testsuite.driver.classic import TestAbortWithError
from e3.testsuite.driver.diff import PatternSubstitute
from e3.fs import cp

from drivers.base_driver import BaseDriver


class Address_Hider(PatternSubstitute):
    """
    Refiner that identifies addresses from a symbolic Ada traceback and hides
    the addresses, to stabilize the output
    """
    def __init__(self):
        super().__init__(
            pattern=r"0x[0-9a-f]{8,}",
            replacement="<addr>"
        )


class GNATTestTgenDriver(BaseDriver):
    """
    Test driver to execute gnattest in test generation mode (with TGen)
    The driver will attempt to run "gnattest --gen-test-vectors
    --dump-test-inputs -P<proj>" on the first .gpr file found on the test
    directory, and log the output, then build and run the generated test
    harness. The test harness execution log is also recorded in the output.
    """

    @property
    def baseline_file(self):
        """
        Only override the baseline filename when in gnatfuzz test mode
        """
        if os.environ.get("GNATTEST_GNATFUZZ", None):
            return "gnattest_baseline.out", False
        else:
            return super().baseline_file

    @property
    def tool_dirname(self):
        return "test"

    @property
    def output_refiners(self):
        return super().output_refiners + [Address_Hider()]

    def run(self):
        # generate a default gpr file if the test asks for one
        if self.test_env.get("default-gpr", False):
            gpr_file = os.path.join(self.working_dir(), "user_project.gpr")
            cp(
                os.path.join(
                    self.shared_dir(),
                    "gnatfuzz_default_resources",
                    "user_project.gpr",
                ),
                gpr_file,
            )

        # Try to locate one in the test dir otherwise
        else:
            gpr_file = glob(
                os.path.join(self.working_dir(), "*.gpr"), recursive=False
            )
            if len(gpr_file) == 0:
                raise TestAbortWithError(
                    "Could not find gpr project file on which to run gnattest:"
                    f" {gpr_file} (cwd: {self.working_dir()})"
                )

            gpr_file = gpr_file[0]

        gnattest_args = [
            "gnattest",
            f"-P{gpr_file}",
            "--gen-test-vectors",
            "-gnat2022",
        ]

        # Suppress test input dumping if specified in the test.yaml file
        if not self.test_env.get("suppress_test_dump", False):
            gnattest_args.append("--dump-test-inputs")

        if os.environ.get("GNATTEST_DEBUG", None):
            # Enable debug log and preserve generation harness
            gnattest_args.extend(["-d1", "-dn"])
        else:
            # Silence execution
            gnattest_args.append("-q")

        # Append extra arguments specified in the test.yaml file
        gnattest_args += self.test_env.get("extra_gnattest_args", [])

        # Generate a test harness with tests generated by TGen
        self.shell(gnattest_args)

        # Get the object directory to be able to build the harness. This assumes
        # there is no Harness_Dir attribute in the project file.
        harness_dir = os.path.join(
            self.working_dir(), "obj", "gnattest", "harness"
        )
        td_prj = os.path.join(harness_dir, "test_driver.gpr")
        support_lib_prj = os.path.join(
            harness_dir, "tgen_support", "tgen_support.gpr"
        )
        if not os.path.exists(td_prj):
            raise TestAbortWithError("Could not locate test harness project")

        gprbuild_args = [
            "gprbuild",
            f"-P{td_prj}",
            "-q"
        ]

        # Determine if the test harness needs to be compiled for test dumping mode
        has_test_dump = "--dump-test-inputs" in gnattest_args
        if has_test_dump:
            gprbuild_args += [
                "--src-subdirs=gnattest-instr",
                f"--implicit-with={support_lib_prj}",
            ]

        # Append extra gpbuild args if specified in the test.yaml file
        gprbuild_args += self.test_env.get("extra_gprbuild_args", [])

        self.shell(gprbuild_args)

        # Finally, run the test harness. Do not register failure exit codes as
        # it is likely that some tests may crash the subprogram under test.
        td_runner = os.path.join(os.path.dirname(td_prj), "test_runner")
        self.shell(
            [td_runner] + self.test_env.get("extra_run_args", []),
            catch_error=False
        )

        # Log the number of test files that were created by the test runner, if
        # test input dumping was enabled.
        if has_test_dump:
            num_tests = len(
                glob(os.path.join(self.working_dir(), "tgen_test_inputs", "*"))
            )
            self.output.log += f"Test runner dumped {num_tests} tests\n"
