/*	$Id: utils.c,v 1.2 2005/07/24 14:09:55 itohy Exp $	*/

/*
 * Copyright (c) 2005 ITOH Yasufumi.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form is unlimited.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <inttypes.h>	/* int*_t, uint*_t */
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "utils.h"

/*----------------------------------------------------------------------
 * progress bar
 */

#define PROGRESS_LEN	79
#define PROGRESS_STARS	\
"******************************************************************************"

void
progress_init(p, max, msg)
	struct progress *p;
	uint32_t max;
	const char *msg;
{
	char buf[256];

	if (max == 0 || !isatty(1)) {
		p->active = 0;
		return;
	}
	printf("%s (%u)\n", msg, (unsigned)max);
	p->active = 1;
	p->maxcnt = max;
	p->maxnumlen = sprintf(buf, "%u", (unsigned)max);	/* length */
	p->maxbar = PROGRESS_LEN - 1 - p->maxnumlen;
	p->curbar = 0;
}

void
progress_setval(p, val)
	struct progress *p;
	uint32_t val;
{
	static pid_t pgrp = -1;
	int bar;

	if (!p->active)
		return;

	bar = p->maxbar * val / p->maxcnt;
	if (bar != p->curbar) {
		if (pgrp == -1)
			pgrp = getpgrp();
		if (tcgetpgrp(1) != pgrp)
			return;		/* not in foreground process group */

		p->curbar = bar;
		printf("\r%.*s%*s %*u",
		    bar, PROGRESS_STARS, p->maxbar - bar, "", p->maxnumlen,
		    (unsigned)val);
		fflush(stdout);
	}
}

void
progress_end(p)
	struct progress *p;
{

	if (p->active && p->curbar > 0)
		printf("\n");
	p->active = 0;
}

/*----------------------------------------------------------------------
 * variable length array
 */

#define VARARRAY_CHUNK	256	/* must be a power of 2 */

void
vararray_init(v, itemsize, dtor)
	struct vararray *v;
	size_t itemsize;
	void (*dtor)__P((void *));	/* destructor of an item */
{

	v->itemsize = itemsize;
	v->dtor = dtor;
	v->allocsize = 0;
	v->nitem = 0;
	v->buf = NULL;
}

void
vararray_removeall(v)
	struct vararray *v;
{
	char *p, *pend;

	if (v->buf) {
		if (v->dtor) {
			for (p = v->buf, pend = p + v->itemsize * v->nitem;
			    p < pend;
			    p += v->itemsize)
				(*v->dtor)(p);
		}
		free(v->buf);
	}
	v->allocsize = 0;
	v->nitem = 0;
	v->buf = NULL;
}

void *
vararray_item(v, idx)
	struct vararray *v;
	int idx;
{

	return (char *)v->buf + v->itemsize * idx;
}

void *
vararray_last(v)
	struct vararray *v;
{

	if (v->nitem == 0)
		return NULL;
	return vararray_item(v, v->nitem - 1);
}

int
vararray_add(v, obj)
	struct vararray *v;
	void *obj;
{
	size_t newsize;
	void *newbuf;

	if (v->allocsize < v->itemsize * (v->nitem + 1)) {
		newsize = v->allocsize + VARARRAY_CHUNK + v->itemsize;
		newsize &= ~((size_t)VARARRAY_CHUNK - 1);
		if ((newbuf = realloc(v->buf, newsize)) == NULL) {
			perror("realloc");
			return -1;
		}
		v->allocsize = newsize;
		v->buf = newbuf;
	}

	memcpy(vararray_item(v, v->nitem), obj, v->itemsize);
	v->nitem++;

	return 0;
}

void
vararray_qsort(v, compar)
	struct vararray *v;
	int (*compar)__P((const void *, const void *));
{

	qsort(v->buf, v->nitem, v->itemsize, compar);
}

/*----------------------------------------------------------------------
 * variable length buffer
 */

#define VARBUF_CHUNK	256	/* must be a power of 2 */

void
varbuf_init(b)
	struct varbuf *b;
{

	b->size = 0;
	b->allocsize = 0;
	b->buf = NULL;
}

/* free and reinit varbuf */
void
varbuf_reinit(b)
	struct varbuf *b;
{

	if (b->buf)
		free(b->buf);
	varbuf_init(b);
}

int
varbuf_add(b, ptr, sz)
	struct varbuf *b;
	void *ptr;
	size_t sz;
{
	void *newbuf;
	size_t newsize;

	if (b->allocsize < b->size + sz) {
		newsize = b->allocsize + VARBUF_CHUNK + sz;
		newsize &= ~((size_t)VARBUF_CHUNK - 1);
		if ((newbuf = realloc(b->buf, newsize)) == NULL) {
			perror("realloc");
			return -1;
		}
		b->allocsize = newsize;
		b->buf = newbuf;
	}
	memcpy((char *)b->buf + b->size, ptr, sz);
	b->size += sz;

	return 0;
}

void *
varbuf_getbuf(b, psize)
	struct varbuf *b;
	size_t *psize;		/* output */
{
	void *p;

	if (psize)
		*psize = b->size;

	if ((p = realloc(b->buf, b->size)) == NULL) {
		perror("realloc");
		return NULL;
	}

	varbuf_init(b);	/* just in case */

	return p;
}

/*----------------------------------------------------------------------
 * others
 */

void *
stash(str)
	const void *str;
{

	return str ? strdup(str) : NULL;
}

void *
stash2(src, len)
	const void *src;
	size_t len;
{
	char *p;

	if ((p = malloc(len + 1)) == NULL)
		return NULL;

	memcpy(p, src, len);
	p[len] = '\0';

	return p;
}

void
freestash(p)
	void *p;
{

	if (p)
		free(p);
}
