/*
 * ksi_type.c
 *
 * Copyright (C) 2009-2010, ivan demakov.
 *
 * The software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * The software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the software; see the file COPYING.LESSER.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 *
 * Author:        ivan demakov <ksion@users.sourceforge.net>
 * Creation date: Fri Feb 13 23:37:53 2009
 * Last Update:   Sat Apr 24 22:50:31 2010
 *
 */

#include "ksi_type.h"
#include "ksi_printf.h"
#include "ksi_int.h"
#include "ksi_env.h"
#include "ksi_comp.h"
#include "ksi_klos.h"
#include "ksi_util.h"


static unsigned
hash_sym (void* obj, unsigned num, void* unused)
{
    return ksi_hash_str (KSI_SYM_PTR (obj), KSI_SYM_LEN (obj), num);
}

static int
cmp_sym (void* x1, void* x2, void* unused)
{
    int len1 = KSI_SYM_LEN (x1);
    int len2 = KSI_SYM_LEN (x2);
    const char* str1 = KSI_SYM_PTR (x1);
    const char* str2 = KSI_SYM_PTR (x2);

    int cmp = memcmp (str1, str2, MIN (len1, len2));
    return cmp ? cmp : len1 - len2;
}

static ksi_obj
ksi_void_proc (int ac, ksi_obj *av)
{
    return ksi_void;
}

static ksi_obj
ksi_true_proc (int ac, ksi_obj *av)
{
    return ksi_true;
}

static ksi_obj
ksi_false_proc (int ac, ksi_obj *av)
{
    return ksi_false;
}

struct Ksi_Data *
ksi_internal_data(void)
{
    static struct Ksi_Data *data = 0;
    int i;

    if (!ksi_int_data) {
        ksi_errlog_msg (ERRLOG_EMERG, "Ksi library is not initialized. (You should call ksi_init() before any other ksi function)\n");
        ksi_quit();
    }

    if (!data) {
        data = ksi_malloc_eternal(sizeof (*data));
        bzero(data, sizeof *data);

        KSI_INIT_LOCK(data->lock);

        data->obj_nil = ksi_new_imm(KSI_TAG_IMM);
        data->obj_false = ksi_new_imm(KSI_TAG_IMM);
        data->obj_true = ksi_new_imm(KSI_TAG_IMM);
        data->obj_void = ksi_new_imm(KSI_TAG_IMM);
        data->obj_eof = ksi_new_imm(KSI_TAG_IMM);
        data->obj_err = ksi_new_imm(KSI_TAG_IMM);

        data->symtab = ksi_new_valtab(1000, hash_sym, cmp_sym, 0);
        data->keytab = ksi_new_valtab(1000, hash_sym, cmp_sym, 0);
        data->envtab = 0; /* initialized in ksi_env.c */

        data->sym_quote = ksi_str02sym ("quote");
        data->sym_begin = ksi_str02sym ("begin");
        data->sym_if = ksi_str02sym ("if");
        data->sym_and = ksi_str02sym ("and");
        data->sym_or = ksi_str02sym ("or");
        data->sym_lambda = ksi_str02sym ("lambda");
        data->sym_define = ksi_str02sym ("define");
        data->sym_defmacro = ksi_str02sym ("define-macro!");
        data->sym_set = ksi_str02sym ("set!");
        data->sym_case = ksi_str02sym ("case");
        data->sym_cond = ksi_str02sym ("cond");
        data->sym_else = ksi_str02sym ("else");
        data->sym_let = ksi_str02sym ("let");
        data->sym_letrec = ksi_str02sym ("letrec");
        data->sym_letrec_star = ksi_str02sym ("letrec*");
        data->sym_letstar = ksi_str02sym ("let*");
        data->sym_quasiquote = ksi_str02sym ("quasiquote");
        data->sym_unquote = ksi_str02sym ("unquote");
        data->sym_unquote_splicing = ksi_str02sym ("unquote-splicing");
        data->sym_syntax = ksi_str02sym ("syntax");
        data->sym_quasisyntax = ksi_str02sym ("quasisyntax");
        data->sym_unsyntax = ksi_str02sym ("unsyntax");
        data->sym_unsyntax_splicing = ksi_str02sym ("unsyntax-splicing");
        data->sym_import = ksi_str02sym ("import");
        data->sym_export = ksi_str02sym ("export");
        data->sym_library = ksi_str02sym ("library");
        data->sym_rename = ksi_str02sym ("rename");
        data->sym_prefix = ksi_str02sym ("prefix");
        data->sym_except = ksi_str02sym ("except");
        data->sym_only = ksi_str02sym ("only");
        data->sym_for = ksi_str02sym ("for");

        data->sym_plus = ksi_str02sym ("+");
        data->sym_minus = ksi_str02sym ("-");
        data->sym_arrow = ksi_str02sym ("=>");
        data->sym_dots = ksi_str02sym ("...");

        data->sym_inactive = ksi_str02sym ("inactive");
        data->sym_wait = ksi_str02sym ("wait");
        data->sym_ready = ksi_str02sym ("ready");
        data->sym_timeout = ksi_str02sym ("timeout");

        data->sym_apply_generic = ksi_str02sym ("apply-generic");
        data->sym_no_applicable_method = ksi_str02sym ("no-applicable-method");
        data->sym_no_next_method = ksi_str02sym ("no-next-method");

        data->sym_cname = ksi_str02sym (S_cname);
        data->sym_gname = ksi_str02sym (S_gname);
        data->sym_dsupers = ksi_str02sym (S_dsupers);
        data->sym_dslots = ksi_str02sym (S_dslots);
        data->sym_defargs = ksi_str02sym (S_defargs);
        data->sym_cpl = ksi_str02sym (S_cpl);
        data->sym_slots = ksi_str02sym (S_slots);
        data->sym_nfields = ksi_str02sym (S_nfields);
        data->sym_gns = ksi_str02sym (S_gns);
        data->sym_gf = ksi_str02sym (S_gf);
        data->sym_specs = ksi_str02sym (S_specs);
        data->sym_proc = ksi_str02sym (S_proc);
        data->sym_methods = ksi_str02sym (S_methods);
        data->sym_arity = ksi_str02sym (S_arity);
        data->sym_combination = ksi_str02sym (S_combination);
        data->sym_after = ksi_str02sym (S_after);
        data->sym_before = ksi_str02sym (S_before);
        data->sym_around = ksi_str02sym (S_around);
        data->sym_primary = ksi_str02sym (S_primary);

        data->key_initform = ksi_str02key (K_initform);
        data->key_initarg = ksi_str02key (K_initarg);
        data->key_defargs = ksi_str02key (K_defargs);
        data->key_type = ksi_str02key (K_type);
        data->key_name = ksi_str02key (K_name);
        data->key_dsupers = ksi_str02key (K_dsupers);
        data->key_dslots = ksi_str02key (K_dslots);
        data->key_specs = ksi_str02key (K_specs);
        data->key_proc = ksi_str02key (K_proc);
        data->key_gf = ksi_str02key (K_gf);
        data->key_arity = ksi_str02key (K_arity);
        data->key_combination = ksi_str02key (K_combination);

        data->not_proc = ksi_new_imm (KSI_TAG_NOT);
        data->eq_proc = ksi_new_imm (KSI_TAG_EQP);
        data->eqv_proc = ksi_new_imm (KSI_TAG_EQVP);
        data->equal_proc = ksi_new_imm (KSI_TAG_EQUALP);
        data->list_proc = ksi_new_imm (KSI_TAG_LIST);
        data->vector_proc = ksi_new_imm (KSI_TAG_MK_VECTOR);
        data->list2vector_proc = ksi_new_imm (KSI_TAG_LIST2VECTOR);
        data->append_proc = ksi_new_imm (KSI_TAG_APPEND);
        data->nullp_proc = ksi_new_imm (KSI_TAG_NULLP);
        data->pairp_proc = ksi_new_imm (KSI_TAG_PAIRP);
        data->listp_proc = ksi_new_imm (KSI_TAG_LISTP);
        data->cons_proc = ksi_new_imm (KSI_TAG_CONS);
        data->car_proc = ksi_new_imm (KSI_TAG_CAR);
        data->cdr_proc = ksi_new_imm (KSI_TAG_CDR);
        data->memq_proc = ksi_new_imm (KSI_TAG_MEMQ);
        data->memv_proc = ksi_new_imm (KSI_TAG_MEMV);
        data->member_proc = ksi_new_imm (KSI_TAG_MEMBER);
        data->vectorp_proc = ksi_new_imm (KSI_TAG_VECTORP);
        data->apply_proc = ksi_new_imm (KSI_TAG_APPLY);
        data->call_cc_proc = ksi_new_imm (KSI_TAG_CALL_CC);
        data->call_vs_proc = ksi_new_imm (KSI_TAG_CALL_WITH_VALUES);

        data->void_proc = (ksi_obj) ksi_new_prim("void", ksi_void_proc, KSI_CALL_REST0, 0);
        data->true_proc = (ksi_obj) ksi_new_prim("true", ksi_true_proc, KSI_CALL_REST0, 0);
        data->false_proc = (ksi_obj) ksi_new_prim("false", ksi_false_proc, KSI_CALL_REST0, 0);

        i = 12;
        data->session_id = ksi_malloc_data(i);
        ksi_random_bits(data->session_id, i);
        data->session_id = ksi_base64(data->session_id, i);
        for (i = 0; data->session_id[i]; i++) {
            if (data->session_id[i] == '/')
                data->session_id[i] = '-';
        }
        data->gensym_num = ksi_long2num(0);

        init_top_classes ();

        data->null_port = ksi_new_nul_port ();

        data->syntax_env = ksi_get_lib_env("ksi", "core", "syntax", 0);
    }

    return data;
}


int
ksi_default_tag_equal (ksi_etag tag, ksi_obj x1, ksi_obj x2, int deep)
{
    return 0;
}

const char*
ksi_default_tag_print (ksi_etag tag, ksi_obj x, int slashify)
{
    if (x -> o.itag == KSI_TAG_BROKEN)
        return ksi_aprintf ("#<broken %s %p>", tag->type_name, x);
    else
        return ksi_aprintf ("#<%s %p>", tag->type_name, x);
}


ksi_obj
ksi_new_imm (enum ksi_tag_t tag)
{
    ksi_obj x = (ksi_obj) ksi_malloc(sizeof *x);
    x->o.itag = tag;
    return x;
}

ksi_obj
ksi_new_core (enum ksi_tag_t tag)
{
    ksi_core x = (ksi_core) ksi_malloc (sizeof *x);
    x->o.itag = KSI_TAG_CORE;
    x->core = tag;
    return (ksi_obj) x;
}

ksi_obj
ksi_new_values (int num, ksi_obj* val)
{
    if (num == 1) {
        return val[0];
    } else {
        ksi_values x = ksi_malloc (sizeof(*x));
        x->o.itag = KSI_TAG_VALUES;
        x->vals = ksi_new_list (num, val);

        return (ksi_obj) x;
    }
}

ksi_obj
ksi_void_p (ksi_obj x)
{
    return x == ksi_void ? ksi_true : ksi_false;
}

ksi_obj
ksi_bool_p (ksi_obj x)
{
    return (x == ksi_true || x == ksi_false) ? ksi_true : ksi_false;
}

ksi_obj
ksi_bool_eq_p (int ac, ksi_obj *av)
{
    int i;
    ksi_obj c1, c2;

    if (ac >= 1) {
        c1 = av[0];
        KSI_CHECK(c1, c1 == ksi_true || c1 == ksi_false, "boolean=?: invalid boolean");
        for (i = 1; i < ac; i++) {
            c2 = av[i];
            KSI_CHECK(c2, c2 == ksi_true || c2 == ksi_false, "boolean=?: invalid boolean");
            if (c1 != c2)
                return ksi_false;
            c1 = c2;
        }
    }
    return ksi_true;
}

ksi_obj
ksi_int2bool (int x)
{
    return x ? ksi_true : ksi_false;
}

int
ksi_bool2int (ksi_obj x)
{
    return x == ksi_false ? 0 : 1;
}

ksi_obj
ksi_not (ksi_obj x)
{
    return x == ksi_false ? ksi_true : ksi_false;
}


static inline int
eff_tag (ksi_obj x)
{
  int tag = x->o.itag;
  if (tag == KSI_TAG_CONST_STRING)
    return KSI_TAG_STRING;
  if (tag == KSI_TAG_CONST_VECTOR)
    return KSI_TAG_VECTOR;
  if (tag == KSI_TAG_CONST_PAIR)
    return KSI_TAG_PAIR;
  if (tag == KSI_TAG_LOCAL || tag == KSI_TAG_IMPORTED)
    return KSI_TAG_FREEVAR;

  return tag;
}

ksi_obj
ksi_eqv_p (ksi_obj x1, ksi_obj x2)
{
    if (x1 == x2)
        return ksi_true;

    if (eff_tag (x1) != eff_tag (x2))
        return ksi_false;

    /* now x1->itag == x2->itag */

    if (x1->o.itag == KSI_TAG_EXTENDED) {
        ksi_etag etag = KSI_EXT_TAG (x1);
        return (etag -> equal (etag, x1, x2, 0) ? ksi_true : ksi_false);
    }

    /*ksi_debug("eqv? %s %s", ksi_obj2str(x1), ksi_obj2str(x2));*/
    switch (x1->o.itag) {
    case KSI_TAG_CHAR:
        return (KSI_CHAR_CODE (x1) == KSI_CHAR_CODE (x2) ? ksi_true : ksi_false);

    case KSI_TAG_STRING:
    case KSI_TAG_CONST_STRING:
        return ksi_string_eqv_p (x1, x2);

    case KSI_TAG_BIGNUM:
    case KSI_TAG_FLONUM:
        return ksi_num_eqv_p (x1, x2);

    case KSI_TAG_INSTANCE:
        return ksi_inst_eqv_p (x1, x2);
    }

    return ksi_false;
}

static int
frame_equal (ksi_frame f1, ksi_frame f2)
{
  int i;

again:
  if (f1 == f2)
    return 1;

  if (!f1 || !f2)
    return 0;
  if (f1->num != f2->num)
    return 0;
  if (f1->env != f2->env)
    return 0;

  for (i = 0; i < f1->num; i++)
    if (ksi_equal_p (f1->vals[i], f2->vals[i]) == ksi_false)
      return 0;

  f1 = f1->next;
  f2 = f2->next;
  goto again;
}

ksi_obj
ksi_equal_p (ksi_obj x1, ksi_obj x2)
{
    int i;
    ksi_obj z1, z2;

again:
    if (x1 == x2)
        return ksi_true;

    if (eff_tag(x1) != eff_tag(x2))
        return ksi_false;

    /* x1->o.itag == x2->o.itag */

    if (x1->o.itag == KSI_TAG_EXTENDED) {
        ksi_etag etag = KSI_EXT_TAG(x1);
        return (etag->equal(etag, x1, x2, 1) ? ksi_true : ksi_false);
    }

    switch (x1->o.itag) {
    case KSI_TAG_CHAR:
        return (KSI_CHAR_CODE(x1) == KSI_CHAR_CODE(x2) ? ksi_true : ksi_false);

    case KSI_TAG_STRING:
    case KSI_TAG_CONST_STRING:
        return ksi_string_equal_p(x1, x2);

    case KSI_TAG_BIGNUM:
    case KSI_TAG_FLONUM:
        return ksi_num_equal_p(x1, x2);

    case KSI_TAG_PAIR:
    case KSI_TAG_CONST_PAIR:
        for (i = 0, z1 = x1, z2 = x2; ; ) {
            if (ksi_equal_p(KSI_CAR(x1), KSI_CAR(x2)) == ksi_false)
                return ksi_false;

            x1 = KSI_CDR(x1);
            x2 = KSI_CDR(x2);

            /* if cdr's are equal then lists equal, if both lists end,
             * x1 and x2 are ksi_nil (and equal too).
             */
            if (x1 == x2)
                return ksi_true;

            if (!KSI_PAIR_P(x1) || !KSI_PAIR_P(x2))
                goto again;

            if (ksi_equal_p(KSI_CAR(x1), KSI_CAR(x2)) == ksi_false)
                return ksi_false;

            x1 = KSI_CDR(x1);
            x2 = KSI_CDR(x2);

            if (x1 == x2)
                return ksi_true;

            if (!KSI_PAIR_P(x1) || !KSI_PAIR_P(x2))
                goto again;

            z1 = KSI_CDR(z1);
            z2 = KSI_CDR(z2);

            /* if both lists are circular they are equal */
            if (z1 == x1) i |= 1;
            if (z2 == x2) i |= 2;
            if (i == 3)
                return ksi_true;
            KSI_CHECK_EVENTS;
        }

    case KSI_TAG_VECTOR:
    case KSI_TAG_CONST_VECTOR:
        if (KSI_VEC_LEN(x1) == KSI_VEC_LEN(x2)) {
            for (i = 0; i < KSI_VEC_LEN(x1) - 1; i++) {
                if (ksi_equal_p(KSI_VEC_REF(x1, i), KSI_VEC_REF(x2, i)) == ksi_false)
                    return ksi_false;
                KSI_CHECK_EVENTS;
            }
            x1 = KSI_VEC_REF(x1, i);
            x2 = KSI_VEC_REF(x2, i);
            goto again;
        }
        return ksi_false;

    case KSI_TAG_INSTANCE:
        return ksi_inst_equal_p(x1, x2);

    case KSI_TAG_CLOSURE:
        if (KSI_CLOS_NUMS(x1) == KSI_CLOS_NUMS(x2)
            && KSI_CLOS_NARY(x1) == KSI_CLOS_NARY(x2)
            && KSI_CLOS_OPTS(x1) == KSI_CLOS_OPTS(x2)
            && frame_equal(KSI_CLOS_FRM(x1), KSI_CLOS_FRM(x2))) {
            x1 = KSI_CLOS_BODY(x1);
            x2 = KSI_CLOS_BODY(x2);
            goto again;
        }
        return ksi_false;

    case KSI_TAG_VAR0:
    case KSI_TAG_VAR1:
    case KSI_TAG_VAR2:
    case KSI_TAG_VARN:
        if (KSI_VARBOX_NUM(x1) == KSI_VARBOX_NUM(x2) && KSI_VARBOX_LEV(x1) == KSI_VARBOX_LEV(x2))
            return ksi_true;
        return ksi_false;

    case KSI_TAG_FREEVAR:
    case KSI_TAG_LOCAL:
    case KSI_TAG_IMPORTED:
        if (KSI_FREEVAR_SYM(x1) == KSI_FREEVAR_SYM(x2))
            return ksi_true;
        return ksi_false;

    default:
        if (KSI_CODE_P(x1) && KSI_CODE_NUM(x1) == KSI_CODE_NUM(x2)) {
            for (i = 0; i < KSI_CODE_NUM(x1); ++i) {
                if (ksi_equal_p(KSI_CODE_VAL(x1, i), KSI_CODE_VAL(x2, i)) == ksi_false)
                    return ksi_false;
                KSI_CHECK_EVENTS;
            }
            x1 = KSI_CODE_VAL(x1, i);
            x2 = KSI_CODE_VAL(x2, i);
            goto again;
        }
    }

    return ksi_false;
}


ksi_symbol
ksi_lookup_sym (const char* key, size_t len, int append)
{
  ksi_symbol sym;

  /* no need to add 1 for the nul char at the end of a name,
   * because it already counted in the Ksi_Symbol structure.
   */
  if (!append) {
    ksi_symbol sym = (ksi_symbol) alloca(sizeof(*sym) + len);
    sym->len = len;
    memcpy((char *) sym->ptr, key, len);
    ((char *)sym->ptr)[len] = 0;
    return (ksi_symbol) ksi_lookup_vtab (ksi_data->symtab, sym, 0);
  }

  sym = (ksi_symbol) ksi_malloc(sizeof(*sym) + len);
  sym->o.itag = KSI_TAG_SYMBOL;
  sym->len = len;
  memcpy((char *) sym->ptr, key, len);
  ((char *)sym->ptr)[len] = 0;

  return (ksi_symbol) ksi_lookup_vtab (ksi_data->symtab, sym, 1);
}

static char
tohex (int c)
{
  return (c < 10 ? c + '0' : (c-10) + 'a');
}

const char*
ksi_symbol2str (ksi_symbol sym)
{
  int i, k, num, len;
  const char* nam;
  char* ptr;

  len = KSI_SYM_LEN (sym);
  if (len <= 0)
    return "||";

  nam = KSI_SYM_PTR (sym);
  for (i = 0, k = 0, num = 0; i < len; ++i) {
    switch (nam[i]) {
    case '\0':
    case '#':
    case ' ': case '(': case ')': case '[': case ']': case '{': case '}':
    case ';': case '`': case ',': case '\'': case '"':
    case '|': case '\\':
      ++k; num += 4;
      break;

    case '1': case '2': case '3': case '4': case '5':
    case '6': case '7': case '8': case '9': case '0':
      if (i == 0) {
        ++k; num += 4;
      }
      break;

    default:
      if (!isgraph (nam[i])) {
        ++k; num += 4;
      }
    }
  }
  if (k == 0 && nam [len] == '\0')
    return nam;

  ptr = (char*) ksi_malloc_data (len + num + 1);
  k = 0;
  for (i = 0; i < len; ++i) {
    char c = nam[i];
    switch (c) {
    case '\0':
    case '#':
    case ' ': case '(': case ')': case '[': case ']': case '{': case '}':
    case ';': case '`': case ',': case '\'': case '"':
    case '|': case '\\':
      goto hex;

    case '1': case '2': case '3': case '4': case '5':
    case '6': case '7': case '8': case '9': case '0':
      if (i == 0)
        goto hex;
      ptr[k++] = c;
      break;

    default:
      if (!isgraph (c)) {
      hex:
        ptr[k++] = '\\';
        ptr[k++] = 'x';
        ptr[k++] = tohex ((c >> 4) & 0xf);
        ptr[k++] = tohex (c & 0xf);
        ptr[k++] = ';';
      } else {
        ptr[k++] = c;
      }
    }
  }
  ptr[k++] = '\0';

  return ptr;
}

ksi_obj
ksi_symbol_p (ksi_obj x)
{
    return KSI_SYM_P(x) ? ksi_true : ksi_false;
}

ksi_obj
ksi_symbol_eq_p (int ac, ksi_obj *av)
{
    int i;
    ksi_obj c1, c2;

    if (ac >= 1) {
        c1 = av[0];
        KSI_CHECK(c1, KSI_SYM_P(c1), "symbol=?: invalid symbol");
        for (i = 1; i < ac; i++) {
            c2 = av[i];
            KSI_CHECK(c2, KSI_SYM_P(c2), "symbol=?: invalid symbol");
            if (c1 != c2)
                return ksi_false;
            c1 = c2;
        }
    }
    return ksi_true;
}

ksi_obj
ksi_symbol2string (ksi_obj sym)
{
    KSI_CHECK (sym, KSI_SYM_P(sym), "symbol->string: invalid symbol");
    return ksi_str02string(KSI_SYM_PTR(sym));
}

ksi_obj
ksi_string2symbol (ksi_obj str)
{
    KSI_CHECK(str, KSI_STR_P(str), "string->symbol: invalid string");
    return ksi_intern(KSI_STR_PTR(str), KSI_STR_LEN(str));
}

ksi_obj
ksi_gensym (ksi_obj name, ksi_obj unused)
{
    const char *str = 0;
    char *buf = 0;
    int len = 0, sz = 0, ses_len = strlen(ksi_data->session_id);
    ksi_obj num;
    char* ptr;
    int l;

    if (name) {
        if (KSI_STR_P(name)) {
            str = KSI_STR_PTR(name);
            len = KSI_STR_LEN(name);
        } else if (KSI_SYM_P(name)) {
            str = KSI_SYM_PTR(name);
            len = KSI_SYM_LEN(name);
        } else {
            ksi_exn_error(ksi_assertion_s, name, "gensym: invalid object in arg1");
        }
    } else {
        str = "g";
        len = 1;
    }

    for (;;) {
        KSI_LOCK_W(ksi_data->lock);
        ksi_data->gensym_num = ksi_add(ksi_data->gensym_num, ksi_long2num(1));
        num = ksi_data->gensym_num;
        KSI_UNLOCK_W(ksi_data->lock);

        ptr = ksi_num2str(num, 10);
        l = 1 + ses_len + 1 + strlen(ptr);
        if (sz <= len + l) {
            sz = len + l;
            buf = (char*) ksi_malloc_data(sz + 1);
        }
        if (len)
            memcpy (buf, str, len);
        buf[len] = '$';
        memcpy(buf+len+1, ksi_data->session_id, ses_len);
        buf[len+1+ses_len] = '$';
        memcpy (buf+len+1+ses_len+1, ptr, l);

        if (!ksi_lookup_sym (buf, len+l, 0))
            return ksi_str02sym (buf);
    }
}


ksi_keyword
ksi_lookup_key (const char* key, size_t len, int append)
{
  ksi_keyword sym;
  if (!append) {
    ksi_keyword sym = (ksi_keyword) alloca(sizeof(*sym) + len);
    sym->len = len;
    memcpy((char *) sym->ptr, key, len);
    ((char *)sym->ptr)[len] = 0;
    return (ksi_keyword) ksi_lookup_vtab (ksi_data->keytab, sym, 0);
  }

  sym = (ksi_keyword) ksi_malloc(sizeof(*sym) + len);
  sym->o.itag = KSI_TAG_KEYWORD;
  sym->len = len;
  memcpy((char *) sym->ptr, key, len);
  ((char *)sym->ptr)[len] = 0;
  return (ksi_keyword) ksi_lookup_vtab (ksi_data->keytab, sym, 1);
}

const char*
ksi_key2str (ksi_keyword key)
{
  int i, k, num, len;
  const char* nam;
  char* ptr;

  len = KSI_KEY_LEN (key);
  if (len <= 0)
    return "||:";

  nam = KSI_KEY_PTR (key);
  for (i = 0, num = 0; i < len; ++i) {
    switch (nam[i]) {
    case '\0':
    case '#':
    case ' ': case '(': case ')': case '[': case ']': case '{': case '}':
    case ';': case '`': case ',': case '\'': case '"':
    case '|': case '\\':
      num += 4;
      break;

    case '1': case '2': case '3': case '4': case '5':
    case '6': case '7': case '8': case '9': case '0':
      if (i == 0) {
        num += 4;
      }
      break;

    default:
      if (!isgraph (nam[i])) {
        num += 4;
      }
    }
  }

  ptr = (char*) ksi_malloc_data (len + num + 2);
  k = 0;
  for (i = 0; i < len; ++i) {
    char c = nam[i];
    switch (c) {
    case '\0':
    case '#':
    case ' ': case '(': case ')': case '[': case ']': case '{': case '}':
    case ';': case '`': case ',': case '\'': case '"':
    case '|': case '\\':
      goto hex;

    case '1': case '2': case '3': case '4': case '5':
    case '6': case '7': case '8': case '9': case '0':
      if (i == 0)
        goto hex;
      ptr[k++] = c;
      break;

    default:
      if (!isgraph (c)) {
      hex:
        ptr[k++] = '\\';
        ptr[k++] = 'x';
        ptr[k++] = tohex ((c >> 4) & 0xf);
        ptr[k++] = tohex (c & 0xf);
        ptr[k++] = ';';
      } else {
        ptr[k++] = c;
      }
    }
  }
  ptr[k++] = ':';
  ptr[k++] = '\0';

  return ptr;
}

ksi_obj
ksi_key_p (ksi_obj x)
{
    return KSI_KEY_P (x) ? ksi_true : ksi_false;
}

ksi_obj
ksi_string2keyword (ksi_obj str)
{
    KSI_CHECK (str, KSI_STR_P (str), "string->keyword: invalid string");

    if (KSI_C_STR_P (str))
        return ksi_str02key (KSI_STR_PTR (str));
    return ksi_str2key (KSI_STR_PTR (str), KSI_STR_LEN (str));
}

ksi_obj
ksi_keyword2string (ksi_obj key)
{
    KSI_CHECK(key, KSI_KEY_P(key), "keyword->string: invalid keyword");

    return ksi_str02string(KSI_KEY_PTR(key));
}

ksi_obj
ksi_symbol2keyword (ksi_obj sym)
{
    KSI_CHECK(sym, KSI_SYM_P(sym), "symbol->keyword: invalid symbol");

    return ksi_str02key(KSI_SYM_PTR(sym));
}

ksi_obj
ksi_keyword2symbol (ksi_obj key)
{
    KSI_CHECK(key, KSI_KEY_P(key), "keyword->symbol: invalid keyword");

    return ksi_str02sym(KSI_KEY_PTR(key));
}

ksi_obj
ksi_make_keyword (ksi_obj x)
{
    if (KSI_SYM_P(x))
        return ksi_symbol2keyword (x);
    if (KSI_STR_P(x))
        return ksi_string2keyword (x);
    if (KSI_KEY_P(x))
        return x;

    ksi_exn_error(ksi_assertion_s, x, "make-keyword: invalid object in arg1");
    return 0;
}

ksi_obj
ksi_object2string (ksi_obj x)
{
    const char* str = ksi_obj2str (x);
    return ksi_str02string (str);
}


struct abbrev_data_t
{
  int len;
  const char *str;
  ksi_obj res;
};

static int
abbrev_proc (void* val, void* abbrev_data)
{
    struct abbrev_data_t *data = (struct abbrev_data_t *) abbrev_data;
    if (data->len <= KSI_SYM_LEN(val)) {
        if (memcmp (data->str, KSI_SYM_PTR(val), data->len) == 0) {
            data->res = ksi_cons(val, data->res);
        }
    }
    return 0;
}

ksi_obj
ksi_abbrev (char* str, int len)
{
    if (str[0] == '#' && str[1] == '\\') {
        ksi_obj res = ksi_nil;
        int i;

        for (i = 0; ksi_char_names[i]; i++) {
            const char* name = ksi_char_names[i];
            if (memcmp (name, str, len) == 0)
                res = ksi_cons(ksi_str2char(name, strlen(name)), res);
        }
        return res;
    }

    if (str[0] == '#' && str[1] == ':') {
        struct abbrev_data_t data;
        data.len = len - 2;
        data.str = str + 2;
        data.res = ksi_nil;
        ksi_iterate_vtab (ksi_data->keytab, abbrev_proc, &data);
        return data.res;
    } else {
        struct abbrev_data_t data;
        data.len = len;
        data.str = str;
        data.res = ksi_nil;
        ksi_iterate_vtab (ksi_data->symtab, abbrev_proc, &data);
        return data.res;
    }
}

/* End of code */
