/*
 * ksi_str.c
 * strings
 *
 * Copyright (C) 1997-2010, ivan demakov
 *
 * The software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * The software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the software; see the file COPYING.LESSER.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 *
 * Author:        Ivan Demakov <ksion@users.sourceforge.net>
 * Creation date: Sat Jan 25 00:25:23 1997
 * Last Update:   Tue Mar  2 17:06:26 2010
 *
 */

#include "ksi_int.h"


static char* ksi_empty_str = "";

const char*
ksi_string2str (ksi_obj x)
{
    int len, i, k, num;
    const char* str;
    char* ptr;

    KSI_CHECK (x, KSI_STR_P (x), "string2str: invalid string in arg1");

    len = KSI_STR_LEN (x);
    str = KSI_STR_PTR (x);

    if (len <= 0)
        return "\"\"";

    for (i = 0, num = 0; i < len; ++i) {
        int c = str[i] & 0xff;
        switch (c) {
        case '\0':
            num += 3;
            break;
        case '\n': case '\r': case '\t': case '\f': case '\b': case '\a': case 0x1b:
            ++num;
            break;
        case '"': case '\\':
            ++num;
            break;
        default:
            if (!isprint (c)) {
                num += 3;
            }
        }
    }

    ptr = (char*) ksi_malloc_data (len + num + 4);
    ptr [k=0] = '"';
    for (i = 0; i < len; ++i) {
        int c = str[i] & 0xff;
        switch (c) {
        case '\0':
            ptr[++k] = '\\';
            ptr[++k] = '0';
            ptr[++k] = '0';
            c = '0';
            break;
        case '\n':
            ptr[++k] = '\\';
            c = 'n';
            break;
        case '\r':
            ptr[++k] = '\\';
            c = 'r';
            break;
        case '\t':
            ptr[++k] = '\\';
            c = 't';
            break;
        case '\f':
            ptr[++k] = '\\';
            c = 'f';
            break;
        case '\b':
            ptr[++k] = '\\';
            c = 'b';
            break;
        case '\a':
            ptr[++k] = '\\';
            c = 'a';
            break;
        case 0x1b:
            ptr[++k] = '\\';
            c = 'e';
            break;
        case '"': case '\\':
            ptr[++k] = '\\';
            break;
        default:
            if (!isprint (c)) {
                ptr[++k] = '\\';
                ptr[++k] = (char) ('0' + (c / (8 * 8)) % 8);
                ptr[++k] = (char) ('0' + (c / 8) % 8);
                ptr[++k] = (char) ('0' + (c % 8));
                continue;
            }
        }
        ptr[++k] = c;
    }

    ptr[++k] = '"';
    ptr[++k] = '\0';

    return ptr;
}

ksi_obj
ksi_str2string (const char* s, int l)
{
    ksi_string str;

    if (s == 0 || l <= 0) {
        str = (ksi_string) ksi_malloc(sizeof(*str));
        str -> o.itag = KSI_TAG_STRING;
        str -> ptr = ksi_empty_str;
        str -> len = 0;
    } else {
        str = (ksi_string) ksi_malloc(sizeof(*str));
        str -> o.itag = KSI_TAG_STRING;
        str -> ptr = ksi_malloc_data(l + 1);
        memcpy (str -> ptr, s, l);
        str -> ptr [l] = '\0';
        str -> len = l;
    }

    return (ksi_obj) str;
}

ksi_obj
ksi_str02string (const char* s)
{
    return ksi_str2string (s, s ? strlen (s) : 0);
}

ksi_obj
ksi_make_string (int len, char c)
{
    ksi_string str = (ksi_string) ksi_malloc(sizeof(*str));
    str -> o.itag = KSI_TAG_STRING;
    str -> ptr = (char*) ksi_malloc_data(len + 1);
    if (len)
        memset (str -> ptr, c, len);
    str -> ptr [len] = '\0';
    str -> len = len;

    return (ksi_obj) str;
}

ksi_obj
ksi_string_p (ksi_obj x)
{
    return KSI_STR_P (x) ? ksi_true : ksi_false;
}

ksi_obj
ksi_string_length (ksi_obj str)
{
    KSI_CHECK (str, KSI_STR_P (str), "string-length: invalid string");

    return ksi_long2num (KSI_STR_LEN (str));
}

ksi_obj
ksi_string_ref (ksi_obj str, ksi_obj k)
{
    unsigned ind;

    KSI_CHECK (str, KSI_STR_P (str), "string-ref: invalid string in arg1");
    KSI_CHECK (k, KSI_UINT_P (k), "string-ref: invalid integer in arg2");

    ind = ksi_num2ulong (k, "string-ref");

    KSI_CHECK (k, ind < KSI_STR_LEN (str), "string-ref: invalid index in arg2");

    return ksi_int2char ((unsigned char) KSI_STR_PTR (str) [ind]);
}

ksi_obj
ksi_string_set_x (ksi_obj str, ksi_obj k, ksi_obj ch)
{
    unsigned ind;

    KSI_CHECK (str, KSI_STR_P (str), "string-set!: invalid string in arg1");
    KSI_CHECK (str, !KSI_C_STR_P (str), "string-set!: constant string in arg1");
    KSI_CHECK (k, KSI_UINT_P (k), "string-set!: invalid integer in arg2");
    KSI_CHECK (ch, KSI_CHAR_P (ch), "string-set!: invalid char in arg3");

    ind = ksi_num2ulong (k, "string-set!");

    KSI_CHECK (k, ind < KSI_STR_LEN (str), "string-set!: index out of range in arg2");

    KSI_STR_PTR (str) [ind] = KSI_CHAR_CODE (ch);
    return ksi_void;
}

ksi_obj
ksi_substring (ksi_obj str, ksi_obj start, ksi_obj end)
{
    int s, e;

    KSI_CHECK (str, KSI_STR_P (str), "substring: invalid string in arg1");
    KSI_CHECK (start, KSI_EINT_P (start), "substring: invalid index in arg2");
    KSI_CHECK (end, !end || KSI_EINT_P (end), "substring: invalid index in arg3");

    s = ksi_num2long (start, "substring");
    e = (end ? ksi_num2long (end, "substring") : KSI_STR_LEN (str));

    KSI_CHECK (start, 0 <= s && s <= e, "substring: invalid index in arg2");
    KSI_CHECK (end, e <= KSI_STR_LEN (str), "substring: invalid index in arg3");

    return (ksi_obj) ksi_str2string (KSI_STR_PTR (str) + s, e - s);
}

ksi_obj
ksi_string_append (int argc, ksi_obj* args)
{
    ksi_obj str;
    int i, len = 0;
    char* ptr;

    for (i = 0; i < argc; i++) {
        if (KSI_CHAR_P (args[i])) {
            len += 1;
        } else {
            KSI_CHECK (args[i], KSI_STR_P (args[i]), "string-append: invalid string");
            len += KSI_STR_LEN (args[i]);
        }
    }

    str = (ksi_obj) ksi_make_string (len, '\0');
    ptr = KSI_STR_PTR (str);
    for (i = 0; i < argc; i++) {
        if (KSI_CHAR_P (args[i])) {
            *ptr++ = KSI_CHAR_CODE (args[i]);
        } else if (KSI_STR_LEN (args[i]) > 0) {
            memcpy (ptr, KSI_STR_PTR (args[i]), KSI_STR_LEN (args[i]));
            ptr += KSI_STR_LEN (args[i]);
        }
    }

    return str;
}

ksi_obj
ksi_string2list (ksi_obj str)
{
    int len;
    char* ptr;
    ksi_obj list;

    KSI_CHECK (str, KSI_STR_P (str), "string->list: invalid string");

    len = KSI_STR_LEN (str);
    ptr = KSI_STR_PTR (str);
    list = ksi_nil;
    while (len--)
        list = ksi_cons (ksi_int2char ((unsigned char) ptr [len]), list);

    return list;
}

ksi_obj
ksi_list2string (ksi_obj list)
{
    int i, len = ksi_list_len (list);
    ksi_obj str;
    char* ptr;

    KSI_CHECK (list, len >= 0, "list->string: invalid list in arg1");

    str = (ksi_obj) ksi_make_string (len, '\0');
    ptr = KSI_STR_PTR (str);
    for (i = 0; i < len; ++i, list = KSI_CDR (list)) {
        ksi_obj x = KSI_CAR (list);
        KSI_CHECK (x, KSI_CHAR_P (x), "list->string: invalid char");
        ptr [i] = KSI_CHAR_CODE (x);
    }

    return str;
}

ksi_obj
ksi_new_string (int argc, ksi_obj* argv)
{
    int i;
    ksi_obj str = ksi_make_string (argc, '\0');
    char* ptr = KSI_STR_PTR (str);

    for (i = 0; i < argc; ++i) {
        KSI_CHECK (argv[i], KSI_CHAR_P (argv[i]), "string: invalid char");
        ptr [i] = KSI_CHAR_CODE (argv[i]);
    }

    return str;
}

ksi_obj
ksi_string_copy (ksi_obj str)
{
    KSI_CHECK (str, KSI_STR_P (str), "string-copy: invalid string");
    return (ksi_obj) ksi_str2string (KSI_STR_PTR (str), KSI_STR_LEN (str));
}

ksi_obj
ksi_string_fill_x (ksi_obj str, ksi_obj c)
{
    int i;

    KSI_CHECK (str, KSI_STR_P (str), "string-fill!: invalid string in arg1");
    KSI_CHECK (str, !KSI_C_STR_P (str), "string-fill!: constant string in arg1");
    KSI_CHECK (c, KSI_CHAR_P (c), "string-fill!: invalid char in arg2");

    for (i = 0; i < KSI_STR_LEN (str); ++i)
        KSI_STR_PTR (str) [i] = KSI_CHAR_CODE (c);

    return ksi_void;
}

ksi_obj
ksi_string_eqv_p (ksi_obj s1, ksi_obj s2)
{
    int l1, l2;
    char *p1, *p2;

    KSI_CHECK (s1, KSI_STR_P (s1), "string=?: invalid string in arg1");
    KSI_CHECK (s2, KSI_STR_P (s2), "string=?: invalid string in arg2");

    l1 = KSI_STR_LEN (s1);
    l2 = KSI_STR_LEN (s2);
    p1 = KSI_STR_PTR (s1);
    p2 = KSI_STR_PTR (s2);

    if (l1 == l2 && p1 == p2)
        return ksi_true;

    return ksi_false;
}

ksi_obj
ksi_string_equal_p (ksi_obj s1, ksi_obj s2)
{
    int l1, l2;
    char *p1, *p2;

    KSI_CHECK (s1, KSI_STR_P (s1), "string=?: invalid string");
    KSI_CHECK (s2, KSI_STR_P (s2), "string=?: invalid string");

    l1 = KSI_STR_LEN (s1);
    l2 = KSI_STR_LEN (s2);
    p1 = KSI_STR_PTR (s1);
    p2 = KSI_STR_PTR (s2);

    if (l1 == l2) {
        if (p1 != p2)
            while (l1--)
                if (*p1++ != *p2++)
                    return ksi_false;

        return ksi_true;
    }

    return ksi_false;
}

ksi_obj
ksi_string_ci_equal_p (ksi_obj s1, ksi_obj s2)
{
    int l1, l2;
    unsigned char *p1, *p2;

    KSI_CHECK (s1, KSI_STR_P (s1), "string-ci=?: invalid string");
    KSI_CHECK (s2, KSI_STR_P (s2), "string-ci=?: invalid string");

    l1 = KSI_STR_LEN (s1);
    l2 = KSI_STR_LEN (s2);
    p1 = (unsigned char*) KSI_STR_PTR (s1);
    p2 = (unsigned char*) KSI_STR_PTR (s2);

    if (l1 == l2) {
        if (p1 != p2)
            while (l1--)
                if (tolower (*p1++) != tolower (*p2++))
                    return ksi_false;

        return ksi_true;
    }

    return ksi_false;
}

ksi_obj
ksi_string_eq_p (int argc, ksi_obj* args)
{
    int i;
    for (i = 1; i < argc; i++) {
        if (ksi_string_equal_p (args[0], args[i]) == ksi_false)
            return ksi_false;
    }

    return ksi_true;
}

ksi_obj
ksi_string_ci_eq_p (int argc, ksi_obj* args)
{
    int i;
    for (i = 1; i < argc; i++) {
        if (ksi_string_ci_equal_p (args[0], args[i]) == ksi_false)
            return ksi_false;
    }

    return ksi_true;
}


#ifdef HAVE_STRCOLL
static int
string_less_p (ksi_obj s1, ksi_obj s2, char* fname)
{
    int l1, l2;
    unsigned char *p1, *p2;
    char a1[2], a2[2];
    a1[1] = '\0';
    a2[1] = '\0';

    if (!KSI_STR_P (s1))
        ksi_exn_error (ksi_assertion_s, s1, "%s: invalid string", fname);
    if (!KSI_STR_P (s2))
        ksi_exn_error (ksi_assertion_s, s2, "%s: invalid string", fname);

    l1 = KSI_STR_LEN (s1);
    l2 = KSI_STR_LEN (s2);
    p1 = (unsigned char*) KSI_STR_PTR (s1);
    p2 = (unsigned char*) KSI_STR_PTR (s2);

    for (; l1 && l2; --l1, --l2) {
        int c;
        a1[0] = *p1++;
        a2[0] = *p2++;
        c = strcoll (a1, a2);
        if (c > 0) return 0;
        if (c < 0) return 1;
    }

    return (l1 < l2);
}

static int
string_ci_less_p (ksi_obj s1, ksi_obj s2, char* fname)
{
    int l1, l2;
    unsigned char *p1, *p2;
    char a1[2], a2[2];
    a1[1] = '\0';
    a2[1] = '\0';

    if (!KSI_STR_P (s1))
        ksi_exn_error (ksi_assertion_s, s1, "%s: invalid string", fname);
    if (!KSI_STR_P (s2))
        ksi_exn_error (ksi_assertion_s, s2, "%s: invalid string", fname);

    l1 = KSI_STR_LEN (s1);
    l2 = KSI_STR_LEN (s2);
    p1 = (unsigned char*) KSI_STR_PTR (s1);
    p2 = (unsigned char*) KSI_STR_PTR (s2);

    for (; l1 && l2; --l1, --l2) {
        int c;
        a1[0] = tolower (*p1); p1++;
        a2[0] = tolower (*p2); p2++;
        c = strcoll (a1, a2);
        if (c > 0) return 0;
        if (c < 0) return 1;
    }

    return (l1 < l2);
}

#else
static int
string_less_p (ksi_obj s1, ksi_obj s2, char* fname)
{
    int l1, l2;
    unsigned char *p1, *p2;
    if (!KSI_STR_P (s1))
        ksi_exn_error (ksi_assertion_s, s1, "%s: invalid string", fname);
    if (!KSI_STR_P (s2))
        ksi_exn_error (ksi_assertion_s, s2, "%s: invalid string", fname);

    l1 = KSI_STR_LEN (s1);
    l2 = KSI_STR_LEN (s2);
    p1 = (unsigned char*) KSI_STR_PTR (s1);
    p2 = (unsigned char*) KSI_STR_PTR (s2);
    for (; l1 && l2; --l1, --l2) {
        int c = *p1++ - *p2++;
        if (c > 0) return 0;
        if (c < 0) return 1;
    }
    return (l1 < l2);
}

static int
string_ci_less_p (ksi_obj s1, ksi_obj s2, char* fname)
{
    int l1, l2;
    unsigned char *p1, *p2;

    if (!KSI_STR_P (s1))
        ksi_exn_error (ksi_assertion_s, s1, "%s: invalid string", fname);
    if (!KSI_STR_P (s2))
        ksi_exn_error (ksi_assertion_s, s2, "%s: invalid string", fname);

    l1 = KSI_STR_LEN (s1);
    l2 = KSI_STR_LEN (s2);
    p1 = (unsigned char*) KSI_STR_PTR (s1);
    p2 = (unsigned char*) KSI_STR_PTR (s2);
    for (; l1 && l2; --l1, --l2) {
        int c = tolower (*p1) - tolower (*p2);
        p1++; p2++;
        if (c > 0) return 0;
        if (c < 0) return 1;
    }
    return (l1 < l2);
}
#endif

#define DEF_CMP(fun, cmp)			\
ksi_obj						\
fun (int ac, ksi_obj* av)			\
{						\
  int i;					\
  for (i = 1; i < ac; i++)			\
    {						\
      if (cmp)					\
        return ksi_false;			\
    }						\
  return ksi_true;				\
}

DEF_CMP (ksi_string_ls_p, !string_less_p (av[i-1], av[i], "string<?"))
DEF_CMP (ksi_string_gt_p, !string_less_p (av[i], av[i-1], "string>?"))
DEF_CMP (ksi_string_le_p, string_less_p (av[i], av[i-1], "string<=?"))
DEF_CMP (ksi_string_ge_p, string_less_p (av[i-1], av[i], "string>=?"))

DEF_CMP (ksi_string_ci_ls_p, !string_ci_less_p (av[i-1], av[i], "string<?"))
DEF_CMP (ksi_string_ci_gt_p, !string_ci_less_p (av[i], av[i-1], "string>?"))
DEF_CMP (ksi_string_ci_le_p, string_ci_less_p (av[i], av[i-1], "string<=?"))
DEF_CMP (ksi_string_ci_ge_p, string_ci_less_p (av[i-1], av[i], "string>=?"))


ksi_obj
ksi_scm_make_string (ksi_obj k, ksi_obj c)
{
    if (!c)
        c = ksi_int2char ('\0');

    KSI_CHECK (k, KSI_EINT_P (k), "make-string: invalid integer in arg1");
    KSI_CHECK (c, KSI_CHAR_P (c), "make-string: invalid char in arg2");

    return (ksi_obj) ksi_make_string (ksi_num2long (k, "make-string"), KSI_CHAR_CODE (c));
}


ksi_obj
ksi_string_index (ksi_obj str, ksi_obj chr, ksi_obj beg)
{
    const char* ptr;
    unsigned l, b;

    KSI_CHECK (str, KSI_STR_P (str), "string-index: invalid string in arg1");
    KSI_CHECK (chr, KSI_CHAR_P (chr), "string-index: invalid char in arg2");

    if (!beg) {
        b = 0;
    } else {
        KSI_CHECK (beg, KSI_EINT_P (beg), "string-index: invalid integer in arg3");
        b = ksi_num2ulong (beg, "string-index");
    }

    ptr = KSI_STR_PTR (str);
    l = KSI_STR_LEN (str);
    if (b < l) {
        char* where = (char*) memchr (ptr+b, KSI_CHAR_CODE (chr), l-b);
        if (where)
            return ksi_long2num (where - ptr);
    }

    return ksi_false;
}

ksi_obj
ksi_string_rindex (ksi_obj str, ksi_obj chr, ksi_obj beg)
{
    unsigned len, pos;

    KSI_CHECK (str, KSI_STR_P (str), "string-rindex: invalid string in arg1");
    KSI_CHECK (chr, KSI_CHAR_P (chr), "string-rindex: invalid char in arg2");

    len = KSI_STR_LEN (str);
    if (!beg) {
        pos = len;
    } else {
        KSI_CHECK (beg, KSI_EINT_P (beg), "string-rindex: invalid integer in arg3");
        pos = ksi_num2ulong (beg, "string-rindex");
        if (pos > len)
            pos = len;
    }

    while (--pos >= 0)
        if (KSI_STR_PTR (str) [pos] == KSI_CHAR_CODE (chr))
            return ksi_long2num (pos);

    return ksi_false;
}

ksi_obj
ksi_string_upcase_x (ksi_obj s)
{
    int i;

    KSI_CHECK (s, KSI_STR_P (s), "string-upcase!: invalid string in arg1");
    KSI_CHECK (s, !KSI_C_STR_P (s), "string-upcase!: constant string in arg1");

    for (i = 0; i < KSI_STR_LEN (s); i++)
        KSI_STR_PTR (s) [i] = toupper ((unsigned char) KSI_STR_PTR (s) [i]);
    return s;
}

ksi_obj
ksi_string_downcase_x (ksi_obj s)
{
    int i;

    KSI_CHECK (s, KSI_STR_P (s), "string-downcase!: invalid string in arg1");
    KSI_CHECK (s, !KSI_C_STR_P (s), "string-downcase!: constant string in arg1");

    for (i = 0; i < KSI_STR_LEN (s); i++)
        KSI_STR_PTR (s) [i] = tolower ((unsigned char) KSI_STR_PTR (s) [i]);
    return s;
}

ksi_obj
ksi_string_capitalize_x (ksi_obj str)
{
    char *ptr;
    int i, len, fl;

    KSI_CHECK(str, KSI_STR_P(str), "string-capitalize!: invalid string in arg1");
    KSI_CHECK(str, !KSI_C_STR_P(str), "string-capitalize!: constant string in arg1");

    ptr = KSI_STR_PTR(str);
    len = KSI_STR_LEN(str);
    fl = 0;
    for (i = 0; i < len; i++) {
        if (isalpha((unsigned char) ptr[i])) {
            if (fl) {
                ptr [i] = tolower((unsigned char) ptr [i]);
            } else {
                ptr [i] = toupper((unsigned char) ptr [i]);
                fl = 1;
            }
        } else {
            fl = 0;
        }
    }

    return str;
}

ksi_obj
ksi_string_for_each (ksi_obj proc, ksi_obj str, int ac, ksi_obj *av)
{
    int i, len, n;

    KSI_CHECK(proc, KSI_PROC_P(proc), "string-for-each: invalid procedure in arg1");
    KSI_CHECK(str, KSI_STR_P(str), "string-for-each: invalid string in arg2");

    len = KSI_STR_LEN(str);
    for (i = 0; i < ac; i++) {
        if (!KSI_STR_P(av[i]))
            ksi_exn_error(0, av[i], "string-for-each: invalid string in arg%d", i+3);
        if (len != KSI_STR_LEN(av[i]))
            ksi_exn_error(0, av[i], "string-for-each: invalid string length in arg%d", i+3);
    }
    if (ksi_procedure_has_arity_p(proc, ksi_long2num(ac+1), 0) == ksi_false)
        ksi_exn_error(0, proc, "string-for-each: invalid arity of the procedure in arg1");

    if (ac == 0) {
        char *ptr = KSI_STR_PTR(str);
        for (i = 0; i < len; i++) {
            ksi_obj ch = ksi_int2char(ptr[i]);
            KSI_CHECK_EVENTS;
            ksi_apply_1(proc, ch);
        }
    } else {
        ksi_obj *xs = (ksi_obj*) alloca((ac + 1) * sizeof *xs);
        for (i = 0; i < len; i++) {
            char *ptr = KSI_STR_PTR(str);
            KSI_CHECK_EVENTS;
            xs[0] = ksi_int2char(ptr[i]);
            for (n = 0; n < ac; n++) {
                ptr = KSI_STR_PTR(av[n]);
                xs[n+1] = ksi_int2char(ptr[i]);
            }
            ksi_apply_proc(proc, ac+1, xs);
        }
    }

    return ksi_void;
}

 /* End of code */
