/* -*- Mode: C; c-file-style: "gnu" -*-
   html.c -- spew specific to html.
   Created: Chris Toshok <toshok@hungry.com>, 5-Dec-1997.
 */
/*
  This file is part of Japhar, the GNU Virtual Machine for Java Bytecodes.
  Japhar is a project of The Hungry Programmers, GNU, and OryxSoft.

  Copyright (C) 1997, 1998 The Hungry Programmers

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "config.h"
#include "japharh.h"

#include <string.h>
#include <stdlib.h>

extern JNIEnv *the_env;

char *
html_escape_name(char *name)
{
  char new_name[200];
  int j = 0;
  int i;

  new_name[0] = '\0';

  for (i = 0; i < strlen(name); i ++)
    {
      if (name[i] == '<')
	{
	  new_name[j++] = '&';
	  new_name[j++] = 'l';
	  new_name[j++] = 't';
	  new_name[j++] = ';';
	}
      else if (name[i] == '>')
	{
	  new_name[j++] = '&';
	  new_name[j++] = 'g';
	  new_name[j++] = 't';
	  new_name[j++] = ';';
	}
      else if (name[i] == '/')
	new_name[j++] = '.';
      else
	new_name[j++] = name[i];
    }

  new_name[j] = 0;

  return strdup(new_name);
}

void
html_header(FILE *fp, char *classname)
{
  fprintf(fp, 
	  "<html>\n"
	  "<head>\n"
	  "  <title>%s</title>\n"
	  "</head>\n"
	  "<body bgcolor=#aaaaaa text=#000000>\n"
	  "<a name=\"TOP\">\n", classname);
}

void
html_footer(FILE *fp)
{
  fprintf(fp,
	  "</body>\n"
	  "</html>\n");
}

void
html_begin_access_flags(FILE *fp)
{
  fprintf(fp, "<font color=\"#0000ff\">");
}

void
html_end_access_flags(FILE *fp)
{
  fprintf(fp, "</font>");
}

void
html_begin_comment(FILE *fp)
{
  fprintf(fp, "<b><font color=\"#003300\">");
}

void
html_end_comment(FILE *fp)
{
  fprintf(fp, "</font></b>");
}

void
html_begin_method_name(FILE *fp)
{
  fprintf(fp, "<em>");
}

void
html_end_method_name(FILE *fp)
{
  fprintf(fp, "</em>");
}

void
html_begin_field_name(FILE *fp)
{
  fprintf(fp, "<em>");
}

void
html_end_field_name(FILE *fp)
{
  fprintf(fp, "</em>");
}

void
html_begin_type(FILE *fp)
{
  fprintf(fp, "<b>");
}

void
html_end_type(FILE *fp)
{
  fprintf(fp, "</b>");
}

void
html_newline(FILE *fp)
{
  fprintf(fp, "<br>\n");
}

void
html_begin_opcodes(FILE *fp)
{
  fprintf(fp, "<font size=-1>\n");
  fprintf(fp, "<pre>\n");
}

void
html_end_opcodes(FILE *fp)
{
  fprintf(fp, "</pre>\n");
  fprintf(fp, "</font>\n");
}

void
html_begin_methods(FILE *fp)
{
  fprintf(fp, "<dl><dd>\n");
}

void
html_end_methods(FILE *fp)
{
  fprintf(fp, "</dl>\n");
}

void
html_begin_fields(FILE *fp)
{
  fprintf(fp, "<dl><dd>\n");
}

void
html_end_fields(FILE *fp)
{
  fprintf(fp, "</dl>\n");
}

void
html_begin_link_to_class(FILE *fp,
			 ClazzFile *clazz)
{
  char *foo = html_escape_name(clazz->class_name);
  fprintf(fp, "<a href=\"%s.html#TOP\">", 
	  foo);
  free(foo);
}

void
html_begin_link_to_method(FILE *fp,
			  MethodStruct *method)
{
  Signature *sig = SIG_parseFromJavaSig(the_env, method->sig_str);
  char *foo = html_escape_name(method->clazz->class_name);
  char *mname = html_escape_name(method->name);

  fprintf(fp, "<a href=\"%s.html#METHOD_%s_%s\">", 
	  foo,
	  mname,
	  SIG_formatToJavaSig(the_env, sig));
  free(foo);
  free(mname);
  SIG_free(the_env, sig);
}

void
html_begin_link_to_field(FILE *fp,
			 FieldStruct *field)
{
  Signature *sig = SIG_parseFromJavaSig(the_env, field->sig_str);
  char *foo = html_escape_name(field->clazz->class_name);
  char *fname = html_escape_name(field->name);
  fprintf(fp, "<a href=\"%s.html#FIELD_%s_%s\">", 
	  foo,
	  fname,
	  SIG_formatToJavaSig(the_env, sig));
  free(foo);
  free(fname);
  SIG_free(the_env, sig);
}

void
html_end_link(FILE *fp)
{
  fprintf(fp, "</a>");
}

void
html_location_for_method(FILE *fp, MethodStruct *method)
{
  Signature *sig = SIG_parseFromJavaSig(the_env, method->sig_str);
  char *mname = html_escape_name(method->name);

  fprintf(fp, "<a name=\"METHOD_%s_%s\">",
	  mname,
	  SIG_formatToJavaSig(the_env, sig));

  free(mname);
  SIG_free(the_env, sig);
}

void
html_location_for_field(FILE *fp, FieldStruct *field)
{
  Signature *sig = SIG_parseFromJavaSig(the_env, field->sig_str);
  char *fname = html_escape_name(field->name);

  fprintf(fp, "<a name=\"FIELD_%s_%s\">",
	  fname,
	  SIG_formatToJavaSig(the_env, sig));

  free(fname);
  SIG_free(the_env, sig);
}

void
html_begin_exception_style(FILE *fp)
{
  fprintf(fp, "<b><font color=\"#990000\">");
}

void
html_end_exception_style(FILE *fp)
{
  fprintf(fp, "</font></b>");
}
