/*-
 * Copyright (c) 2003-2004 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipa_sdb_dump.c,v 1.9 2012/07/09 20:21:51 simon Exp $";
#endif /* !lint */

#include <sys/types.h>
#include <sys/stat.h>

#include <netinet/in.h>

#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <regex.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "ipa_sdb.h"

#define IPA_SDB_DUMP_NAME	"ipa_sdb_dump"

static const char *fname = NULL;	/* -f file */
static FILE	*src_fp = NULL;		/* FILE pointer. */
static int	src_fd;			/* File descriptor. */
unsigned int	lineno;			/* Current line number. */

static const char *envprogname;

static bool	ignore_date_err;	/* Set if no -e. */
static bool	use_stdin;		/* Set if read data from stdin. */

#define RULE_NREC_READ						\
	(sizeof(struct ipa_sdb_rule_record) < 1024 ?		\
	 (1024 / sizeof(struct ipa_sdb_rule_record)) : 1)

#define LIMIT_NREC_READ						\
	(sizeof(struct ipa_sdb_limit_record) < 1024 ?		\
	 (1024 / sizeof(struct ipa_sdb_limit_record)) : 1)

#ifdef WITH_LIMITS
# define LIMIT_START_STR		"Start        :"
# define LIMIT_RESTART_STR		"Restart      :"
# define LIMIT_RESTART_EXEC_STR		"Restart exec :"
# define LIMIT_REACH_STR		"Reach        :"
# define LIMIT_REACH_EXEC_STR		"Reach exec   :"
# define LIMIT_EXPIRE_STR		"Expire       :"
# define LIMIT_EXPIRE_EXEC_STR		"Expire exec  :"
# define LIMIT_UPDATED_STR		"Updated      :"
# define LIMIT_LIMIT_STR		"Limit        :"
# define LIMIT_COUNTER_STR		"Counter      :"
# define LIMIT_STR_LEN			(sizeof(LIMIT_COUNTER_STR) - 1)
#endif

#ifdef WITH_THRESHOLDS
# define THRESHOLD_STARTED_STR		"Started      :"
# define THRESHOLD_UPDATED_STR		"Updated      :"
# define THRESHOLD_THRESHOLD_STR	"Threshold    :"
# define THRESHOLD_COUNTER_STR		"Counter      :"
# define THRESHOLD_STR_LEN		(sizeof(THRESHOLD_COUNTER_STR) - 1)
#endif

#define PAT_COUNTER	"^[[:digit:]]{20}$"

static void	output(const char *, ...) ATTR_FORMAT(printf, 1, 2);
static void	logmsgx(const char *, ...) ATTR_FORMAT(printf, 1, 2);
static void	wrong_date(const char *, ...) ATTR_FORMAT(printf, 1, 2);
static void	exit_vmsg(int, int, const char *, va_list) ATTR_NORETURN;
static void	exit_err(const char *, ...) ATTR_NORETURN
		    ATTR_FORMAT(printf, 1, 2);
static void	exit_errx(const char *, ...) ATTR_NORETURN
		    ATTR_FORMAT(printf, 1, 2);
static void	exit_log(const char *, ...) ATTR_NORETURN
		    ATTR_FORMAT(printf, 1, 2);
static void	exit_logx(const char *, ...) ATTR_NORETURN
		    ATTR_FORMAT(printf, 1, 2);

#define cnt_to_base(cnt, c_high, c_low) do {		\
	c_high = htonl((uint32_t)(cnt >> 32));		\
	c_low  = htonl((uint32_t)cnt);			\
} while (/* CONSTCOND */ 0)

#define cnt_from_base(cnt, c_high, c_low) do {		\
	cnt = ntohl(c_high);				\
	cnt <<= 32;					\
	cnt += ntohl(c_low);				\
} while (/* CONSTCOND */ 0)

/*
 * Output version number and settings (-v and -h switches).
 */
static void
show_version(void)
{
	printf(IPA_SDB_DUMP_NAME ", version " PACKAGE_VERSION
	"\nRuntime settings: "
#ifdef IPA_SDB_NOT_PACKED
	"database structures are not packed."
#else
	"database structures are packed."
#endif
	"\nSupports: rules"
#ifdef WITH_LIMITS
	", limits"
#endif
#ifdef WITH_THRESHOLDS
	", thresholds"
#endif
	".\n");
}

/*
 * Output the help message (-h switch).
 */
static void
usage(void)
{
	show_version();
	printf("\
Usage: %s [options]\n\
 Convert ipa_sdb(5) database file to text representation and back\n\
 Options are:\n\
  -b\t\tConvert text representation of database file to original\n\
\t\tdatabase binary format and output it to stdout\n\
  -e\t\tIgnore errors in dates\n\
  -f file\tRead data from the given file instead of reading from stdin\n",
	    envprogname);
	printf(
#ifdef WITH_LIMITS
"  -l\t\tTreat the given data for a limit, by default data is\n\
\t\tconsidered as data for a rule\n"
#endif
#ifdef WITH_THRESHOLDS
"  -t\t\tTreat the given data for a threshold, by default data is\n\
\t\tconsidered as data for a rule\n"
#endif
"  -h\t\tOutput this help message and exit\n\
  -v\t\tOutput version number, configuration settings and exit\n");
}

static void
exit_vmsg(int in_file, int error, const char *format, va_list ap)
{
	fflush(stdout);
	fprintf(stderr, "%s: ", envprogname);
	if (in_file)
		fprintf(stderr, "file %s, line %u: ", fname, lineno);
	vfprintf(stderr, format, ap);
	if (error != 0)
		fprintf(stderr, ": %s", strerror(error));
	fprintf(stderr, "\n");
	exit(EXIT_FAILURE);
}

static void
exit_err(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	exit_vmsg(0, errno, format, ap);
	va_end(ap);
}

static void
exit_errx(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	exit_vmsg(0, 0, format, ap);
	va_end(ap);
}

static void
exit_log(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	exit_vmsg(1, errno, format, ap);
	va_end(ap);
}

static void
exit_logx(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	exit_vmsg(1, 0, format, ap);
	va_end(ap);
}

/*
 * A wrapper for read(2) syscall.
 */
static ssize_t
readn(int fd, void *vptr, size_t n)
{
	char *ptr;
	ssize_t	nread;
	size_t nleft;

	ptr = vptr;
	nleft = n;
	for (;;) {
		nread = read(fd, ptr, nleft);
		if (nread < 0) {
			if (errno == EINTR)
				nread = 0;
			else
				return (-1);
		} else if (nread == 0) {
			/* EOF */
			break;
		}
		nleft -= nread;
		if (nleft == 0)
			break;
		ptr += nread;
	}
	return (n - nleft);
}

/*
 * Output formatted string to stdout.
 */
static void
output(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	if (vprintf(format, ap) < 0)
		exit_err("output: vprintf failed");
	va_end(ap);
}

/*
 * Flush stdout.
 */
static void
output_flush(void)
{
	if (fflush(stdout) != 0)
		exit_err("output_flush: fflush(stdout)");
}

/*
 * Flush stdout and output message about error.
 */
static void
logmsgx(const char *format, ...)
{
	va_list ap;

	output_flush();
	fprintf(stderr, "ERROR: ");
	va_start(ap, format);
	vfprintf(stderr, format, ap);
	va_end(ap);
	fprintf(stderr, "\n");
}

/*
 * Output information about wrong date and exit if ignore_date_err is not set.
 */
static void
wrong_date(const char *format, ...)
{
	va_list ap;

	output_flush();
	fprintf(stderr, "ERROR: wrong date info: ");
	va_start(ap, format);
	vfprintf(stderr, format, ap);
	va_end(ap);
	fprintf(stderr, "\n");
	if (!ignore_date_err)
		exit(EXIT_FAILURE);
}

/*
 * Simplified version of regexec(3).
 */
static int
regexec_simple(const regex_t *re, const char *str)
{
	return (regexec(re, str, 0, (regmatch_t *)NULL, 0));
}

/*
 * Form an error message in buffer according to code using regerror()
 * function and return pointer to it.
 */
static const char *
regerrbuf(int code)
{
	static char buf[128];

	regerror(code, (regex_t *)NULL, buf, sizeof(buf));
	return (buf);
}

/*
 * Check whether date in binary rule's record is correct.
 */
static int
check_date_in_rule(const struct ipa_sdb_rule_record *p)
{
	if (p->mday > 31 || p->mday == 0 ||
	    ( (p->h1 > 23 || p->m1 > 59 || p->s1 > 59) &&
              !(p->h1 == 24 && p->m1 == 0 && p->s1 == 0) ) ||
	    ( (p->h2 > 23 || p->m2 > 59 || p->s2 > 59) &&
              !(p->h2 == 24 && p->m2 == 0 && p->s2 == 0) ) ||
	    (p->h2 * 60 * 60 + p->m2 * 60 + p->s2) <
	     (p->h1 * 60 * 60 + p->m1 * 60 + p->s1) ) {
		logmsgx("wrong date in the record");
		return (-1);
	}
	return (0);
}

static void
rule_rec_bin_to_txt(const struct ipa_sdb_rule_record *rec, unsigned int nrec)
{
	uint64_t cnt;

	cnt_from_base(cnt, rec->c_high, rec->c_low);
	output("%02u/%02u:%02u:%02u-%02u:%02u:%02u %020"PRIu64"\n",
	    rec->mday, rec->h1, rec->m1, rec->s1,
	    rec->h2, rec->m2, rec->s2, cnt);
	if (check_date_in_rule(rec) < 0)
		wrong_date("record %u", nrec);
}

static void
rule_bin_to_txt(void)
{
	struct stat statbuf;
	struct ipa_sdb_rule_record *rec, *recs;
	ssize_t nread;
	size_t recs_size;
	unsigned int nrec;

	if (use_stdin)
		recs_size = RULE_NREC_READ;
	else {
		if (fstat(src_fd, &statbuf) < 0)
			exit_err("fstat(%s)", fname);
		recs_size = (size_t)(1 + statbuf.st_size / sizeof(*recs));
		if (recs_size > RULE_NREC_READ)
			recs_size = RULE_NREC_READ;
	}
	recs_size *= sizeof(*recs);
	recs = malloc(recs_size);
	if (recs == NULL)
		exit_err("malloc(%lu)", (unsigned long)recs_size);

	nrec = 0;
	while ((nread = readn(src_fd, recs, recs_size)) > 0)
		for (rec = recs; nread > 0; ++nrec, ++rec) {
			nread -= sizeof(*rec);
			if (nread < 0)
				exit_errx("file %s: wrong size of file",
				    fname);
			rule_rec_bin_to_txt(rec, nrec);
		}
	if (nread < 0)
		exit_errx("file %s: could not read from file", fname);

	free(recs);
	output_flush();
}

static void
rule_rec_txt_to_bin(struct ipa_sdb_rule_record *rec, const char *buf)
{
	uint64_t cnt;
	unsigned int mday, h1, m1, s1, h2, m2, s2;

	if (sscanf(buf, "%02u/%02u:%02u:%02u-%02u:%02u:%02u %020"SCNu64"\n",
	    &mday, &h1, &m1, &s1, &h2, &m2, &s2, &cnt) != 8)
		exit_logx("sscanf failed");

	rec->mday = (uint8_t)mday;
	rec->h1 = (uint8_t)h1;
	rec->m1 = (uint8_t)m1;
	rec->s1 = (uint8_t)s1;
	rec->h2 = (uint8_t)h2;
	rec->m2 = (uint8_t)m2;
	rec->s2 = (uint8_t)s2;

	if (check_date_in_rule(rec) < 0)
		wrong_date("file %s, line %u", fname, lineno);

	cnt_to_base(cnt, rec->c_high, rec->c_low);
}

static void
rule_txt_to_bin(void)
{
#define BUF_SIZE							\
	sizeof("xx/xx:xx:xx-xx:xx:xx 18446744073709551615\n")
#define PAT_RULE_LINE							\
	"^[[:digit:]]{2}/[[:digit:]]{2}:[[:digit:]]{2}:[[:digit:]]{2}-"	\
	"[[:digit:]]{2}:[[:digit:]]{2}:[[:digit:]]{2} [[:digit:]]{20}\n$"

	struct ipa_sdb_rule_record rec;
	regex_t reg;
	int error;
	char buf[BUF_SIZE];

	error = regcomp(&reg, PAT_RULE_LINE, REG_EXTENDED|REG_NOSUB);
	if (error != 0)
		exit_errx("cannot compile regular expression \"%s\": %s",
		    PAT_RULE_LINE, regerrbuf(error));

	while (fgets(buf, sizeof(buf), src_fp) != NULL) {
		if (regexec_simple(&reg, buf) != 0)
			exit_logx("wrong format of line");
		rule_rec_txt_to_bin(&rec, buf);
		if (fwrite(&rec, sizeof(rec), 1, stdout) != 1)
			exit_err("file %s: fwrite", fname);
		++lineno;
	}

	if (ferror(src_fp) != 0)
		exit_log("fgets failed");

	regfree(&reg);
	output_flush();

#undef BUF_SIZE
#undef PAT_RULE_LINE
}

#ifdef WITH_ANY_LIMITS
/*
 * Return last month day in the given year/mon.
 */
static int
last_mday(uint16_t year, uint8_t mon)
{
	static int const last_mday_arr[] =
	    /*   Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec */
	    {  0, 31,  0, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

	if (mon == 2) {
		/* February */
		return ((year % 4 == 0 &&
		    (year % 100 != 0 || year % 400 == 0)) ? 29 : 28);
	}
	return (last_mday_arr[mon]);
}

static int
check_db_date(const ipa_sdb_date *p, int exist)
{
	if (exist)
		if (p->mon > 12 || p->mday > last_mday(p->year, p->mon) ||
		    p->hour > 23 || p->min > 59 || p->sec > 59) {
			logmsgx("wrong date in the record");
			return (-1);
		}
	return (0);
}

static void
output_db_date(const char *msg, const ipa_sdb_date *p, int exist)
{
	output("%s ", msg);
	if (exist)
		output("%u.%02u.%02u/%02u:%02u:%02u\n", ntohs(p->year),
		    p->mon, p->mday, p->hour, p->min, p->sec);
	else
		output("-\n");
}
#endif /* WITH_ANY_LIMITS */

#ifdef WITH_LIMITS
static void
limit_rec_bin_to_txt(const struct ipa_sdb_limit_record *rec, unsigned int nrec)
{
	uint64_t value;
	bool error;
	uint8_t exist;

	error = false;

	exist = rec->set & IPA_SDB_LIMIT_EVENT_START_SET;
	output_db_date(LIMIT_START_STR, &rec->start, exist);
	if (check_db_date(&rec->start, exist) < 0)
		error = true;

	exist = rec->set & IPA_SDB_LIMIT_EVENT_RESTART_SET;
	output_db_date(LIMIT_RESTART_STR, &rec->restart, exist);
	if (check_db_date(&rec->restart, exist) < 0)
		error = true;

	exist = rec->set & IPA_SDB_LIMIT_EVENT_RESTART_EXEC_SET;
	output_db_date(LIMIT_RESTART_EXEC_STR, &rec->restart_exec, exist);
	if (check_db_date(&rec->restart_exec, exist) < 0)
		error = true;

	exist = rec->set & IPA_SDB_LIMIT_EVENT_REACH_SET;
	output_db_date(LIMIT_REACH_STR, &rec->reach, exist);
	if (check_db_date(&rec->reach, exist) < 0)
		error = true;

	exist = rec->set & IPA_SDB_LIMIT_EVENT_REACH_EXEC_SET;
	output_db_date(LIMIT_REACH_EXEC_STR, &rec->reach_exec, exist);
	if (check_db_date(&rec->reach_exec, exist) < 0)
		error = true;

	exist = rec->set & IPA_SDB_LIMIT_EVENT_EXPIRE_SET;
	output_db_date(LIMIT_EXPIRE_STR, &rec->expire, exist);
	if (check_db_date(&rec->expire, exist) < 0)
		error = true;

	exist = rec->set & IPA_SDB_LIMIT_EVENT_EXPIRE_EXEC_SET;
	output_db_date(LIMIT_EXPIRE_EXEC_STR, &rec->expire_exec, exist);
	if (check_db_date(&rec->expire_exec, exist) < 0)
		error = true;

	exist = rec->set & IPA_SDB_LIMIT_EVENT_UPDATED_SET;
	output_db_date(LIMIT_UPDATED_STR, &rec->updated, exist);
	if (check_db_date(&rec->updated, exist) < 0)
		error = true;

	cnt_from_base(value, rec->l_high, rec->l_low);
	output("%s %020"PRIu64"\n", LIMIT_LIMIT_STR, value);

	cnt_from_base(value, rec->c_high, rec->c_low);
	output("%s %020"PRIu64"\n", LIMIT_COUNTER_STR, value);

	if (error)
		wrong_date("file %s, record %u", fname, nrec);

	output("\n");
}

static void
limit_bin_to_txt(void)
{
	struct stat statbuf;
	struct ipa_sdb_limit_record *rec, *recs;
	ssize_t	nread;
	size_t recs_size;
	unsigned int nrec;

	if (use_stdin)
		recs_size = LIMIT_NREC_READ;
	else {
		if (fstat(src_fd, &statbuf) < 0)
			exit_err("file %s: fstat", fname);
		recs_size = (size_t)(1 + statbuf.st_size / sizeof(*recs));
		if (recs_size > LIMIT_NREC_READ)
			recs_size = LIMIT_NREC_READ;
	}
	recs_size *= sizeof(*recs);
	recs = malloc(recs_size);
	if (recs == NULL)
		exit_err("malloc(%lu)", (unsigned long)recs_size);

	nrec = 0;
	while ((nread = readn(src_fd, recs, recs_size)) > 0)
		for (rec = recs; nread > 0; ++nrec, ++rec) {
			nread -= sizeof(*rec);
			if (nread < 0)
				exit_errx("file %s: wrong size of file",
				    fname);
			limit_rec_bin_to_txt(rec, nrec);
		}
	if (nread < 0)
		exit_err("file %s: read", fname);

	free(recs);
	output_flush();
}

struct limit_event {
	const char	*name;
	ipa_sdb_date	*db_date;
	uint8_t		set;
};

static void
limit_txt_to_bin(void)
{
	static struct ipa_sdb_limit_record rec;

#define LIMIT_EVENT(X, p) \
	{ LIMIT_ ## X ## _STR, p, IPA_SDB_LIMIT_EVENT_ ## X ## _SET }
	static struct limit_event limit_event_tbl[] = {
		LIMIT_EVENT(START,		&rec.start),
		LIMIT_EVENT(RESTART,		&rec.restart),
		LIMIT_EVENT(RESTART_EXEC,	&rec.restart_exec),
		LIMIT_EVENT(REACH,		&rec.reach),
		LIMIT_EVENT(REACH_EXEC,		&rec.reach_exec),
		LIMIT_EVENT(EXPIRE,		&rec.expire),
		LIMIT_EVENT(EXPIRE_EXEC,	&rec.expire_exec),
		LIMIT_EVENT(UPDATED,		&rec.updated),
		{ NULL, NULL, 0 }
	};
#undef LIMIT_EVENT

#define BUF_SIZE	128
#define PAT_DB_DATE							\
	"^([[:digit:]]{4,}\\.[[:digit:]]{2}\\.[[:digit:]]{2}/"		\
	"[[:digit:]]{2}:[[:digit:]]{2}:[[:digit:]]{2}|-)$"

	char buf[BUF_SIZE];
	regex_t	reg_db_date, reg_counter;
	uint64_t value;
	struct limit_event *event;
	ipa_sdb_date *db_date;
	char *ptr;
	size_t len;
	unsigned int year, mon, mday, hour, min, sec;
	int error;
	bool limit_str_flag, counter_str_flag;

	error = regcomp(&reg_db_date, PAT_DB_DATE, REG_EXTENDED|REG_NOSUB);
	if (error != 0)
		exit_errx("cannot compile regular expression \"%s\": %s",
		    PAT_DB_DATE, regerrbuf(error));

	error = regcomp(&reg_counter, PAT_COUNTER, REG_EXTENDED|REG_NOSUB);
	if (error != 0)
		exit_errx("cannot compile regular expression \"%s\": %s",
		    PAT_COUNTER, regerrbuf(error));

	limit_str_flag = counter_str_flag = false;
	event = limit_event_tbl;
	rec.set = 0;

	while (fgets(buf, sizeof(buf), src_fp) != NULL) {
		len = strlen(buf);
		if (buf[len - 1] != '\n')
			exit_logx("too long line or '\\n' is missed");
		if (buf[0] == '\n')
			continue;
		buf[len - 1] = '\0';
		if (len < LIMIT_STR_LEN + 1)
			exit_logx("too short line");

		if (event->name != NULL) {
			if (strncmp(buf, event->name, LIMIT_STR_LEN) != 0)
				exit_logx("line started with \"%s\" "
				    "is expected", event->name);
			ptr = buf + LIMIT_STR_LEN + 1;
			if (regexec_simple(&reg_db_date, ptr) != 0)
				goto wrong_format;
			db_date = event->db_date;
			if (*ptr != '-') {
				if (sscanf(ptr, "%u.%02u.%02u/%02u:%02u:%02u",
				    &year, &mon, &mday, &hour, &min, &sec) != 6)
					exit_logx("sscanf failed");
				if (year > UINT16_MAX)
					exit_logx("too big value for "
					    "year in date");
				db_date->year = htons((uint16_t)year);
				db_date->mon = (uint8_t)mon;
				db_date->mday = (uint8_t)mday;
				db_date->hour = (uint8_t)hour;
				db_date->min = (uint8_t)min;
				db_date->sec = (uint8_t)sec;
				rec.set |= event->set;
				if (check_db_date(db_date, 1) < 0)
					wrong_date("file %s, line %u",
					    fname, lineno);
			}
			++event;
		} else {
			ptr = buf + LIMIT_STR_LEN + 1;
			if (regexec_simple(&reg_counter, ptr) != 0)
				goto wrong_format;
			if (sscanf(ptr, "%020"SCNu64, &value) != 1)
				exit_logx("sscanf failed");
			if (!limit_str_flag) {
				if (strncmp(buf, LIMIT_LIMIT_STR,
				    LIMIT_STR_LEN) != 0)
					exit_logx("line started with \"%s\" "
					    "is expected", LIMIT_LIMIT_STR);
				limit_str_flag = true;
				counter_str_flag = false;
				cnt_to_base(value, rec.l_high, rec.l_low);
			} else {
				if (strncmp(buf, LIMIT_COUNTER_STR,
				    LIMIT_STR_LEN) != 0)
					exit_logx("line started with \"%s\" "
					    "is expected", LIMIT_COUNTER_STR);
				limit_str_flag = false;
				counter_str_flag = true;
				cnt_to_base(value, rec.c_high, rec.c_low);
				if (fwrite(&rec, sizeof(rec), 1, stdout) != 1)
					exit_err("fwrite(stdout)");
				rec.set = 0;
				event = limit_event_tbl;
			}
		}
		++lineno;
	}

	if (ferror(src_fp) != 0)
		exit_log("fgets failed");

	if (!counter_str_flag)
		exit_errx("file %s: not enough lines for last record", fname);

	output_flush();

	regfree(&reg_db_date);
	regfree(&reg_counter);

	return;

wrong_format:
	exit_logx("wrong format of line");

#undef BUF_SIZE
#undef PAT_DB_DATE
}
#endif /* WITH_LIMITS */

#ifdef WITH_THRESHOLDS
static void
threshold_rec_bin_to_txt(struct ipa_sdb_threshold_record *rec)
{
	uint64_t value;
	bool error;

	error = false;

	output_db_date(THRESHOLD_STARTED_STR, &rec->tm_started, 1);
	if (check_db_date(&rec->tm_started, 1) < 0)
		error = true;

	output_db_date(THRESHOLD_UPDATED_STR, &rec->tm_updated, 1);
	if (check_db_date(&rec->tm_updated, 1) < 0)
		error = true;

	cnt_from_base(value, rec->t_high, rec->t_low);
	output("%s %020"PRIu64"\n", THRESHOLD_THRESHOLD_STR, value);

	cnt_from_base(value, rec->c_high, rec->c_low);
	output("%s %020"PRIu64"\n", THRESHOLD_COUNTER_STR, value);

	if (error)
		wrong_date("file %s", fname);
}

static void
threshold_bin_to_txt(void)
{
	struct ipa_sdb_threshold_record rec;
	ssize_t	nread;

	nread = readn(src_fd, &rec, sizeof(rec));
	if (nread > 0) {
		if (nread < sizeof(rec))
			exit_errx("file %s: wrong size of file", fname);
		threshold_rec_bin_to_txt(&rec);
	}
	if (nread < 0)
		exit_err("file %s: read", fname);

	/* Try to read one character. */
	switch (readn(src_fd, &rec, 1)) {
	case 0:
		/* EOF */
		break;
	case 1:
		exit_errx("file %s: wrong size of file", fname);
		/* NOTREACHED */
	default: /* -1 */
		exit_err("file %s: read", fname);
	}
}

struct threshold_event {
	const char	*name;
	ipa_sdb_date	*db_date;
};

static void
threshold_txt_to_bin(void)
{
	static struct ipa_sdb_threshold_record rec;

#define THRESHOLD_EVENT(X, p) { THRESHOLD_ ## X ## _STR, p }
	static struct threshold_event threshold_event_tbl[] = {
		THRESHOLD_EVENT(STARTED,	&rec.tm_started),
		THRESHOLD_EVENT(UPDATED,	&rec.tm_updated),
		{ NULL, NULL }
	};
#undef THRESHOLD_EVENT

#define BUF_SIZE	128
#define PAT_DB_DATE							\
	"^[[:digit:]]{4,}\\.[[:digit:]]{2}\\.[[:digit:]]{2}/"		\
	"[[:digit:]]{2}:[[:digit:]]{2}:[[:digit:]]{2}$"

	char buf[BUF_SIZE];
	regex_t	reg_db_date, reg_counter;
	uint64_t value;
	struct threshold_event *event;
	ipa_sdb_date *db_date;
	char *ptr;
	size_t len;
	unsigned int  year, mon, mday, hour, min, sec;
	int error;
	bool threshold_str_flag, counter_str_flag;

	error = regcomp(&reg_db_date, PAT_DB_DATE, REG_EXTENDED|REG_NOSUB);
	if (error != 0)
		exit_errx("cannot compile regular expression \"%s\": %s",
		    PAT_DB_DATE, regerrbuf(error));

	error = regcomp(&reg_counter, PAT_COUNTER, REG_EXTENDED|REG_NOSUB);
	if (error != 0)
		exit_errx("cannot compile regular expression \"%s\": %s",
		    PAT_COUNTER, regerrbuf(error));

	threshold_str_flag = counter_str_flag = false;
	event = threshold_event_tbl;

	while (fgets(buf, sizeof(buf), src_fp) != NULL) {
		len = strlen(buf);
		if (buf[len - 1] != '\n')
			exit_logx("too long line or '\\n' is missed");
		if (buf[0] == '\n')
			continue;
		buf[len - 1] = '\0';
		if (len < THRESHOLD_STR_LEN + 1)
			exit_logx("too short line");

		if (event->name != NULL) {
			if (strncmp(buf, event->name, THRESHOLD_STR_LEN) != 0)
				exit_logx("line started with \"%s\" "
				    "is expected", event->name);
			ptr = buf + THRESHOLD_STR_LEN + 1;
			if (regexec_simple(&reg_db_date, ptr) != 0)
				goto wrong_format;
			db_date = event->db_date;
			if (sscanf(ptr, "%u.%02u.%02u/%02u:%02u:%02u",
			    &year, &mon, &mday, &hour, &min, &sec) != 6)
				exit_logx("sscanf failed");
			if (year > UINT16_MAX)
				exit_logx("too big value for year "
				    "in date");
			db_date->year = htons((uint16_t)year);
			db_date->mon = (uint8_t)mon;
			db_date->mday = (uint8_t)mday;
			db_date->hour = (uint8_t)hour;
			db_date->min = (uint8_t)min;
			db_date->sec = (uint8_t)sec;
			if (check_db_date(db_date, 1) < 0)
				wrong_date("file %s, line %u", fname, lineno);
			++event;
		} else {
			ptr = buf + THRESHOLD_STR_LEN + 1;
			if (regexec_simple(&reg_counter, ptr) != 0)
				goto wrong_format;
			if (sscanf(ptr, "%020"SCNu64, &value) != 1)
				exit_logx("sscanf failed");
			if (threshold_str_flag == 0) {
				if (strncmp(buf, THRESHOLD_THRESHOLD_STR,
				    THRESHOLD_STR_LEN) != 0)
					exit_logx("line started with \"%s\" "
					    "is expected",
					    THRESHOLD_THRESHOLD_STR);
				threshold_str_flag = true;
				cnt_to_base(value, rec.t_high, rec.t_low);
			} else if (!counter_str_flag) {
				if (strncmp(buf, THRESHOLD_COUNTER_STR,
				    THRESHOLD_STR_LEN) != 0)
					exit_logx("line started with \"%s\" "
					    "is expected",
					    THRESHOLD_COUNTER_STR);
				counter_str_flag = true;
				cnt_to_base(value, rec.c_high, rec.c_low);
				if (fwrite(&rec, sizeof(rec), 1, stdout) != 1)
					exit_err("fwrite(stdout)");
			} else
				exit_logx("unexpected line");
		}
		++lineno;
	}

	if (ferror(src_fp) != 0)
		exit_log("fgets failed");

	if (!counter_str_flag)
		exit_errx("file %s: not enough lines", fname);

	output_flush();

	regfree(&reg_db_date);
	regfree(&reg_counter);

	return;

wrong_format:
	exit_logx("wrong format of line");

#undef BUF_SIZE
#undef PAT_DB_DATE
}
#endif /* WITH_THRESHOLDS */

#ifdef WITH_LIMITS
# define OPTSTRING_LIMIT "l"
#else
# define OPTSTRING_LIMIT ""
#endif

#ifdef WITH_THRESHOLDS
# define OPTSTRING_THRESHOLD "t"
#else
# define OPTSTRING_THRESHOLD ""
#endif

#define OPTSTRING ":bef:h" OPTSTRING_LIMIT OPTSTRING_THRESHOLD "v"

int
main(int argc, char *argv[])
{
	int opt;
	bool b_flag = false;	/* -b */
#ifdef WITH_LIMITS
	bool l_flag = false;	/* -l */
#endif
#ifdef WITH_THRESHOLDS
	bool t_flag = false;	/* -t */
#endif

	/* Save the program name. */
	envprogname = strrchr(argv[0], '/');
	if (envprogname != NULL)
		++envprogname;
	else
		envprogname = argv[0];

	opterr = 0;
	ignore_date_err = false;
	while ((opt = getopt(argc, argv, OPTSTRING)) != -1)
		switch (opt) {
		case ':':
			exit_errx("option requires an argument -- %c", optopt);
			/* NOTREACHED */
		case '?':
			exit_errx("illegal option -- %c", optopt);
			/* NOTREACHED */
		case 'b':
			b_flag = true;
			break;
		case 'e':
			ignore_date_err = true;
			break;
		case 'f':
			fname = optarg;
			break;
		case 'h':
			usage();
			return (EXIT_SUCCESS);
#ifdef WITH_LIMITS
		case 'l':
			l_flag = true;
			break;
#endif
#ifdef WITH_THRESHOLDS
		case 't':
			t_flag = true;
			break;
#endif
		case 'v':
			show_version();
			return (EXIT_SUCCESS);
		default:
			exit_errx("unexpected option -- %c", optopt);
		}

	if (optind < argc)
		exit_errx("non-switch argument \"%s\"", argv[optind]);

	if (fname == NULL) {
		use_stdin = true;
		fname = "<stdin>";
		if (b_flag)
			src_fp = stdin;
		else
			src_fd = STDIN_FILENO;
	} else {
		use_stdin = false;
		if (b_flag) {
			src_fp = fopen(fname, "r");
			if (src_fp == NULL)
				exit_err("fopen(%s, \"r\")", fname);
		} else {
			src_fd = open(fname, O_RDONLY);
			if (src_fd < 0)
				exit_err("open(%s, O_RDONLY)", fname);
		}
	}

	lineno = 1;

#if defined(WITH_LIMITS) && defined(WITH_THRESHOLDS)
	if (l_flag && t_flag)
		exit_errx("-l and -t flags cannot be used together");
#endif

#ifdef WITH_LIMITS
	if (l_flag) {
		/* -l */
		if (b_flag)
			limit_txt_to_bin();
		else
			limit_bin_to_txt();
	} else
#endif
#ifdef WITH_THRESHOLDS
	if (t_flag) {
		/* -t */
		if (b_flag)
			threshold_txt_to_bin();
		else
			threshold_bin_to_txt();
	} else
#endif
	{
		if (b_flag)
			rule_txt_to_bin();
		else
			rule_bin_to_txt();
	}

	if (!use_stdin) {
		if (b_flag) {
			if (fclose(src_fp) != 0)
				exit_err("file %s: fclose", fname);
		} else {
			if (close(src_fd) < 0)
				exit_err("file %s: close", fname);
		}
	}

	return (EXIT_SUCCESS);
}
