/*-
 * Copyright (c) 2004 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipastat_conf.c,v 1.3.2.1 2011/11/15 18:12:29 simon Exp $";
#endif /* !lint */

#include <sys/types.h>
#include <sys/stat.h>

#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fnmatch.h>
#include <regex.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"
#include "parser.h"
#include "pathnames.h"

#include "ipastat_conf.h"
#include "ipastat_log.h"
#include "ipastat_rules.h"
#include "ipastat_st.h"
#include "ipastat_main.h"

char		*ipastat_conf_file = IPASTAT_CONF_FILE;	/* -f conf_file */

char		mimic_real_config = 0;	/* Set if -tt. */

/*
 * Give limitation on the depth of included files.
 */
#define INCLUDE_DEPTH_MAX 100
static unsigned int include_depth;

static unsigned int section;		/* Current section ID. */

static char	global_section_set;	/* Set if has global{}. */

static unsigned int conf_event_no;	/* Current section ordinal number. */
static const void *conf_event_arg;	/* Current argument for conf_event(). */

static const unsigned char conf_event_begin[] = {
	0,					/* Does not exist.	*/
	0,					/* Root section.	*/
	IPA_CONF_EVENT_GLOBAL_BEGIN,		/* global{}		*/
	IPA_CONF_EVENT_RULE_BEGIN,		/* rule{}		*/
	IPA_CONF_EVENT_LIMIT_BEGIN,		/* limit{}		*/
	IPA_CONF_EVENT_THRESHOLD_BEGIN,		/* threshold{}		*/
	0,					/* Not my section.	*/
	IPA_CONF_EVENT_RULEPAT_BEGIN		/* rulepat{}		*/
};

static regex_t	reg_list;

static struct rule *currule;		/* Current rule. */

static struct rulepat *currulepat;	/* Current rulepat. */
static unsigned int rulepatno = 0;

static signed char posix_re_pattern;	/* posix_re_pattern parameter. */

#ifdef WITH_LIMITS
static struct limit *curlimit;		/* Current limit. */
static unsigned int limitno;		/* Current limit ordinal number. */
static struct limits_list *limits_list;
#endif

#ifdef WITH_THRESHOLDS
static struct threshold *curthreshold;	/* Current threshold. */
static unsigned int thresholdno;	/* Current threshold ordinal number. */
static struct thresholds_list *thresholds_list;
#endif

static const struct {
	const char	*str;
	int		val;
} value_units_tbl[] = {
	{ "bytes",	IPA_CONF_TYPE_BYTES	},
	{ "time",	IPA_CONF_TYPE_TIME	},
	{ "number",	IPA_CONF_TYPE_UINT64	},
	{ "any",	0			}
};

#define VALUE_UNITS_TBL_SIZE \
	(sizeof(value_units_tbl) / sizeof(value_units_tbl[0]))

static char	use_log;		/* 0, if -t or -x switch was used. */

static void	confvlogmsgx(int, const char *, const char *, va_list)
		    ATTR_FORMAT(printf, 3, 0);

/*
 * Exported support functions for modules.
 */
static const ipa_suppfunc suppfunc = {
	print_string,		/* print_string		*/
	print_bytes,		/* print_bytes		*/
	print_time,		/* print_time		*/
	print_value,		/* print_value		*/
	print_boolean,		/* print_boolean	*/
	print_space,		/* print_space		*/
	mod_print_param_name,	/* print_param_name	*/
	mod_print_args,		/* print_param_args	*/
	mod_print_param_end,	/* print_param_end	*/
	mod_print_sect_name,	/* print_sect_name	*/
	mod_print_args,		/* print_sect_args	*/
	print_sect_begin,	/* print_sect_begin	*/
	mod_print_sect_end,	/* print_sect_end	*/
	open_close_log,		/* open_log		*/
	open_close_log,		/* close_log		*/
	mod_logmsg,		/* logmsg		*/
	mod_logconferr,		/* logconferr		*/
	parser_local_sym_add,	/* local_sym_add	*/
	parser_local_sym_del,	/* local_sym_del	*/
	parser_global_sym_add,	/* global_sym_add	*/
	parser_global_sym_del	/* global_sym_del	*/
};

/*
 * Log message prepending it with the prefix.
 * If use_log is set, then log is used, else printf(3) is used.
 * Do not use mem_*() functions in this function.
 */
static void
confvlogmsgx(int priority, const char *prefix, const char *format, va_list ap)
{
	if (use_log) {
		char buf[LOG_BUF_SIZE];
		char *msg;
		int rv;

		rv = vsnprintf(buf, sizeof(buf), format, ap);
		if (rv < 0) {
			logmsg(IPA_LOG_ERR, "confvlogmsgx: vsnprintf failed");
			goto log_unformated;
		}
		if (rv < sizeof(buf)) {
			logmsgx(priority, "%s: %s", prefix, buf);
			return;
		}
		msg = malloc(++rv);
		if (msg == NULL) {
			logmsgx(IPA_LOG_ERR, "confvlogmsgx: malloc failed");
			goto log_unformated;
		}
		if (vsnprintf(msg, rv, format, ap) < 0) {
			logmsg(IPA_LOG_ERR, "confvlogmsgx: vsnprintf failed");
			free(msg);
			goto log_unformated;
		}
		logmsgx(priority, "%s: %s", prefix, msg);
		free(msg);
		return;
log_unformated:
		logmsgx(priority, "%s: unformated message: %s",
		    prefix, format);
	} else {
		fflush(stdout);
		fprintf(stderr, "%s: ", prefix);
		vfprintf(stderr, format, ap);
		fprintf(stderr, "\n");
	}
}

/*
 * The wrapper for parser_vlogmsgx.
 */
static void
parser_vlogmsgx_wrapper(const char *format, va_list ap)
{
	confvlogmsgx(IPA_LOG_ERR, "parsing error", format, ap);
}

/*
 * Wrappers for logging about configuration errors.
 */
static void
vlogconfe(const char *format, va_list ap)
{
	confvlogmsgx(IPA_LOG_ERR, "configuration error", format, ap);
}

void
logconfe(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	vlogconfe(format, ap);
	va_end(ap);
}

/*
 * The same as above one, but with priority argument and va_list.
 */
static void
vlogconfx_priority(int priority, const char *format, va_list ap)
{
	confvlogmsgx(priority, "configuration error", format, ap);
}

/*
 * Register a configuration event in st_mod module.
 */
static int
st_mod_conf_event(const struct st_mod *st_mod, unsigned int event,
    unsigned int no, const void *arg)
{
	if (st_mod->ipa_st_mod->conf_event(event, no, arg) < 0) {
		logconfx("module %s: conf_event(IPA_CONF_EVENT_%s) failed",
		    st_mod->mod_file, conf_event_msg[event]);
		return (-1);
	}
	return (0);
}

/*
 * Register a configuration event in each module.
 */
static int
mod_conf_event(unsigned int event, unsigned int no, const void *arg)
{
	const struct st_mod *st_mod;

	SLIST_FOREACH(st_mod, &st_mod_list, link)
		if (st_mod_conf_event(st_mod, event, no, arg) < 0)
			return (-1);
	return (0);
}

/*
 * Parse the "global" section.
 */
/* ARGSUSED */
static int
parse_global(void *arg ATTR_UNUSED)
{
	if (global_section_set) {
		logconfx("this section is duplicated");
		return (-1);
	}
	global_section_set = 1;
	return (0);
}

/*
 * Parse the "rule" section.
 */
static int
parse_rule(void *arg)
{
	struct rule *rule;
	char *name;

	name = *(char **)arg;
	if (conf_validate_name(name) < 0)
		return (-1);
	if (rule_by_name(name) != NULL) {
		logconfx("this section is duplicated");
		return (-1);
	}

	name = mem_strdup(name, m_anon);
	if (name == NULL) {
		logconfx("mem_strdup failed");
		return (-1);
	}
	rule = alloc_rule(name);
	if (rule == NULL) {
		logconfx("alloc_rule failed");
		return (-1);
	}
	rule->free_mask = RULE_FREE_NAME;

#ifdef WITH_LIMITS
	limits_list = &rule->limits;
	STAILQ_INIT(limits_list);
	limitno = 0;
#endif

#ifdef WITH_THRESHOLDS
	thresholds_list = &rule->thresholds;
	STAILQ_INIT(thresholds_list);
	thresholdno = 0;
#endif

	if (parser_local_sym_add("rule", rule->name, 0) < 0)
		return (-1);

	currule = rule;
	conf_event_no = rule->no;
	conf_event_arg = rule->name;

	return (0);
}

/*
 * Parse the "dynamic_rules" parameter.
 */
static int
parse_dynamic_rules(void *arg)
{
	dynamic_rules = (signed char)*(int *)arg;
	return (0);
}

/*
 * Parse the "rulepat" section.
 */
static int
parse_rulepat(void *arg)
{
	struct rulepat *rulepat;
	char *pat;
	int error;

	pat = *(char **)arg;
	STAILQ_FOREACH(rulepat, &rulepats_list, link)
		if (strcmp(rulepat->pat, pat) == 0) {
			logconfx("this section is duplicated");
			return (-1);
		}

	rulepat = mzone_alloc(rulepat_mzone);
	if (rulepat == NULL) {
		logconfx("mzone_alloc failed");
		return (-1);
	}
	STAILQ_INSERT_TAIL(&rulepats_list, rulepat, link);

	error = regcomp(&rulepat->re, pat, REG_EXTENDED|REG_NOSUB);
	if (error != 0) {
		logconfx("regcomp(\"%s\"): %s", pat, regerrbuf(error));
		return (-1);
	}

	rulepat->pat = pat;
	rulepat->no = rulepatno++;
	rulepat->check_next = -1;
	rulepat->st_list = NULL;

#ifdef WITH_LIMITS
	limits_list = &rulepat->limits;
	STAILQ_INIT(limits_list);
	limitno = 0;
#endif

#ifdef WITH_THRESHOLDS
	thresholds_list = &rulepat->thresholds;
	STAILQ_INIT(thresholds_list);
	thresholdno = 0;
#endif

	currulepat = rulepat;
	conf_event_no = rulepat->no;
	conf_event_arg = rulepat->pat;

	return (0);
}

/*
 * Parse the "check_next_rulepat" parameter.
 */
static int
parse_check_next_rulepat(void *arg)
{
	currulepat->check_next = *(int *)arg;
	return (0);
}

/*
 * Parse the "value_units" parameter.
 */
static int
parse_value_units(void *arg)
{
	unsigned int i;

	if (value_units >= 0) {
		logconfx("cannot re-define this parameter");
		return (-1);
	}
	if (got_arg_value) {
		logconfx("this parameter must be used before other parameters "
		    "and sections, that accept the IPA_CONF_TYPE_VALUE data "
		    "type");
		return (-1);
	}
	for (i = 0; i < VALUE_UNITS_TBL_SIZE; ++i)
		if (strcmp(*(char **)arg, value_units_tbl[i].str) == 0) {
			value_units = value_units_tbl[i].val;
			return (0);
		}
	logconfx("wrong value");
	return (-1);
}

#ifdef WITH_LIMITS
/*
 * Parse the "limit" section.
 */
static int
parse_limit(void *arg)
{
	const char *name;
	struct limit *limit;

	name = *(char **)arg;
	if (conf_validate_name(name) < 0)
		return (-1);
	STAILQ_FOREACH(limit, limits_list, link)
		if (strcmp(limit->name, name) == 0) {
			logconfx("this section is duplicated");
			return (-1);
		}
	limit = alloc_limit();
	if (limit == NULL) {
		logconfx("alloc_limit failed");
		return (-1);
	}
	limit->name = mem_strdup(name, m_anon);
	if (limit->name == NULL) {
		logconfx("mem_strdup failed");
		return (-1);
	}

	STAILQ_INSERT_TAIL(limits_list, limit, link);
	limit->no = limitno++;

	limit->free_mask = LIMIT_FREE_NAME;

	curlimit = limit;
	conf_event_no = limit->no;
	conf_event_arg = limit->name;

	return (0);
}

/*
 * Parse the "dynamic_limits" parameter.
 */
static int
parse_dynamic_limits(void *arg)
{
	dynamic_limits = (signed char)*(int *)arg;
	return (0);
}
#endif /* WITH_LIMITS */

#ifdef WITH_THRESHOLDS
/*
 * Parse the "threshold" section.
 */
static int
parse_threshold(void *arg)
{
	const char *name;
	struct threshold *threshold;

	name = *(char **)arg;
	if (conf_validate_name(name) < 0)
		return (-1);
	STAILQ_FOREACH(threshold, thresholds_list, link)
		if (strcmp(threshold->name, name) == 0) {
			logconfx("this section is duplicated");
			return (-1);
		}
	threshold = alloc_threshold();
	if (threshold == NULL) {
		logconfx("alloc_threshold failed");
		return (-1);
	}
	threshold->name = mem_strdup(name, m_anon);
	if (threshold->name == NULL) {
		logconfx("mem_strdup failed");
		return (-1);
	}

	STAILQ_INSERT_TAIL(thresholds_list, threshold, link);
	threshold->no = thresholdno++;

	threshold->free_mask = THRESHOLD_FREE_NAME;

	curthreshold = threshold;
	conf_event_no = threshold->no;
	conf_event_arg = threshold->name;

	return (0);
}

/*
 * Parse the "dynamic_thresholds" parameter.
 */
static int
parse_dynamic_thresholds(void *arg)
{
	dynamic_thresholds = (signed char)*(int *)arg;
	return (0);
}
#endif /* WITH_THRESHOLDS */

/*
 * Check security of configuration file: absolute path, regular file.
 */
static int
check_conf_file(const char *fname, int ignore_non_regular)
{
	struct stat statbuf;

	if (stat(fname, &statbuf) < 0) {
		logconfe("stat(%s): %s", fname, strerror(errno));
		return (-1);
	}
	if (!S_ISREG(statbuf.st_mode)) {
		if (ignore_non_regular)
			return (0);
		logconfe("configuration file \"%s\" should be a regular file",
		    fname);
		return (-1);
	}
	return (0);
}

/*
 * Parse the "st_mod" parameter.
 */
static int
parse_st_mod(void *arg)
{
	const struct st_mod *st_mod2;
	struct ipa_st_mod *ipa_st_mod;
	struct st_mod *st_mod;
	char *mod_name, *sym;

	st_mod = mem_malloc(sizeof(*st_mod), m_anon);
	if (st_mod == NULL) {
		logconfx("mem_malloc failed");
		return (-1);
	}
	st_mod->mod_file = *(char **)arg;
	st_mod->mod_handle = dl_open(st_mod->mod_file);
	if (st_mod->mod_handle == NULL) {
		logconfx("dl_open(%s): %s", st_mod->mod_file, dl_error());
		return (-1);
	}
	mod_name = get_mod_name(st_mod->mod_file);
	if (mod_name == NULL)
		return (-1);
	if (mem_asprintf(m_anon, &sym, "%s_st_mod", mod_name) < 0) {
		logconfx("mem_asprintf failed");
		return (-1);
	}
	st_mod->ipa_st_mod =
	    (struct ipa_st_mod *)dl_lookup_sym(st_mod->mod_handle, sym);
	if (st_mod->ipa_st_mod == NULL) {
		logconfx("given module is not an IPA statistics module "
		    "or unknown symbol naming scheme is used");
		return (-1);
	}
	mem_free(sym, m_anon);
	mem_free(mod_name, m_anon);
	ipa_st_mod = st_mod->ipa_st_mod;

	/* Check ipa_st_mod API version. */
	if (ipa_st_mod->api_ver != IPA_ST_MOD_API_VERSION) {
		logconfx("module %s uses ipa_st_mod API version %u, my "
		    "ipa_st_mod API version is %u", st_mod->mod_file,
		    ipa_st_mod->api_ver, IPA_DB_MOD_API_VERSION);
		return (-1);
	}

	/* Check if module is thread-safe or vice versa. */
#ifdef WITH_PTHREAD
	if (!(ipa_st_mod->mod_flags & IPA_MOD_FLAG_PTHREAD_SAFE)) {
		logconfx("module %s must be thread-safe", st_mod->mod_file);
		return (-1);
	}
#else
	if (ipa_st_mod->mod_flags & IPA_MOD_FLAG_PTHREAD_SAFE) {
		logconfx("module %s must not be thread-safe", st_mod->mod_file);
		return (-1);
	}
#endif /* WITH_PTHREAD */

	if (strcmp(ipa_st_mod->st_name, "null") == 0) {
		logconfx("module's statistics name is \"null\", "
		    "this is a name of the built-in statistics");
		return (-1);
	}

	st_mod2 = st_mod_by_name(ipa_st_mod->st_name);
	if (st_mod2 != NULL) {
		logconfx("duplicated statistics name \"%s\" in %s and %s"
		    "modules", ipa_st_mod->st_name, st_mod->mod_file,
		    st_mod2->mod_file);
		return (-1);
	}
	if (ipa_st_mod->conf_prefix != NULL) {
		st_mod2 = st_mod_by_prefix(ipa_st_mod->conf_prefix);
		if (st_mod2 != NULL) {
			logconfx("duplicated configuration prefix \"%s\" in "
			    "%s and %s modules", ipa_st_mod->conf_prefix,
			    st_mod->mod_file, st_mod2->mod_file);
			return (-1);
		}
	}

	ipa_st_mod->suppfunc = &suppfunc;
	ipa_st_mod->memfunc = &memfunc;

	if (init_conf_tbls(st_mod->mod_file, 1,
	    ipa_st_mod->conf_sect_tbl, &st_mod->conf_sect_hash,
	    ipa_st_mod->conf_param_tbl, &st_mod->conf_param_hash) < 0)
		return (-1);

	if (ipa_st_mod->conf_init() < 0) {
		logconfx("module %s: conf_init failed", st_mod->mod_file);
		return (-1);
	}

	SLIST_INSERT_HEAD(&st_mod_list, st_mod, link);
	return (0);
}

/*
 * Parse the "st_list" parameter.
 */
static int
parse_st_list(void *arg)
{
	const struct st_list **list_ptr;
	const struct st_mod *st_mod;
	const char *st_name;
	struct st_elem *st;
	struct st_set *set;
	struct st_list *list;
	char *ptr;

	switch (section) {
#ifdef WITH_RULES
	case IPA_CONF_SECT_RULE:
		list_ptr = &currule->st_list;
		break;
#endif
	case IPA_CONF_SECT_RULEPAT:
		list_ptr = &currulepat->st_list;
		break;
#ifdef WITH_LIMITS
	case IPA_CONF_SECT_LIMIT:
		list_ptr = &curlimit->st_list;
		break;
#endif
#ifdef WITH_THRESHOLDS
	case IPA_CONF_SECT_THRESHOLD:
		list_ptr = &curthreshold->st_list;
		break;
#endif
	default: /* IPA_CONF_SECT_GLOBAL */
		list_ptr = &global_st_list;
	}

	if (*list_ptr != NULL) {
		logconfx("cannot re-define this parameter");
		return (-1);
	}

	set = NULL;
	list = NULL;

	for (ptr = *(char **)arg; ptr != NULL;) {
		/* Get the name of the next statistics system. */
		st_name = ptr;
		ptr = strchr(ptr, ' ');
		if (ptr != NULL)
			*ptr++ = '\0';

		/* Handle "null" statistics system. */
		if (strcmp(st_name, "null") == 0) {
			if (list != NULL || ptr != NULL) {
				logconfx("built-in statistics system \"null\" "
				    "cannot be used together with another "
				    "statistics systems");
				return (-1);
			}
			*list_ptr = &st_list_null;
			return (0);
		}

		st_mod = st_mod_by_name(st_name);
		if (st_mod == NULL) {
			logconfx("cannot find module with \"%s\" statistics "
			    "system name", st_name);
			return (-1);
		}

		if (set != NULL) {
			/* We already have set for current st_list parameter. */
			STAILQ_FOREACH(st, list, link)
				if (strcmp(st_name,
				    st->ipa_st_mod->st_name) == 0) {
					logconfx("duplicated statistics "
					    "system \"%s\"", st_name);
					return (-1);
				}
		} else {
			/* Create new set for st_list parameter. */
			set = mem_malloc(sizeof(*set), m_anon);
			if (set == NULL) {
				logconfx("mem_malloc failed");
				return (-1);
			}
			list = &set->list;
			STAILQ_INIT(list);
		}

		/* Add new st element to st_list. */
		st = mem_malloc(sizeof(*st), m_anon);
		if (st == NULL) {
			logconfx("mem_malloc failed");
			return (-1);
		}
		st->ipa_st_mod = st_mod->ipa_st_mod;
		st->mod_file = st_mod->mod_file;

		STAILQ_INSERT_TAIL(list, st, link);
	}

	/* New st_list --> add it to st_sets. */
	*list_ptr = list;
	SLIST_INSERT_HEAD(&st_sets, set, link);
	return (0);
}

/*
 * Parse the "posix_re_pattern" parameter.
 */
static int
parse_posix_re_pattern(void *arg)
{
	posix_re_pattern = (signed char)*(int *)arg;
	return (0);
}

/*
 * Parse the "debug_st_null" parameter.
 */
static int
parse_debug_st_null(void *arg)
{
	uint32_t level;

	level = *(uint32_t *)arg;
	if (level > 1) {
		logconfx("too big debug level, max level is 1");
		return (-1);
	}
	debug_st_null = (signed char)level;
	return (0);
}

/*
 * Check file path: non empty string and should start with '/'.
 */
static int
check_file_path(const char *fname, const char *what)
{
	if (*fname == '\0') {
		logconfx("argument should be a non-empty string");
		return (-1);
	}
	if (*fname != '/') {
		logconfx("%s should be given with absolute path", what);
		return (-1);
	}
	return (0);
}

static int
check_include_depth(void)
{
	if (include_depth > INCLUDE_DEPTH_MAX) {
		logconfx("too big (> %u) depth of included files",
		    INCLUDE_DEPTH_MAX);
		return (-1);
	}
	return (0);
}

/*
 * Parse the "include" parameter.
 */
static int
parse_include(void *arg)
{
	struct parser_pb *pb;
	char *fname;

	/* Save offset of current configuration file and close it. */
	fname = *(char **)arg;
	pb = parser_top_pb();
	pb->foff = ftell(pb->fp);
	if (pb->foff < 0) {
		logconf("ftell(%s)", pb->fname);
		return (-1);
	}
	if (fclose(pb->fp) != 0) {
		logconf("fclose(%s)", pb->fname);
		return (-1);
	}

	/* Validate an argument. */
	if (check_file_path(fname, "file") < 0)
		return (-1);

	++include_depth;
	if (check_include_depth() < 0)
		return (-1);

	/* Check security of configuration file. */
	if (check_conf_file(fname, 0) < 0)
		return (-1);

	/* Open included configuration file and put it to stack. */
	pb = parser_new_pb(0);
	if (pb == NULL)
		return (-1);
	pb->fp = fopen(fname, "r");
	if (pb->fp == NULL) {
		logconf("fopen(%s, \"r\")", fname);
		return (-1);
	}
	pb->fname = fname;
	return (parser_push_pb(pb));
}

/*
 * Parse the "include_files" parameter.
 */
static int
parse_include_files(void *arg)
{
	struct stat statbuf;
	regex_t re;
	const struct dirent *dp;
	struct parser_pb *pb, *tpb;
	char *dir, *pat, *fname;
	DIR *dirp;

	/* Validate an argument. */
	dir = *(char **)arg;
	if (check_file_path(dir, "directory") < 0)
		return (-1);

	pat = strrchr(dir, '/');
	*pat++ = '\0';
	if (posix_re_pattern > 0) {
		int error;

		error = regcomp(&re, pat, REG_EXTENDED|REG_NOSUB);
		if (error != 0) {
			logconfx("cannot compile regular expression: "
			    "regcomp(\"%s\"): %s", pat, regerrbuf(error));
			return (-1);
		}
	}

	/* Check security of the given directory. */
	if (lstat(dir, &statbuf) < 0) {
		logconf("lstat(%s)", dir);
		return (-1);
	}
	if (!S_ISDIR(statbuf.st_mode)) {
		logconfx("given pathname is not a directory");
		return (-1);
	}

	/* Read directory and push pb to stack. */
	dirp = opendir(dir);
	if (dirp == NULL) {
		logconf("opendir(%s)", dir);
		return (-1);
	}
	pb = NULL;
	tpb = parser_top_pb();
	for (;;) {
		errno = 0;
		dp = readdir(dirp);
		if (dp == NULL) {
			if (errno != 0) {
				logconf("readdir(%s)", dir);
				return (-1);
			}
			break;
		}
		/* Ignore "." amd ".." directories. */
		if (dp->d_name[0] == '.')
			switch (dp->d_name[1]) {
			case '\0':
				continue;
			case '.':
				if (dp->d_name[2] == '\0')
					continue;
			}
		if (posix_re_pattern != 1) {
			if (fnmatch(pat, dp->d_name, FNM_PERIOD) == FNM_NOMATCH)
				continue;
		} else {
			if (regexec_simple(&re, dp->d_name) != 0)
				continue;
		}
		if (mem_asprintf(m_parser, &fname, "%s/%s", dir,
		    dp->d_name) < 0) {
			logconfx("mem_asprintf failed");
			return (-1);
		}
		switch (check_conf_file(fname, 1)) {
		case -1:
			return (-1);
		case 0:
			mem_free(fname, m_parser);
			continue;
		}
		pb = parser_new_pb(0);
		if (pb == NULL)
			return (-1);
		pb->fname = fname;
		pb->foff = 0;
		if (parser_push_pb(pb) < 0)
			return (-1);
	}
	if (posix_re_pattern > 0)
		regfree(&re);
	if (closedir(dirp) < 0) {
		logconf("closedir(%s)", dir);
		return (-1);
	}
	mem_free(dir, m_parser);
	if (pb != NULL) {
		tpb->foff = ftell(tpb->fp);
		if (tpb->foff < 0) {
			logconf("ftell(%s)", tpb->fname);
			return (-1);
		}
		if (fclose(tpb->fp) != 0) {
			logconf("fclose(%s)", tpb->fname);
			return (-1);
		}
		++include_depth;
		if (check_include_depth() < 0)
			return (-1);
		pb->fp = fopen(pb->fname, "r");
		if (pb->fp == NULL) {
			logconf("fopen(%s, \"r\")", pb->fname);
			return (-1);
		}
	}
	return (0);
}

/*
 * Log information about chain of included files.
 */
static void
log_include_history(void)
{
	const struct parser_pb *pb;
	const char *fname;

	pb = parser_top_pb();
	if (pb != NULL) {
		fname = pb->fname;
		while ((pb = parser_top_pb()) != NULL) {
			if (fname != pb->fname) {
				fname = pb->fname;
				logconfe("included from %s:%u",
				    fname, pb->lineno);
			}
			(void)parser_pop_pb();
		}
	}
}

static void
set_global_params(void)
{
	global_section_set = 1;
	if (global_st_list == NULL)
		global_st_list = &st_list_null;
	if (debug_st_null < 0)
		debug_st_null = 0;
	if (dynamic_rules < 0)
		dynamic_rules = 0;
#ifdef WITH_LIMITS
	if (dynamic_limits < 0)
		dynamic_limits = 0;
#endif
#ifdef WITH_THRESHOLDS
	if (dynamic_thresholds < 0)
		dynamic_thresholds = 0;
#endif
	if (value_units < 0)
		value_units = 0;
}

static const unsigned int sect_root[] = { IPA_CONF_SECT_ROOT, 0 };
static const unsigned int sect_for_st_list[] = { IPA_CONF_SECT_GLOBAL,
	IPA_CONF_SECT_RULE,
#ifdef WITH_LIMITS
	IPA_CONF_SECT_LIMIT,
#endif
#ifdef WITH_THRESHOLDS
	IPA_CONF_SECT_THRESHOLD,
#endif
	IPA_CONF_SECT_RULEPAT, 0
};
static const unsigned int sect_rulepat[] = { IPA_CONF_SECT_RULEPAT, 0 };
#ifdef WITH_ANY_LIMITS
static const unsigned int sect_for_any_limit[] = { IPA_CONF_SECT_RULE,
	IPA_CONF_SECT_RULEPAT, 0
};
#endif

/*
 * Sections in ipastat.conf.
 */
static ipa_conf_sect conf_sect_tbl[] = {
	{ "global", IPA_CONF_SECT_GLOBAL, 0, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_root, parse_global
	},
	{ "rule", IPA_CONF_SECT_RULE, 1, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_root, parse_rule
	},
	{ "rulepat", IPA_CONF_SECT_RULEPAT, 1, NULL, NULL,
	  IPA_CONF_TYPE_STRING, sect_root, parse_rulepat
	},
#ifdef WITH_LIMITS
	{ "limit", IPA_CONF_SECT_LIMIT, 1, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_for_any_limit, parse_limit
	},
#endif
#ifdef WITH_THRESHOLDS
	{ "threshold", IPA_CONF_SECT_THRESHOLD, 1, NULL, NULL,
	  IPA_CONF_TYPE_MISC, sect_for_any_limit, parse_threshold
	},
#endif
	{ NULL, 0, 0, NULL, NULL,
	  IPA_CONF_TYPE_MISC, NULL, NULL
	}
};

#define PAT_LIST "^[^ \"]+( [^ \"]+)*$"

/*
 * Parameters in ipastat.conf.
 */
static ipa_conf_param conf_param_tbl[] = {
	{ "st_mod", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  sect_root, parse_st_mod
	},
	{ "include", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  NULL, parse_include
	},
	{ "include_files", 1, NULL, NULL, IPA_CONF_TYPE_STRING,
	  NULL, parse_include_files
	},
	{ "posix_re_pattern", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_root, parse_posix_re_pattern
	},
	{ "st_list", -1, PAT_LIST, &reg_list, IPA_CONF_TYPE_MISC,
	  sect_for_st_list, parse_st_list
	},
	{ "check_next_rulepat", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_rulepat, parse_check_next_rulepat
	},
	{ "debug_st_null", 1, NULL, NULL, IPA_CONF_TYPE_UINT32,
	  sect_root, parse_debug_st_null
	},
	{ "dynamic_rules", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_root, parse_dynamic_rules
	},
	{ "value_units", 1, NULL, NULL, IPA_CONF_TYPE_MISC,
	  sect_root, parse_value_units
	},
#ifdef WITH_LIMITS
	{ "dynamic_limits", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_root, parse_dynamic_limits
	},
#endif
#ifdef WITH_THRESHOLDS
	{ "dynamic_thresholds", 1, NULL, NULL, IPA_CONF_TYPE_BOOLEAN,
	  sect_root, parse_dynamic_thresholds
	},
#endif
	{ NULL, 0, NULL, NULL, IPA_CONF_TYPE_MISC,
	  NULL, NULL
	}
};

/* Module which own current custom section. */
static const struct st_mod *cur_st_mod;

/* Hash tables for ipastat.conf configuration. */
static struct conf_sect_hash *conf_sect_hash;
static struct conf_param_hash *conf_param_hash;

/* Pointers to configuration tables. */
static const ipa_conf_sect *conf_sect_tbl_ptr;
static const ipa_conf_param *conf_param_tbl_ptr;
static const struct conf_sect_hash *conf_sect_hash_ptr;
static const struct conf_param_hash *conf_param_hash_ptr;

/* Set if in own ipastat.conf section. */
static char	in_own_sect;

/* Stack of nested configuration sections. */
struct sect_stack {
	SLIST_ENTRY(sect_stack) link;
	unsigned int	section;
};

SLIST_HEAD(EMPTY, sect_stack) sect_stack_list =
	SLIST_HEAD_INITIALIZER(sect_stack_list);

static int
init_config_data(void)
{
	if ((m_result = mem_type_new_local(MTYPE_NAME(result),
	    "Memory for query results", 0)) == NULL ||
	    (m_parser = mem_type_new_local(MTYPE_NAME(parser),
	    "Memory of parser", MEMTYPE_FLAGS)) == NULL)
		return (-1);

	value_units = -1;
	got_arg_value = 0;

	global_st_list = NULL;
	debug_st_null = -1;
	SLIST_INIT(&st_mod_list);

	conf_sect_he_mzone = mzone_init(MZONE_NAME(conf_sect_he),
	    "Config sections hash entries", 0, sizeof(struct conf_sect_he),
	    CONF_SECT_HE_NSIZE, CONF_SECT_HE_NALLOC);
	if (conf_sect_he_mzone == NULL)
		return (-1);

	conf_param_he_mzone = mzone_init(MZONE_NAME(conf_param_he),
	    "Config parameters hash entries", 0, sizeof(struct conf_param_he),
	    CONF_PARAM_HE_NSIZE, CONF_PARAM_HE_NALLOC);
	if (conf_param_he_mzone == NULL)
		return (-1);

	rules_hash_init();

	rule_mzone = mzone_init(MZONE_NAME(rule), "Rules", 0,
	    sizeof(struct rule), RULE_NSIZE, RULE_NALLOC);
	if (rule_mzone == NULL)
		return (-1);
	dynamic_rules = -1;

	rulepat_mzone = mzone_init(MZONE_NAME(rulepat), "Rules patterns", 0,
	    sizeof(struct rulepat), RULEPAT_NSIZE, RULEPAT_NALLOC);
	if (rulepat_mzone == NULL)
		return (-1);

#ifdef WITH_LIMITS
	limit_mzone = mzone_init(MZONE_NAME(limit), "Limits", 0,
	    sizeof(struct limit), LIMIT_NSIZE, LIMIT_NALLOC);
	if (limit_mzone == NULL)
		return (-1);
	dynamic_limits = -1;
#endif

#ifdef WITH_THRESHOLDS
	threshold_mzone = mzone_init(MZONE_NAME(threshold), "Thresholds", 0,
	    sizeof(struct threshold), THRESHOLD_NSIZE, THRESHOLD_NALLOC);
	if (threshold_mzone == NULL)
		return (-1);
	dynamic_thresholds = -1;
#endif

	posix_re_pattern = -1;

	if (build_conf_re() < 0)
		return (-1);
	if (init_conf_tbls((char *)NULL, 1, conf_sect_tbl, &conf_sect_hash,
	    conf_param_tbl, &conf_param_hash) < 0)
		return (-1);

	return (0);
}

/*
 * Deinitialize not needed configuration data structures.
 */
static void
deinit_config_data(void)
{
	deinit_conf_tbls(0, conf_sect_tbl, conf_sect_hash,
	    conf_param_tbl, conf_param_hash);

	mzone_deinit(conf_sect_he_mzone);
	mzone_deinit(conf_param_he_mzone);

	if (mzone_is_empty(rulepat_mzone)) {
		mzone_deinit(rulepat_mzone);
		rulepat_mzone = NULL;
	}
}

static int
token_section_begin(void)
{
	const unsigned int *id;
	const char *prefix;
	const ipa_conf_sect *conf_sect;
	const struct get_arg *get_arg;
	struct sect_stack *sect_stack;
	struct st_mod *st_mod;
	char *ptr;
	int nargs;

	/* Put previous section on top of sections stack. */
	sect_stack = mem_malloc(sizeof(*sect_stack), m_anon);
	if (sect_stack == NULL) {
		logconfe("token_section_begin: mem_malloc failed");
		return (-1);
	}
	sect_stack->section = section;
	SLIST_INSERT_HEAD(&sect_stack_list, sect_stack, link);

	/* Get name of current section: "section" or "prefix:section". */
	cursect = parser_token;
	if (in_own_sect && (ptr = strchr(cursect, ':')) != NULL) {
		/*
		 * Previous section is not a custom section and
		 * new custom section appears.
		 */
		*ptr = '\0';
		prefix = cursect;
		cursect = ptr + 1;
		st_mod = st_mod_by_prefix(prefix);
		if (st_mod != NULL) {
			conf_sect_hash_ptr = st_mod->conf_sect_hash;
			conf_param_hash_ptr = st_mod->conf_param_hash;
			conf_sect_tbl_ptr = st_mod->ipa_st_mod->conf_sect_tbl;
			conf_param_tbl_ptr = st_mod->ipa_st_mod->conf_param_tbl;
			curmodfile = st_mod->mod_file;
			cur_st_mod = st_mod;
		} else {
			logconfx("cannot find module with \"%s\" "
			    "configuration prefix", prefix);
			return (-1);
		}
		in_own_sect = 0;
	}

	/* Find section. */
	conf_sect = find_conf_sect(conf_sect_tbl_ptr, conf_sect_hash_ptr,
	    cursect);
	if (conf_sect == NULL) {
		logconfx("unknown section");
		return (-1);
	}

	/* Validate type of argument. */
	if (conf_sect->arg_type > IPA_CONF_TYPE_MISC) {
		logconfx("internal error: unknown type %u of function's "
		    "argument", conf_sect->arg_type);
		return (-1);
	}

	/* Check whether current section is used in correct place. */
	id = conf_sect->sect_where;
	if (id != NULL)
		for (;; ++id) {
			if (*id == section)
				break;
			if (*id == 0) {
				logconfx("this section is not expected here");
				return (-1);
			}
		}

	/* Check number of section's arguments. */
	nargs = conf_sect->arg_nargs;
	if (nargs >= 0) {
		if (parser_nargs != nargs) {
			logconfx("wrong number of arguments (has %d, "
			    "should have %d)", parser_nargs, nargs);
			return (-1);
		}
	} else {
		nargs = -nargs;
		if (parser_nargs < nargs) {
			logconfx("this section should have at "
			    "least %d argument%s", nargs,
			    plural_form((unsigned int)nargs));
			return (-1);
		}
	}

	/* Validate arguments if needed. */
	if (conf_sect->arg_regexp != NULL)
		if (regexec_simple(conf_sect->arg_regexp, parser_args) != 0) {
			logconfx("wrong format of an argument");
			return (-1);
		}

	section = conf_sect->sect_id;
	if (!in_own_sect) {
		/* Register configuration event in module for its section. */
		if (st_mod_conf_event(cur_st_mod,
		    IPA_CONF_EVENT_CUSTOM_SECT_BEGIN, section,
		    (void *)NULL) < 0)
			return (-1);
	}

	/* Parse it. */
	conf_event_no = 0;
	conf_event_arg = NULL;
	get_arg = &get_arg_tbl[conf_sect->arg_type];
	if (get_arg->reg != NULL)
		if (regexec_simple(get_arg->reg, parser_args) != 0) {
			logconfx("wrong format of an argument");
			return (-1);
		}
	if (get_arg->parse(get_arg->argp) < 0)
		return (-1);
	if (conf_sect->arg_parse != NULL)
		if (conf_sect->arg_parse(get_arg->argp) < 0)
			return (-1);

	if (in_own_sect) {
		/*
		 * Register configuration event in all modules for
		 * ipastat.conf's section.
		 */
		if (mod_conf_event(conf_event_begin[section], conf_event_no,
		    conf_event_arg) < 0)
			return (-1);
	}

	return (0);
}

static int
token_parameter(void)
{
	const unsigned int *id;
	const char *prefix;
	const ipa_conf_param *conf_param;
	const struct get_arg *get_arg;
	const struct st_mod *st_mod;
	char *ptr;
	int nargs;

	/* Get name of current parameter: "parameter" or "prefix:parameter". */
	curparam = parser_token;
	if (in_own_sect && (ptr = strchr(curparam, ':')) != NULL) {
		/*
		 * Current section is not a custom section
		 * and custom parameter appears.
		 */
		*ptr = '\0';
		prefix = curparam;
		curparam = ptr + 1;
		st_mod = st_mod_by_prefix(prefix);
		if (st_mod != NULL) {
			conf_param_hash_ptr = st_mod->conf_param_hash;
			conf_param_tbl_ptr = st_mod->ipa_st_mod->conf_param_tbl;
			curmodfile = st_mod->mod_file;
		} else {
			logconfx("cannot find module with \"%s\" "
			    "configuration prefix", prefix);
			return (-1);
		}
	}

	/* Find parameter. */
	conf_param = find_conf_param(conf_param_tbl_ptr, conf_param_hash_ptr,
	    curparam);
	if (conf_param == NULL) {
		logconfx("unknown parameter");
		return (-1);
	}

	/* Validate type of argument. */
	if (conf_param->arg_type > IPA_CONF_TYPE_MISC) {
		logconfx("internal error: unknown type %u of an argument",
		    conf_param->arg_type);
		return (-1);
	}

	/* Check whether parameter is used in correct place. */
	id = conf_param->param_where;
	if (id != NULL)
		for (;; ++id) {
			if (*id == section)
				break;
			if (*id == 0) {
				logconfx("this parameter is not expected here");
				return (-1);
			}
		}

	/* Check number of parameter's arguments. */
	nargs = conf_param->arg_nargs;
	if (nargs >= 0) {
		if (parser_nargs != nargs) {
			logconfx("wrong number of arguments (has %d, "
			    "should have %d)", parser_nargs, nargs);
			return (-1);
		}
	} else {
		nargs = -nargs;
		if (parser_nargs < nargs) {
			logconfx("this parameter should have at "
			    "least %d argument%s", nargs,
			    plural_form((unsigned int)nargs));
			return (-1);
		}
	}

	/* Validate arguments if needed. */
	if (conf_param->arg_regexp != NULL)
		if (regexec_simple(conf_param->arg_regexp, parser_args) != 0) {
			logconfx("wrong format of an argument");
			return (-1);
		}

	/* Parse it. */
	get_arg = &get_arg_tbl[conf_param->arg_type];
	if (get_arg->reg != NULL)
		if (regexec_simple(get_arg->reg, parser_args) != 0) {
			logconfx("wrong format of an argument");
			return (-1);
		}
	if (get_arg->parse(get_arg->argp) < 0)
		return (-1);
	if (conf_param->arg_parse != NULL)
		if (conf_param->arg_parse(get_arg->argp) < 0)
			return (-1);

	curparam = NULL;
	if (curmodfile != NULL && in_own_sect) {
		/* Restore ipastat.conf parameters tables. */
		curmodfile = NULL;
		conf_param_hash_ptr = conf_param_hash;
		conf_param_tbl_ptr = conf_param_tbl;
	}

	return (0);
}

static int
token_section_end(void)
{
	struct sect_stack *sect_stack;

	if (!in_own_sect) {
		/* Register configuration event in module for its section. */
		if (st_mod_conf_event(cur_st_mod,
		    IPA_CONF_EVENT_CUSTOM_SECT_END, section, (void *)NULL) < 0)
			return (-1);
	} else {
		/*
		 * Register configuration event in all modules for
		 * ipastat.conf's section.
		 */
		if (mod_conf_event(conf_event_begin[section] + 1,
		    conf_event_no, conf_event_arg) < 0)
			return (-1);
#ifdef WITH_ANY_LIMITS
		switch (section) {
# ifdef WITH_LIMITS
		case IPA_CONF_SECT_LIMIT:
			(void)parser_local_sym_del("limit");
			break;
# endif
# ifdef WITH_THRESHOLDS
		case IPA_CONF_SECT_THRESHOLD:
			(void)parser_local_sym_del("threshold");
			break;
# endif
		}
#endif /* WITH_ANY_LIMITS */
	}

	sect_stack = SLIST_FIRST(&sect_stack_list);
	SLIST_REMOVE_HEAD(&sect_stack_list, link);
	section = sect_stack->section;
	mem_free(sect_stack, m_anon);
	if (!in_own_sect && section < IPA_CONF_SECT_CUSTOM_OFFSET) {
		/*
		 * We were in custom section and previous section
		 * is not a custom section.  Restore ipastat.conf tables.
		 */
		curmodfile = NULL;
		conf_sect_tbl_ptr = conf_sect_tbl;
		conf_param_tbl_ptr = conf_param_tbl;
		conf_sect_hash_ptr = conf_sect_hash;
		conf_param_hash_ptr = conf_param_hash;
		in_own_sect = 1;
	}

	return (0);
}

static int
deinit_config_mods(void)
{
	const struct ipa_st_mod *ipa_st_mod;
	const struct st_mod *st_mod;

	SLIST_FOREACH(st_mod, &st_mod_list, link) {
		ipa_st_mod = st_mod->ipa_st_mod;
		if (mimic_real_config)
			if (ipa_st_mod->conf_mimic_real() < 0) {
				logconfe("module %s: conf_mimic_real failed",
				    st_mod->mod_file);
				return (-1);
			}
		deinit_conf_tbls(1,
		    ipa_st_mod->conf_sect_tbl, st_mod->conf_sect_hash,
		    ipa_st_mod->conf_param_tbl, st_mod->conf_param_hash);
		if (ipa_st_mod->conf_deinit() < 0) {
			logconfe("module %s: conf_deinit failed",
			    st_mod->mod_file);
			return (-1);
		}
	}
	return (0);
}

int
configure(PARSING_MODE mode)
{
	struct parser_pb *pb;

	use_log = mode != TEST_PARSING ? 1 : 0;

	/* Set wrappers for log functions during configuration. */
	xvlogmsgx = vlogconfx_priority;
	mvlogmsgx = vlogconfe;
	parser_vlogmsgx = parser_vlogmsgx_wrapper;

	if (init_config_data() < 0)
		goto failed;

	section = IPA_CONF_SECT_ROOT;
	memfunc.m_parser = m_parser;

	if (check_conf_file(ipastat_conf_file, 0) < 0)
		goto failed;

	/* Initialize parser and first pb. */
	if (parser_init() < 0)
		goto failed;
	pb = parser_new_pb(0);
	if (pb == NULL)
		return (-1);
	pb->fname = ipastat_conf_file;
	pb->fp = fopen(ipastat_conf_file, "r");
	if (pb->fp == NULL) {
		logconfe("fopen(%s, \"r\"): %s", ipastat_conf_file,
		    strerror(errno));
		goto failed;
	}
	if (parser_push_pb(pb) < 0)
		goto failed;
	include_depth = 1;

	/* Needed for log messages. */
	curparam = cursect = curmodfile = NULL;

	/* Set ipastat.conf configuration tables. */
	conf_sect_tbl_ptr = conf_sect_tbl;
	conf_param_tbl_ptr = conf_param_tbl;
	conf_sect_hash_ptr = conf_sect_hash;
	conf_param_hash_ptr = conf_param_hash;
	in_own_sect = 1;

	for (;;) {
		switch (parser_read_string()) {
		case 1:
			/* Successfully read one logical line. */
			break;
		case 0:
			/* EOF of current configuration file. */
			--include_depth;
			pb = parser_top_pb();
			if (fclose(pb->fp) != 0) {
				logconfe("fclose(%s): %s", pb->fname,
				    strerror(errno));
				goto failed;
			}
			if (pb->fname != ipastat_conf_file)
				mem_free(pb->fname, m_parser);
			pb = parser_pop_pb();
			if (pb == NULL)
				goto end_of_parsing;
			/* Initialize previous file for parsing. */
			pb->fp = fopen(pb->fname, "r");
			if (pb->fp == NULL) {
				logconfe("fopen(%s, \"r\"): %s", pb->fname,
				    strerror(errno));
				goto failed;
			}
			if (fseek(pb->fp, pb->foff, SEEK_SET) < 0) {
				logconfe("fseek(%s, %ld, SEEK_SET): %s",
				    pb->fname, pb->foff, strerror(errno));
				goto failed;
			}
			continue;
		default: /* -1 */
			goto failed;
		}
		switch (parser_token_id) {
		case TOKEN_ID_SECTION_BEGIN:
			if (token_section_begin() < 0)
				goto failed;
			break;
		case TOKEN_ID_SECTION_END:
			if (token_section_end() < 0)
				goto failed;
			break;
		case TOKEN_ID_PARAMETER:
			if (token_parameter() < 0)
				goto failed;
			break;
		}
	}

end_of_parsing:
	if (deinit_config_mods() < 0)
		goto failed;

	if (parser_deinit() < 0)
		goto failed;

	deinit_config_data();

	if (mode != TEST_PARSING)
		mimic_real_config = 1;

	if (mimic_real_config) {
		set_global_params();
		rulepats_inherit();
		if (rules_inherit() < 0)
			goto failed;
	}

	/* Set wrappers for log functions after configuration. */
	mvlogmsgx = mvlogmsgx_wrapper;
	xvlogmsgx = vlogmsgx;
	return (0);

failed:
	log_include_history();
	logconfe("configuration file parsing failed!");
	return (-1);
}

/*
 * Unload all modules and free memory used by structures which
 * describe loaded modules.  Note that any pointer should not
 * references any data in unloaded modules' memory.  So, the best
 * place when to call this function: after memfunc_deinit_1() and
 * before memfunc_deinit_2().
 */
int
unload_all_mods(void)
{
	struct st_mod *st_mod, *st_mod_next;
	int rv;

	rv = 0;
	SLIST_FOREACH_SAFE(st_mod, &st_mod_list, link, st_mod_next) {
		if (dl_close(st_mod->mod_handle) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: dl_close failed: %s",
			    st_mod->mod_file, dl_error());
			rv = -1;
		}
		mem_free(st_mod->mod_file, m_parser);
		mem_free(st_mod, m_anon);
	}
	return (rv);
}

static void
mod_conf_show(unsigned int sect_id, unsigned int no)
{
	const struct st_mod *st_mod;

	need_nl = 0;
	SLIST_FOREACH(st_mod, &st_mod_list, link)
		st_mod->ipa_st_mod->conf_show(sect_id, no);
	if (sect_id == IPA_CONF_SECT_ROOT)
		print_nl_cond();
}

static void
show_st_list(const struct st_list *list)
{
	if (list != NULL) {
		print_param_name("st_list");
		if (list != &st_list_null) {
			const struct st_elem *st;

			for (st = STAILQ_FIRST(list);;) {
				printf("%s", st->ipa_st_mod->st_name);
				st = STAILQ_NEXT(st, link);
				if (st == NULL)
					break;
				printf(" ");
			}
		} else
			printf("null");
		print_param_end();
	}
}

#ifdef WITH_LIMITS
static void
show_limits(const struct limits_list *list)
{
	const struct limit *limit;

	STAILQ_FOREACH(limit, list, link) {
		print_sect_name("limit");
		printf("%s ", limit->name);
		print_sect_begin();
		show_st_list(limit->st_list);
		mod_conf_show(IPA_CONF_SECT_LIMIT, limit->no);
		print_sect_end();
	}
}
#endif

#ifdef WITH_THRESHOLDS
static void
show_thresholds(const struct thresholds_list *list)
{
	const struct threshold *threshold;

	STAILQ_FOREACH(threshold, list, link) {
		print_sect_name("threshold");
		printf("%s ", threshold->name);
		print_sect_begin();
		show_st_list(threshold->st_list);
		mod_conf_show(IPA_CONF_SECT_THRESHOLD, threshold->no);
		print_sect_end();
	}
}
#endif

/*
 * Main function for outputting configuration.
 */
void
show_config(void)
{
	const struct rulepat *rulepat;
	const struct st_mod *st_mod;
	const struct rule *rule;
	unsigned int i;

	printf(
"/*\n\
 * This output is not identical to the original content of the\n\
 * configuration file(s), this is just how ipastat(8) and IPA modules\n\
 * see their configurations.  Any \"include\" or \"include_files\"\n\
 * parameters are not printed and all macro variables are expanded.\n\
 */\n\n");

	if (mimic_real_config)
		printf("/* Mimic real configuration regime (" IPASTAT_NAME "-"
		    PACKAGE_VERSION "). */\n\n");

	SLIST_FOREACH(st_mod, &st_mod_list, link) {
		print_param_name0("st_mod");
		print_string(st_mod->mod_file);
		print_param_end();
		need_nl = 1;
	}
	print_nl_cond();

	if (posix_re_pattern >= 0) {
		print_param_name("posix_re_pattern");
		print_boolean(posix_re_pattern);
		print_param_end();
		need_nl = 1;
	}
	print_nl_cond();

	if (debug_st_null >= 0) {
		print_param_name("debug_st_null");
		printf("%d", debug_st_null);
		print_param_end();
		need_nl = 1;
	}
	print_nl_cond();

	if (dynamic_rules >= 0) {
		print_param_name("dynamic_rules");
		print_boolean(dynamic_rules);
		print_param_end();
		need_nl = 1;
	}
#ifdef WITH_LIMITS
	if (dynamic_limits >= 0) {
		print_param_name("dynamic_limits");
		print_boolean(dynamic_limits);
		print_param_end();
		need_nl = 1;
	}
#endif
#ifdef WITH_THRESHOLDS
	if (dynamic_thresholds >= 0) {
		print_param_name("dynamic_thresholds");
		print_boolean(dynamic_thresholds);
		print_param_end();
		need_nl = 1;
	}
#endif
	print_nl_cond();

	if (value_units >= 0) {
		print_param_name("value_units");
		for (i = 0; i < VALUE_UNITS_TBL_SIZE; ++i)
			if (value_units == value_units_tbl[i].val) {
				printf("%s", value_units_tbl[i].str);
				print_param_end();
				print_nl();
				break;
			}
	}

	mod_conf_show(IPA_CONF_SECT_ROOT, 0);

	if (global_section_set) {
		print_sect_name("global");
		print_sect_begin();
		show_st_list(global_st_list);
		mod_conf_show(IPA_CONF_SECT_GLOBAL, 0);
		print_sect_end();
		print_nl();
	}

	STAILQ_FOREACH(rulepat, &rulepats_list, link) {
		print_sect_name("rulepat");
		print_string(rulepat->pat);
		printf(" ");
		print_sect_begin();
		if (rulepat->check_next >= 0) {
			print_param_name("check_next_rulepat");
			print_boolean(rulepat->check_next);
			print_param_end();
		}
		show_st_list(rulepat->st_list);
		mod_conf_show(IPA_CONF_SECT_RULE, rulepat->no);
#ifdef WITH_LIMITS
		show_limits(&rulepat->limits);
#endif
#ifdef WITH_THRESHOLDS
		show_thresholds(&rulepat->thresholds);
#endif
		print_sect_end();
		print_nl();
	}

	STAILQ_FOREACH(rule, &rules_list, list) {
		print_sect_name("rule");
		printf("%s ", rule->name);
		print_sect_begin();
		show_st_list(rule->st_list);
		mod_conf_show(IPA_CONF_SECT_RULE, rule->no);
#ifdef WITH_LIMITS
		show_limits(&rule->limits);
#endif
#ifdef WITH_THRESHOLDS
		show_thresholds(&rule->thresholds);
#endif
		print_sect_end();
		print_nl();
	}
}
