/*-
 * Copyright (c) 2003 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *   @(#)$Id: ipa_thresholds.h,v 1.2 2011/01/23 18:42:35 simon Exp $
 */

#ifndef IPA_THRESHOLD_H
#define IPA_THRESHOLD_H

#ifdef WITH_THRESHOLDS

#ifndef THRESHOLD_NSIZE
# define THRESHOLD_NSIZE 30
#endif

#ifndef THRESHOLD_NALLOC
# define THRESHOLD_NALLOC 20
#endif

#define THRESHOLD_JUMP_OVER_INACTIVE	0x1
#define THRESHOLD_JUMP_OVER_STOPPED	0x2
#define THRESHOLD_JUMP_OVER_ALLBITS	(THRESHOLD_JUMP_OVER_INACTIVE | \
					 THRESHOLD_JUMP_OVER_STOPPED)

#define THRESHOLD_FLAG_ACTIVE	0x1	/* Threshold is active. */
#define THRESHOLD_FLAG_SET	0x2	/* "threshold" parameter was given. */
#define THRESHOLD_FLAG_INITED	0x4	/* Threshold was initialized. */

#define THRESHOLD_IS_ACTIVE(x)	  ((x)->thr_flags & THRESHOLD_FLAG_ACTIVE)
#define THRESHOLD_IS_INACTIVE(x)  (!THRESHOLD_IS_ACTIVE(x))
#define THRESHOLD_IS_SET(x)	  ((x)->thr_flags & THRESHOLD_FLAG_SET)
#define THRESHOLD_IS_NOTSET(x)	  (!THRESHOLD_IS_SET(x))
#define THRESHOLD_IS_INITED(x)	  ((x)->thr_flags & THRESHOLD_FLAG_INITED)

#define THRESHOLD_SET_ACTIVE(x)	  ((x)->thr_flags |= THRESHOLD_FLAG_ACTIVE)
#define THRESHOLD_SET_INACTIVE(x) ((x)->thr_flags &= ~THRESHOLD_FLAG_ACTIVE)
#define THRESHOLD_SET_INITED(x)	  ((x)->thr_flags |= THRESHOLD_FLAG_INITED)

struct rule;

/*
 * rule { threshold {}} section.
 * Fields tm_started and tm_updated are dynamically changed.
 */
struct threshold {
	STAILQ_ENTRY(threshold) link;	/* Link for list of thresholds. */

	unsigned int	no;		/* Threshold ordinal number. */

	unsigned int	thr_flags;	/* ORed THRESHOLD_FLAG_xxx */

	uint64_t	thr_max;	/* thr + thr_dev */
	uint64_t	thr_min;	/* thr - thr_dev */
	uint64_t	cnt;		/* Positive counter. */
	uint64_t	cnt_neg;	/* Negative counter. */
	uint64_t	*cnt_slice;	/* Counters for time slices. */
	BITMAP_TYPE	*cnt_slice_sign;/* Signs for cnt_slice. */
	unsigned int	cnt_slice_n;	/* Number of time slices. */
	unsigned int	cnt_slice_i;	/* Current index in cnt_slice. */

	unsigned int	time_width;	/* { threshold_time_width } */
	const struct tevent *time_slice;/* { threshold_time_slice } */
	const struct worktime *worktime;/* { worktime } */

	const struct db_list *db_list;	/* { db_list } */

	unsigned int	below_lim;	/* X:-:- { threshold_balance } */
	unsigned int	equal_lim;	/* -:X:- { threshold_balance } */
	unsigned int	above_lim;	/* -:-:X { threshold_balance } */
	unsigned int	below_cnt;	/* Rest from below_lim. */
	unsigned int	equal_cnt;	/* Rest from equal_lim. */
	unsigned int	above_cnt;	/* Rest from above_lim. */

	struct cmds	below_thr;	/* { below_threshold {}} */
	struct cmds	equal_thr;	/* { equal_threshold {}} */
	struct cmds	above_thr;	/* { above_threshold {}} */

	int		shift_window;	/* If set, then update tm_started. */
	ipa_tm		tm_started;	/* Time when time window was started. */
	ipa_tm		tm_updated;	/* Time when threshold was updated. */
	unsigned int	check_sec;	/* Time when to check threshold. */

	struct wpid	wpid;		/* Threshold's wpid structure. */

	const struct rule *rule;	/* Pointer to threshold's rule. */

	uint64_t	thr;		/* { threshold } */
	uint64_t	thr_dev;	/* { threshold_deviation } */
	unsigned int	thr_dev_pc;	/* { threshold_deviation } in xx%. */

	char		*name;		/* Name of this threshold. */
	char		*info;		/* { info } */

	unsigned char	cnt_type;	/* Type of "threshold" parameter. */
	signed char	load_thr;	/* { load_threshold } */
	signed char	thr_type;	/* { threshold_type } */

	struct cmds	rc[2];		/* { startup|shutdown {}} */
};

/*
 * List of all thresholds in one rule.
 */
STAILQ_HEAD(thresholds_list, threshold);

extern const char *const threshold_event_msg[];

extern signed char global_debug_threshold;
extern signed char global_debug_threshold_init;
extern signed char global_threshold_type;
extern signed char global_load_threshold;
extern unsigned int global_threshold_time_width;
extern const struct tevent *global_threshold_time_slice;

extern unsigned int global_threshold_below_lim;
extern unsigned int global_threshold_equal_lim;
extern unsigned int global_threshold_above_lim;

extern ipa_mzone *threshold_mzone;

extern void	set_thr_min_max(struct threshold *);
extern int	init_thresholds(const struct rule *);
extern int	check_thresholds(const struct rule *, unsigned int *);
extern int	thresholds_newday(struct rule *);

extern int	threshold_add_chunk(const struct rule *, struct threshold *,
		    const uint64_t *);
extern int	threshold_sub_chunk(const struct rule *, struct threshold *,
		    const uint64_t *);
extern int	thresholds_add_chunk(const struct rule *, const uint64_t *);
extern int	thresholds_sub_chunk(const struct rule *, const uint64_t *);

#define set_threshold_inactive(r, t) mod_set_threshold_active((r), (t), 0)
extern int	mod_set_threshold_active(const struct rule *,
		    struct threshold *, int);

extern int	copy_thresholds(struct rule *, const struct thresholds_list *);
extern void	free_thresholds(unsigned int, struct thresholds_list *, int);

extern void	threshold_init_cmds(struct threshold *);
extern void	threshold_inherit(struct threshold *threshold);

extern struct threshold *threshold_by_name(const struct rule *, const char *);

#endif /* WITH_THRESHOLDS */

extern unsigned int nstatthresholds;
extern unsigned int ndynthresholds;

#endif /* !IPA_THRESHOLD_H */
