/*-
 * Copyright (c) 2000-2003 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *   @(#)$Id: ipa_cmd.h,v 1.2 2011/01/23 18:42:34 simon Exp $
 */

#ifndef IPA_CMD_H
#define IPA_CMD_H

#define CMD_NSIZE	30
#define CMD_NALLOC	20

#define RC_STARTUP	0		/* Any startup {} */
#define RC_SHUTDOWN	1		/* Any shutdown {} */
#define RC_NOTSET	2		/* Not set value. */

#ifdef WITH_LIMITS
struct limit;
#endif

#ifdef WITH_THRESHOLDS
struct threshold;
#endif

enum {
#ifdef WITH_LIMITS
	WPID_TYPE_LIMIT,
#endif
#ifdef WITH_SUBLIMITS
	WPID_TYPE_SUBLIMIT,
#endif
#ifdef WITH_THRESHOLDS
	WPID_TYPE_THRESHOLD,
#endif
	WPID_TYPE_NOT_USED
};

/*
 * Wpid ("wait PID") describes one forked process, which runs
 * a list of commands from some section in background.
 */
struct wpid {
	LIST_ENTRY(wpid) link;		/* For list building in hash. */
	pid_t		pid;		/* PID of background process. */
	unsigned int	type;		/* Selector for next union. */
#ifdef WITH_ANY_LIMITS
	union {
# ifdef WITH_LIMITS
		const struct limit *limit;
# endif
# ifdef WITH_SUBLIMITS
		const struct sublimit *sublimit;
# endif
# ifdef WITH_THRESHOLDS
		const struct threshold *threshold;
# endif
	} u;
#endif /* WITH_ANY_LIMITS */
	signed char	debug_exec;	/* Copy of debug_exec parameter. */
};

#ifdef WITH_ANY_LIMITS

#ifndef WPID_HASH_BUCKETS
# define  WPID_HASH_BUCKETS 32		/* Must be power of 2. */
#endif

LIST_HEAD(wpid_hash, wpid);

extern struct wpid_hash wpid_hash[];

#define wpid_hash_bucket(pid) ((unsigned int)(pid) & (WPID_HASH_BUCKETS - 1))

extern void	wpid_hash_init(void);
extern int	wpid_hash_is_empty(void);
extern void	wpid_hash_rem(struct wpid *);
extern struct wpid *wpid_lookup_pid(pid_t);
extern void	log_term_wpid(const struct wpid *);

#endif /* WITH_ANY_LIMITS */

#define CMD_FREE_STR	0x01		/* Free str buffer. */
#define CMD_FREE_USER	0x02		/* Free user buffer. */

/*
 * Structure for "exec" parameter.
 */
struct cmd {
	STAILQ_ENTRY(cmd) link;		/* List of all cmd. */
	char		*str;		/* Command string. */
	char		*user;		/* User name or NULL. */
	size_t		str_size;	/* Command string size. */
	unsigned int	subst_per_cent;	/* Number of "%%" in str. */
	unsigned int	subst_rule;	/* Number of "%rule%" in str. */
	unsigned char	free_mask;	/* CMD_FREE_xxx bits. */
};

STAILQ_HEAD(cmd_list, cmd);

/*
 * All commands in one section, which can contain "exec" and "ictl"
 * parameters.  By default any "startup" and "shutdown" section has
 * non-zero 'sync' field, any other section has 'sync' equal to zero.
 */
struct cmds {
	signed char	sync;		/* "sync_exec" parameter. */
	char		sect_set;	/* Set if section is specified. */
	char		has_cmd;	/* Set if has "exec" or "ictl". */
	struct cmd_list	cmd_list;	/* List of "exec" commands. */
#ifdef WITH_RULES
	struct ictl_list ictl_list;	/* List of "ictl" commands. */
#endif
};

/*
 * rule|rulepat|autorule { startup|shutdown{}}
 */
struct cmds_rule {
	struct cmds	cmds;
#ifdef WITH_LIMITS
	struct cmds	cmds_all_reached;
	struct cmds	cmds_all_not_reached;
	struct cmds	cmds_any_reached;
	struct cmds	cmds_any_not_reached;
#endif
};

#ifdef WITH_LIMITS
/*
 * limit|sublimit { startup|shutdown{}}
 */
struct cmds_limit {
	struct cmds	cmds;
	struct cmds	cmds_reached;
	struct cmds	cmds_not_reached;
};
#endif

extern signed char global_debug_exec;

extern char	*shell_path;
extern char	*shell_path_default;
extern char	*shell_arg1;
extern char	*shell_arg1_default;

const char *const rc_sect_name[2];

extern ipa_mzone *cmd_mzone;

extern ipa_mem_type *m_cmd;

extern struct cmds cmds_global[2];

extern int	setsuppgids(const char *, gid_t);

struct rule;

extern int	run_all_cmds(unsigned int);
extern int	run_cmds(const struct rule *, struct wpid *,
		    const struct cmds *, const char *, ...)
		    ATTR_FORMAT(printf, 4, 5);
extern int	run_cmds_cons(const struct cmds *, const char *, ...)
		    ATTR_FORMAT(printf, 2, 3);
extern int	run_cmds_rule(struct rule *, unsigned int);

extern void	cmds_init(struct cmds *);
extern void	cmds_free(struct cmds *);
extern int	cmds_copy(const struct rule *, struct cmds *,
		    const struct cmds *);

extern void	cmds_rule_init(struct cmds_rule *);
extern void	cmds_rule_free(struct cmds_rule *);
extern void	cmds_rule_set_sync(struct cmds_rule *);
extern int	cmds_rule_copy(const struct rule *, struct cmds_rule *,
		    const struct cmds_rule *);

#ifdef WITH_LIMITS
extern void	cmds_limit_init(struct cmds_limit *);
extern void	cmds_limit_free(struct cmds_limit *);
extern void	cmds_limit_set_sync(struct cmds_limit *);
extern int	cmds_limit_copy(const struct rule *, struct cmds_limit *,
		    const struct cmds_limit *);
#endif

#endif /* !IPA_CMD_H */
