/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method 
 *
 * Copyright (C) 2003 Botond Botyanszki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * Based on fontset.c from nabi by Choe Hwanjin
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <glib.h>

#include "../error.h"
#include "../nls.h"

#include "xim-fontset.h"

static GHashTable *fontset_hash = NULL;
static GSList *fontset_list = NULL;
static Display *m_display = NULL;

static IMJAXimFontSet *im_ja_xim_fontset_new(const char *name) {
	XFontSet xfontset;
	XFontStruct **font_structs;
	int i, nfonts, ascent, descent;
	char **font_names;
	IMJAXimFontSet *fontset;
	char **missing_list;
	int missing_list_count;
	char *error_message = NULL;

	IM_JA_DEBUG("im_ja_xim_fontset_new: %s\n", name);

	xfontset = XCreateFontSet(m_display,
														name,
														&missing_list,
														&missing_list_count,
														&error_message);

	/*
	if (error_message != NULL) {
		IM_JA_DEBUG("Error loading fontset: [%s]\n", error_message);
		return NULL;
	}
	*/
	if (missing_list_count > 0) {
		int i;
		IM_JA_DEBUG("**WARNING** MISSING CHARSETS IN FONTSET:\n");
		IM_JA_DEBUG("\t FONT: %s\n", name);
		for (i = 0; i < missing_list_count; i++) {
			IM_JA_DEBUG("\t CHARSET: %s\n", missing_list[i]);
		}
		XFreeStringList(missing_list);
		/* return NULL; */

		/* Create the fontset from a hardcoded one, otherwise it won't display anything in the preedit */
		xfontset = XCreateFontSet(m_display,
															"-misc-fixed-medium-r-normal--14-130-75-75-c-140-jisx0208.1983-0,-misc-fixed-medium-r-normal--14-130-75-75-c-70-jisx0201.1976-0",
															&missing_list,
															&missing_list_count,
															&error_message);
		
	}

	/* get acsent and descent */
	nfonts = XFontsOfFontSet(xfontset, &font_structs, &font_names);
	/* find width, height */
	for (i = 0, ascent = 1, descent = 0; i < nfonts; i++) {
		if (ascent < font_structs[i]->ascent)
			ascent = font_structs[i]->ascent;
		if (descent < font_structs[i]->descent)
			descent = font_structs[i]->descent;
	}

	fontset = g_new0(IMJAXimFontSet,1 );
	fontset->name = g_strdup(name);
	fontset->ref = 1;
	fontset->xfontset = xfontset;
	fontset->ascent = ascent;
	fontset->descent = descent;

	g_hash_table_insert(fontset_hash, fontset->name, fontset);
	fontset_list = g_slist_prepend(fontset_list, fontset);

	return fontset;
}

static void im_ja_xim_fontset_ref(IMJAXimFontSet *fontset) {
	if (fontset == NULL) return;
	fontset->ref++;
}

static void im_ja_xim_fontset_unref(IMJAXimFontSet *fontset){
	if (fontset == NULL)	return;

	fontset->ref--;
	if (fontset->ref <= 0) {
		g_hash_table_remove(fontset_hash, fontset->name);
		fontset_list = g_slist_remove(fontset_list, fontset);

		XFreeFontSet(m_display, fontset->xfontset);
		g_free(fontset->name);
		g_free(fontset);
	}
}

static IMJAXimFontSet *im_ja_xim_fontset_find_by_xfontset(XFontSet xfontset) {
	IMJAXimFontSet *fontset;
	GSList *list;
	list = fontset_list;

	while (list != NULL) {
		fontset = (IMJAXimFontSet*)(list->data);
		if (fontset->xfontset == xfontset)
	    return fontset;
		list = list->next;
	}

	return NULL;
}

IMJAXimFontSet *im_ja_xim_fontset_create(Display *display, const char *fontset_name) {
	IMJAXimFontSet *im_ja_xim_fontset;

	IM_JA_DEBUG("im_ja_xim_fontset_create\n");

	m_display = display;
	if (fontset_hash == NULL) {
		fontset_hash = g_hash_table_new(g_str_hash, g_str_equal);
	}

	im_ja_xim_fontset = g_hash_table_lookup(fontset_hash, fontset_name);
	if (im_ja_xim_fontset != NULL) {
		im_ja_xim_fontset_ref(im_ja_xim_fontset);
		return im_ja_xim_fontset;
	}

	IM_JA_DEBUG("Loading fontset: %s\n", fontset_name);
	im_ja_xim_fontset = im_ja_xim_fontset_new(fontset_name);

	return im_ja_xim_fontset;
}

void im_ja_xim_fontset_free(Display *display, XFontSet xfontset) {
	IMJAXimFontSet *im_ja_xim_fontset;

	m_display = display;
	im_ja_xim_fontset = im_ja_xim_fontset_find_by_xfontset(xfontset);
	im_ja_xim_fontset_unref(im_ja_xim_fontset);
}

void im_ja_xim_fontset_free_all(Display *display) {
	IMJAXimFontSet *fontset;
	GSList *list;
	m_display = display;

	if (fontset_list != NULL) {
		IM_JA_DEBUG("remaining fontsets will be freed, this must be an error\n");
		list = fontset_list;
		while (list != NULL) {
	    fontset = (IMJAXimFontSet*)(list->data);
	    XFreeFontSet(m_display, fontset->xfontset);
	    g_free(fontset->name);
	    g_free(fontset);
	    list = list->next;
		}
	}

	if (fontset_hash != NULL) g_hash_table_destroy(fontset_hash);
	if (fontset_list != NULL) g_slist_free(fontset_list);
}

