/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 */
#include <config.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "../nls.h"
#include "../im-ja.h"
#include "../im-ja-impl.h"
#include "../error.h"
#include "../common.h"
#include "../conf.h"
#include "../romakana.h"
#include "../candwin.h"

#include "wnnlib.h"

/* * * * * * * * * * * */
extern IMJAConfig cfg;
/* * * * * * * * * * * */

void errorHandler(char *s) {
	gchar *utf8; 

	utf8 = euc2utf8(s);
	IM_JA_DEBUG("wnn server error message: %s\n", utf8);
	im_ja_print_error("wnn error: %s\n", utf8); 
	g_free(utf8);
}

int messageHandler(char *s) {
	gchar *utf8; 
	gboolean retval = FALSE;

	utf8 = euc2utf8(s);
	IM_JA_DEBUG("wnn server message: %s\n", utf8);
	retval = im_ja_print_question(_("wnn server message:\n %s\n"), utf8);
	g_free(utf8);

	return retval;
}


void im_ja_wnn_shutdown(IMJAContext *cn) {
	jcClose(cn->wnn_stat.wnn_buf);
	jcDestroyBuffer(cn->wnn_stat.conv_buf, 0);
	cn->wnn_stat.wnn_buf = NULL;
	cn->wnn_stat.conv_buf = NULL;
}

gboolean im_ja_wnn_init(IMJAContext *cn) {
	IM_JA_DEBUG("im_ja_wnn_init\n");
	if (cn->wnn_stat.envname == NULL) cn->wnn_stat.envname = "im-ja";
	if (cfg.wnnenvrc == NULL) cfg.wnnenvrc = "";

	if (cn->wnn_stat.wnn_buf != NULL) {
		if (jcIsConnect(cn->wnn_stat.wnn_buf) == TRUE) {
			jcClose(cn->wnn_stat.wnn_buf);
			cn->wnn_stat.wnn_buf = NULL;
		}
		jcDestroyBuffer(cn->wnn_stat.conv_buf, 0);
		cn->wnn_stat.wnn_buf = NULL;
	}
	cn->wnn_stat.wnn_buf = jcOpen2(cfg.wnn_address, cn->wnn_stat.envname, FALSE, cfg.wnnenvrc, 
																 cfg.wnnenvrc, errorHandler, messageHandler, 30);
	
	if (cn->wnn_stat.wnn_buf == NULL) {
		im_ja_print_error(_("jcOpen() failed.\n")); 
		return FALSE;
	}
	if (jcIsConnect(cn->wnn_stat.wnn_buf) == FALSE) {
		im_ja_print_error(_("wnn init failed: Couldn't connect to the wnn server.\nPlease check your settings and make sure the wnn server is running!")); 
		return FALSE;
	}
	cn->wnn_stat.conv_buf = jcCreateBuffer(cn->wnn_stat.wnn_buf, 10, 80);
	if (cn->wnn_stat.conv_buf == NULL) {
		im_ja_print_error(_("jcCreateBuffer failed\n"));	 
		return FALSE;
	}

	cn->im_ja_conv_engine_reset_context = im_ja_wnn_reset;
	cn->im_ja_conv_engine_filter_keypress = im_ja_wnn_filter_keypress;
	cn->im_ja_conv_engine_shutdown = im_ja_wnn_shutdown;
	cn->im_ja_conv_engine_select_candidate = im_ja_wnn_select_candidate;
	cn->im_ja_conv_engine_update_preedit = im_ja_wnn_update_preedit;
	cn->im_ja_conv_engine_commit = im_ja_wnn_commit;
	return TRUE;
}

void im_ja_wnn_show_candidates(IMJAContext *cn) {
	gchar *tmpeuc;
	wchar candbuf[256];
	gint candnum, currcand;
	gint cand;
	jcConvBuf *cnv_buf = cn->wnn_stat.conv_buf;

	if (cn->cand_stat == 0) return;

	while (cnv_buf->curClause > cn->cand_stat - 1) {
		jcMove(cnv_buf, 1, JC_BACKWARD);
	}
	/* showBuffers(cnv_buf, "Show candidates"); */
	im_ja_free_candidate_list(cn);
	if (jcCandidateInfo(cnv_buf, 0, &candnum, &currcand) >= 0) {
		for (cand = 0; cand < candnum; cand++) {
			jcGetCandidate(cnv_buf, cand, candbuf);
			tmpeuc = wc2euc(candbuf, 256);
			/* IM_JA_DEBUG("cand %d: %s\n", cand, euc2utf8(tmpeuc)); */
			cn->candidate_list = g_list_append(cn->candidate_list, euc2utf8(tmpeuc));
			g_free(tmpeuc);
		}
		candidate_window_show(cn, currcand);
		IM_JA_DEBUG("Candidates: %d, current: %d\n", candnum, currcand);
	}
}

static void im_ja_wnn_convert(IMJAContext *cn) {
	jcConvBuf *cnv_buf = cn->wnn_stat.conv_buf;
	IM_JA_DEBUG("convert\n");

	jcKana(cnv_buf, 1, JC_HIRAGANA);
	jcConvert(cnv_buf, 0, 0, 1);
	/* jcTop(cnv_buf); */
	while (cnv_buf->curClause > cn->cand_stat - 1) {
		jcMove(cnv_buf, 1, JC_BACKWARD);
	}
	/* showBuffers(cnv_buf, "Convert"); */
	im_ja_wnn_update_preedit(cn);
}

/*
static void im_ja_wnn_unconvert_all(IMJAContext *cn) {
	gint clause;
	jcConvBuf *cnv_buf = cn->wnn_stat.conv_buf;
	for (clause = 0; clause < cnv_buf->nClause; clause++) {
	}
}
*/

void im_ja_wnn_update_preedit(IMJAContext *cn) {
	gchar *tmpeuc;
	gint clause;
	gchar *utf8;

	jcConvBuf *cnv_buf = cn->wnn_stat.conv_buf;
	
	tmpeuc = wc2euc(cnv_buf->displayBuf, cnv_buf->displayEnd - cnv_buf->displayBuf);
	utf8 = euc2utf8(tmpeuc);
	g_strlcpy(cn->preedit_buf, utf8, BUFFERSIZE);
	g_free(utf8);
	cn->preedit_reverse_start = 0;
	cn->preedit_reverse_end = 0;
	g_free(tmpeuc);
	*cn->preedit_buf = 0;

	/* showBuffers(cnv_buf, "update preedit"); */
	
	for (clause = 0; clause < cnv_buf->nClause; clause++) {
		if (clause == cnv_buf->curClause) cn->preedit_reverse_start = strlen(cn->preedit_buf);
		tmpeuc = wc2euc(cnv_buf->clauseInfo[clause].dispp, cnv_buf->clauseInfo[clause + 1].dispp - cnv_buf->clauseInfo[clause].dispp);
		/* copy the current clause into the preedit buffer */
		if (clause == 0) {
			utf8 = euc2utf8(tmpeuc);
			g_strlcpy(cn->preedit_buf, utf8, BUFFERSIZE);
			g_free(utf8);
		}
		else {
			utf8 = euc2utf8(tmpeuc);
			g_strlcat(cn->preedit_buf, utf8, BUFFERSIZE);
			g_free(utf8);
		}
		g_free(tmpeuc);
		if (cnv_buf->clauseInfo[clause].conv == 0) { /* If not converted, don't highlight */
			cn->preedit_reverse_start = 0;
			cn->preedit_reverse_end = 0;
		}
		else if (clause == cnv_buf->curClause) { 
			cn->preedit_reverse_end = strlen(cn->preedit_buf);
		}
	}

	/* update cursor position */
	for (clause = 0; clause < cnv_buf->nClause; clause++) {
		if (cnv_buf->dot <= cnv_buf->clauseInfo[clause].kanap) break;
	}
	if (cnv_buf->dot < cnv_buf->clauseInfo[clause].kanap) clause--;
	cn->cursor_char_pos = cnv_buf->dot - cnv_buf->clauseInfo[clause].kanap;
	/* IM_JA_DEBUG("nClause: %d,  dot: %d.%d\n", cnv_buf->nClause, clause, cn->cursor_char_pos); */
	cn->cursor_char_pos += cnv_buf->clauseInfo[clause].dispp - cnv_buf->displayBuf;
	if (cnv_buf->nClause == clause) cn->cursor_char_pos = g_utf8_strlen(cn->preedit_buf, -1);

	if (g_utf8_validate(cn->preedit_buf, -1, NULL) != TRUE) {
		im_ja_print_error_cmdline("[wnn_preedit_update] utf8_validate failed:  %s\n", cn->preedit_buf);
	}

	/* showBuffers(cnv_buf, "preedit updated"); */
	im_ja_preedit_changed(cn);
	IM_JA_DEBUG("wnn_update_preedit: %s, cursor pos: %d\n", utf82euc(cn->preedit_buf), cn->cursor_char_pos);
	
	/* update candwin position */
	if (cnv_buf->clauseInfo[cnv_buf->curClause].conv == 1) cn->update_candwin_pos = TRUE;
	
}

void im_ja_wnn_reset(IMJAContext *cn) {
	IM_JA_DEBUG("im_ja_wnn_reset\n");
	jcClear(cn->wnn_stat.conv_buf);
	*(cn->preedit_insert) = 0;
}

static gboolean im_ja_wnn_convert_tailing_n(IMJAContext *cn) {
	/* Convert tailing n to hiragana */
	gchar *tmpeuc = NULL;

	if (cn->preedit_buf[strlen(cn->preedit_buf) - 1] == 'n') {
		switch (cn->input_method) {
		case IM_JA_HIRAGANA_INPUT:
			tmpeuc = utf82euc("\xE3\x82\x93");	/* hiragana n */
			break;
		case IM_JA_KATAKANA_INPUT:
			tmpeuc = utf82euc("\xE3\x83\xB3");	/* katakana n */
			break;
		case IM_JA_HALFKATA_INPUT:
			tmpeuc = utf82euc("\xEF\xBE\x9D");	/* narrow katakana n */
			break;
		default:
			return FALSE;
		}

		jcDeleteChar(cn->wnn_stat.conv_buf, 1);
		jcInsertChar(cn->wnn_stat.conv_buf, euc2wc(tmpeuc));
		g_free(tmpeuc);
		cn->preedit_insert[0] = 0; /* Kill temp buffer */
		return TRUE;
	}
	return FALSE;
}

void im_ja_wnn_commit(IMJAContext *cn) {
	if (im_ja_wnn_convert_tailing_n(cn) == TRUE) im_ja_wnn_update_preedit(cn);
	jcFix(cn->wnn_stat.conv_buf);
	im_ja_wnn_reset(cn);
}

gboolean im_ja_wnn_filter_keypress(IMJAContext *cn, GdkEventKey *key) {
  jcConvBuf *cnv_buf = cn->wnn_stat.conv_buf;
  
  if (key->type == GDK_KEY_RELEASE) {
    return FALSE;
  };

  /* ENTER */
  if ((key->keyval == GDK_Return ) || (key->keyval == GDK_KP_Enter) || (ishotkey(key, COMMIT_PREEDIT, &cfg) == TRUE)) {

		if (strlen(cn->preedit_buf) > 0) { /* Commit preedit buffer */
			im_ja_commit(cn);
			return TRUE;
		}
		return FALSE;
  }

  /* BACKSPACE */
  if (key->keyval == GDK_BackSpace ) {
    if (strlen(cn->preedit_insert) > 0) {
			*(g_utf8_prev_char(cn->preedit_insert + strlen(cn->preedit_insert))) = 0;
		}
		if (strlen(cn->preedit_buf) > 0) { /* Delete preedit */
      int isconv = 2;
      cn->cand_stat = 0;
			if (cnv_buf->nClause == cnv_buf->curClause) {
				if (cnv_buf->clauseInfo[cnv_buf->nClause - 1].conv == 1) isconv = TRUE;
			}
      else isconv = jcIsConverted(cnv_buf, cnv_buf->curClause);
      if (isconv == TRUE) {
				jcCancel(cnv_buf);
				jcBottom(cnv_buf);
				/* jcUnconvert(cnv_buf); */
      }
      else {
				jcDeleteChar(cnv_buf, 1);
      }
      im_ja_wnn_update_preedit(cn);
      return TRUE;
    }
    return FALSE; /* Delete normal */
  }

  /* DELETE */
  if (key->keyval == GDK_Delete ) {
    cn->preedit_insert[0] = 0; /* Kill temp buffer */
    if (strlen(cn->preedit_buf) > 0) { /* Delete preedit */
      cn->cand_stat = 0;
			/* if we are at the end of the preedit, don't delete anything. */
			if (g_utf8_strlen(cn->preedit_buf, -1) == cn->cursor_char_pos) return TRUE;
			/* If the current clause is converted, then unconvert it */
			if (jcIsConverted(cnv_buf, cnv_buf->curClause) == 1) {
				jcCancel(cnv_buf);
				jcBottom(cnv_buf);
				jcUnconvert(cnv_buf);
      }
      else {
				jcMove(cn->wnn_stat.conv_buf, 1, JC_FORWARD); 
				jcDeleteChar(cnv_buf, 1);
      }
      im_ja_wnn_update_preedit(cn);
      return TRUE;
    }
    return FALSE; /* Delete normal */
  }

  /* SPACE */
  if (key->keyval == GDK_space) { 
    if (strlen(cn->preedit_buf) == 0) {
			g_strlcpy(cn->preedit_buf, " ", BUFFERSIZE);
			im_ja_commit(cn); /* Insert a space */
			return TRUE;         
    }
		
    im_ja_wnn_convert_tailing_n(cn);
    
    /* FIXME: cand_stat needs to be reviewed. */
    if (cn->cand_stat == 0) { /* Convert to kanji */
      jcBottom(cnv_buf);
      cn->cand_stat = cnv_buf->curClause + 1;
      im_ja_wnn_convert(cn);
    }   
    else if (jcIsConverted(cnv_buf, cnv_buf->curClause) == 0) {
      im_ja_wnn_convert(cn);
    }
    else { /* pop up the candidate window */
      im_ja_wnn_show_candidates(cn);
    }
    return TRUE; 
  }

  /* Convert current bunsetsu, if converted show the next candidate */
  if (ishotkey(key, CONVERT, &cfg) == TRUE) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;

		im_ja_wnn_convert_tailing_n(cn);

    if (cn->cand_stat == 0) { /* Convert to kanji */
      cn->cand_stat = cnv_buf->curClause + 1;
      im_ja_wnn_convert(cn);
    }   
    else if (jcIsConverted(cnv_buf, cnv_buf->curClause) == 0) {
       im_ja_wnn_convert(cn);
    }
    else { /* Select next candidate */
      jcNext(cnv_buf, 1, JC_FORWARD);
      im_ja_wnn_update_preedit(cn);
    }
    return TRUE;
  }

  /* Unconvert current bunsetsu */ /* FIXME: cand stat needs to be set to unconverted, move to next doesn't work */
  if (ishotkey(key, UNCONVERT, &cfg) == TRUE) {
    if (jcIsConverted(cnv_buf, cnv_buf->curClause) == 1) {
			IM_JA_DEBUG("CONVERTED: %d %d\n", cnv_buf->curClause, cnv_buf->nClause);
			jcKana(cnv_buf, 1, JC_HIRAGANA);
			jcUnconvert(cnv_buf);
			cn->cand_stat = 0;
    }
		else {
			gchar *tmpeuc, *tmputf;
			gint clause;
			if (cnv_buf->curClause == cnv_buf->nClause) jcNext(cnv_buf, 1, JC_BACKWARD);
			clause = cnv_buf->curClause;
			if (cnv_buf->curClause == cnv_buf->nClause) clause = cnv_buf->curClause - 1;
			tmpeuc = wc2euc(cnv_buf->clauseInfo[clause].dispp, 
											cnv_buf->clauseInfo[clause + 1].dispp - cnv_buf->clauseInfo[clause].dispp);
			tmputf = euc2utf8(tmpeuc);
			if (isKatakanaChar(g_utf8_get_char(tmputf)) == TRUE) jcKana(cnv_buf, 1, JC_HIRAGANA);
			else if (isHiraganaChar(g_utf8_get_char(tmputf)) == TRUE)	jcKana(cnv_buf, 1, JC_KATAKANA);
			g_free(tmputf);
			g_free(tmpeuc);
		}
    im_ja_wnn_update_preedit(cn);
    return TRUE;
  }

  /* Show candidates, if not converted then convert also */
  if (ishotkey(key, LIST_CANDIDATES, &cfg) == TRUE) {
    if (cn->cand_stat == 0) { /* Convert to kanji */
			im_ja_wnn_convert_tailing_n(cn);
      cn->cand_stat = cnv_buf->curClause + 1;
      im_ja_wnn_convert(cn);
    }   
    im_ja_wnn_show_candidates(cn);
    return TRUE;
  }

  /* Expand current bunsetsu */
  if (ishotkey(key, EXPAND_BUNSETSU, &cfg) == TRUE) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
    jcExpand(cnv_buf, 1, 1);
    im_ja_wnn_update_preedit(cn);
    cn->cand_stat = cnv_buf->curClause + 1;
    return TRUE; 
  }

  /* Shrink current bunsetsu */
  if (ishotkey(key, SHRINK_BUNSETSU, &cfg) == TRUE) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
    jcShrink(cnv_buf, 1, 1);
    im_ja_wnn_update_preedit(cn);
    cn->cand_stat = cnv_buf->curClause + 1;
    return TRUE; 
  }

  /* Select next bunsetsu */
  if ((ishotkey(key, NEXT_BUNSETSU, &cfg) == TRUE) && (cn->cand_stat != 0)) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
    jcMove(cnv_buf, 1, JC_FORWARD);
    cn->cand_stat = cnv_buf->curClause + 1;
		IM_JA_DEBUG("next bunsetsu. cand_stat: %d\n", cn->cand_stat);
    im_ja_wnn_update_preedit(cn);
    return TRUE; 
  } 
 
  /* Select prev bunsetsu */
  if ((ishotkey(key, PREV_BUNSETSU, &cfg) == TRUE) && (cn->cand_stat != 0)) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
    jcMove(cnv_buf, 1, JC_BACKWARD);
    cn->cand_stat = cnv_buf->curClause + 1;
		IM_JA_DEBUG("prev bunsetsu. cand_stat: %d\n", cn->cand_stat);
    im_ja_wnn_update_preedit(cn);
    return TRUE; 
  }

  /* Select previous candidate */
  if (ishotkey(key, PREV_CANDIDATE, &cfg) == TRUE) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
    jcNext(cnv_buf, 1, JC_BACKWARD);
    im_ja_wnn_update_preedit(cn);
    return TRUE; 
  }

  /* Select next candidate */
  if (ishotkey(key, NEXT_CANDIDATE, &cfg) == TRUE) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
    jcNext(cnv_buf, 1, JC_FORWARD);
    im_ja_wnn_update_preedit(cn);
    return TRUE; 
  }

	/* Left arrow key: move cursor */
	if (((key->keyval == GDK_Left) || (key->keyval == GDK_KP_Left)) && (cn->cand_stat == 0)) {
		if (strlen(cn->preedit_buf) == 0) return FALSE;
		cn->preedit_insert[0] = 0; /* Kill temp buffer */
		IM_JA_DEBUG("move backward. cursor pos: %d\n", cn->cursor_char_pos);
		jcMove(cn->wnn_stat.conv_buf, 1, JC_BACKWARD);
		im_ja_wnn_update_preedit(cn);
		return TRUE;
	}

	/* Right arrow key: move cursor */
	if (((key->keyval == GDK_Right) || (key->keyval == GDK_KP_Right)) && (cn->cand_stat == 0)) {
		if (strlen(cn->preedit_buf) == 0) return FALSE;
		cn->preedit_insert[0] = 0; /* Kill temp buffer */
		IM_JA_DEBUG("move forward. cursor pos: %d\n", cn->cursor_char_pos);
		jcMove(cn->wnn_stat.conv_buf, 1, JC_FORWARD);
		im_ja_wnn_update_preedit(cn);
		return TRUE;
	}

  /* NORMAL CHAR KEYS */
	if (im_ja_is_printable_key(key) == TRUE) { 
    gchar utf8strg[7];
    gchar *lastkana, *insertptr, *tmpstr;
		gchar *kanastr; int kanalength;
		int insertlength;
		int i;
		gchar *tmpeuc;

    /* if (cn->cand_stat != 0) jcFix(cnv_buf); */ /* Commit preedit if it was already converted */
    cn->cand_stat = 0;

		utf8strg[g_unichar_to_utf8(gdk_keyval_to_unicode(key->keyval), utf8strg)] = 0;

		/* Move to the end if we are at a converted bunsetu */
		if (jcIsConverted(cnv_buf, cnv_buf->curClause) == 1) jcBottom(cnv_buf);

		tmpeuc = utf82euc(utf8strg);
		if ((unsigned int)tmpeuc[0] > 128) {
			g_free(tmpeuc);
			IM_JA_DEBUG("*ERROR* non-ascii input\n");
			im_ja_input_utf8(cn, utf8strg);
			return TRUE;
		}

		/* insert the char into the wnn buffer */
		jcInsertChar(cn->wnn_stat.conv_buf, euc2wc(tmpeuc));
		g_free(tmpeuc);

		/* add the new char to a temp buffer */
		g_strlcat(cn->preedit_insert, utf8strg, BUFFERSIZE);

		/* convert temp romaji string to kana */
		kanastr = roma2kana(cn->preedit_insert, cn->input_method);

		if (kanastr == NULL) return FALSE;

		kanalength = g_utf8_strlen(kanastr, -1);
		insertlength = g_utf8_strlen(cn->preedit_insert, -1);
		IM_JA_DEBUG("converted \"%s\" to \"%s\", size change %d -> %d\n", 
								utf82euc(cn->preedit_insert), utf82euc(kanastr), insertlength, kanalength);

		/* Find any converted hiragana */
		lastkana = kanastr + strlen(kanastr);
		while (isJPChar(g_utf8_get_char(g_utf8_prev_char(lastkana))) == FALSE) {
			lastkana = g_utf8_prev_char(lastkana);
			if (lastkana == kanastr) break;
		}

		if (lastkana != kanastr) { /* if we could convert any kana */
			/* Delete added chars from wnn buffer */
			for (i = 0; i < insertlength; i++) {
				jcDeleteChar(cn->wnn_stat.conv_buf, 1);
			}
			/* Read the new kana string */
			insertptr = kanastr;
			for (i = 0; i < kanalength; i++) {
				gchar *tmpeuc = utf82euc(insertptr);
				insertptr = g_utf8_next_char(insertptr);
				jcInsertChar(cn->wnn_stat.conv_buf, euc2wc(tmpeuc));
				g_free(tmpeuc);
			}
			/* copy new string into temp buffer */
			tmpstr = g_strdup(lastkana);
			g_strlcpy(cn->preedit_insert, tmpstr, BUFFERSIZE);
			g_free(tmpstr);
		}
		g_free(kanastr);

		/* update preedit from the wnn buffer */
    im_ja_wnn_update_preedit(cn);

		showBuffers(cnv_buf, "Insert Char");
    return TRUE;
  } 

  /* Unconvert current bunsetsu on ESC */
  if (key->keyval == GDK_Escape) {
    if (jcIsConverted(cnv_buf, cnv_buf->curClause) == 1) {
      jcUnconvert(cnv_buf);
			jcBottom(cnv_buf);
    }
    else return FALSE; /* This will kill the preedit if not converted */
    im_ja_wnn_update_preedit(cn);
    return TRUE;
  }
  
  return FALSE; /* We didn't handle the keypress */
}

void im_ja_wnn_select_candidate(IMJAContext *context, int selected) {
  jcSelect(context->wnn_stat.conv_buf, selected);
}

