/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 *
 */

#include <config.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <math.h>
#include <string.h>

#include "im-ja.h"
#include "symbols.h"
#include "error.h"
#include "nls.h"
#include "common.h"

static gint convert_line(gunichar *result, gchar **strg, gchar *line, gint length) {
  /* unichar is not changed if the input is not a 4 byte unichar string*/
  gint i;
  gunichar unichar = 0;

  if (length < 2) return 0;

  if (line[0] == '$') {
    *strg = g_new0(gchar, length - 1);
    g_strlcpy(*strg, line + 1, length - 1);
    return 1;
  }
  if (line[0] == '.' && line[1] == '.') return 2;
  if (length < 5) return 0;
  for (i = 0; i < 4; i++) {
    gint val = g_ascii_xdigit_value(line[i]);
    if (val == -1) return 0;
    unichar += val << 4 * (3 - i);
  }
  *strg = g_new0(gchar, 5);
  g_unichar_to_utf8(unichar, *strg);
  *result = unichar;
  return 1;

}

static GSList *get_symbol_table_data() {

  GIOChannel *symbol_table;
  GError *error = NULL;
  static GSList *symbols = NULL;

	if (symbols != NULL) {
		IM_JA_DEBUG("returning symbols from memory\n");
		return symbols;
	}

  symbol_table = g_io_channel_new_file(IM_JA_DATADIR"/im-ja-symbols.txt", "r", &error);
  
  if (symbol_table != NULL) {
    GIOStatus status;
    gchar *line;
    gsize length;
    gsize newline_pos;
    GSList *symbol_group = NULL;
    gchar *utf8_strg;
    gint conv_result = 0, last_result = 0;
    gunichar unichar = 0, last_unichar, i;
    
    while ((status = g_io_channel_read_line(symbol_table, &line, &length,
																						&newline_pos, &error)) == G_IO_STATUS_NORMAL) {

      /* Skip comments and empty lines */
      if ((line[0] == '#') || (line[0] == '\n')) continue;
      else {
				if ((line[0] == '%') || (line[0] == ' ')) { /* process symbol group */
					gchar *group_name;
					symbol_group = NULL;
					group_name = g_new0(gchar, length);
					g_strlcpy(group_name, line + 2, length - 2); 
					symbol_group = g_slist_append(symbol_group, group_name);
					symbols = g_slist_append(symbols, symbol_group);
					last_result = 0;
				}
				else { /* process group entries */
					last_result = conv_result;
					last_unichar = unichar;
					conv_result = convert_line(&unichar, &utf8_strg, line, length);
					if (conv_result == 1) {
						if (last_result == 2) { /* We need to generate a series of entries (2 == Range) */
							for (i = last_unichar + 1; i <= unichar; i++) {
								utf8_strg = g_new0(gchar, 5);
								g_unichar_to_utf8(i, utf8_strg);
								symbol_group = g_slist_append(symbol_group, utf8_strg);
							}
						}
						else { /* Add only one entry */
							symbol_group = g_slist_append(symbol_group, utf8_strg);
						}
					}
				}
      }
      g_free(line);
    }
    g_io_channel_shutdown(symbol_table, FALSE, &error);
  }

  return symbols;
}

static gboolean key_press_cb(GtkWidget *window, GdkEventKey *event, gpointer data) {
	if (event->keyval == GDK_Escape) {
		gtk_widget_destroy(window);
		return TRUE;
	}
	return FALSE;
}

static void symbol_pressed_cb(GtkWidget *button, IMJAContext *cn) {
	GtkWidget *window;
	gchar *symbol;

	window = g_object_get_data(G_OBJECT(button), "im-ja-symbol-window");
	symbol = g_object_get_data(G_OBJECT(button), "im-ja-utf8-symbol");
	gtk_widget_destroy(window);
	
	/* symbol = gtk_button_get_label(GTK_BUTTON(button)); */
	im_ja_input_utf8(cn, symbol);
}

/* This makes the secondary table with the symbols */
static void show_symbols(GtkWidget *button, IMJAContext *cn) {
	GSList *symbols_ptr;
	GSList *symbols;
  gint rows = 0;
  gint cols = 0;
  gint col, row;
  gint total = 0;
	gint cnt = 0;
	gdouble root;
  GtkWidget *window;
  GtkWidget *grp_window;
  GtkWidget *symbol_table;

	symbols = symbols_ptr = g_object_get_data(G_OBJECT(button), "im-ja-symbols");
	grp_window = g_object_get_data(G_OBJECT(button), "im-ja-symbol-grp-window");
	gtk_widget_destroy(grp_window);
	
	
  /* count the elements in the table */
  while (symbols_ptr != NULL) {
    total++;
    symbols_ptr = g_slist_next(symbols_ptr);
  }
	total--; /* The group name doesn't count; */

  /* Calculate table size; */
  root = sqrt((double) total);
  rows = cols = (gint) root;
	if (((gdouble) rows) < root) {
    rows++;
    if (rows * cols < total) cols++;
  }
  IM_JA_DEBUG("Total: %d, Rows: %d, cols: %d\n", total, rows, cols);

  /* Create the table */
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), _("Symbols"));
	gtk_window_set_modal(GTK_WINDOW(window), TRUE);

	/* FIXME */
#ifdef IMJA_TARGET_GTK
  if (GTK_IS_WINDOW(cn->toplevel_gtk)) {
		gtk_window_set_transient_for(GTK_WINDOW(window), GTK_WINDOW(cn->toplevel_gtk));
	}
#endif
	g_signal_connect(GTK_OBJECT(window), "key_press_event", G_CALLBACK(key_press_cb), NULL);

  symbol_table = gtk_table_new(rows, cols, TRUE); 
  gtk_container_add(GTK_CONTAINER(window), symbol_table);

	symbols_ptr = symbols;
	symbols_ptr = g_slist_next(symbols_ptr); /* skip the group name */

	col = row = 0;
	while (symbols_ptr != NULL) {
		while (col < cols) {
			if (symbols_ptr->data != NULL) {
				GtkWidget *tmpwidget = gtk_button_new_with_label(symbols_ptr->data);
				g_object_set_data(G_OBJECT(tmpwidget), "im-ja-symbol-window", (gpointer) window);
				g_object_set_data(G_OBJECT(tmpwidget), "im-ja-utf8-symbol", symbols_ptr->data);
				g_signal_connect(G_OBJECT(tmpwidget), "clicked", G_CALLBACK(symbol_pressed_cb), cn);

				gtk_table_attach(GTK_TABLE(symbol_table), tmpwidget, col, col + 1, row, row + 1,
												 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
												 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
			
				symbols_ptr = g_slist_next(symbols_ptr);
				cnt++;
				col++;
				if (symbols_ptr == NULL) break;
			}
		}
		col = 0;
		row++;
	}

  gtk_widget_show_all(window);
	
}

/* This creates the main table with the symbol groups */
void im_ja_symbol_table_show(IMJAContext *cn) {
  GSList *symbols_ptr = NULL;
  GSList *symbols = NULL;
  GSList *symbol_group_ptr = NULL;
  gint rows = 0;
  gint cols = 0;
  gint col, row;
  gint total = 0;
	gint cnt = 0;
  GtkWidget *window;
  GtkWidget *symbol_grp_table;

  IM_JA_DEBUG("im_ja_symbol_table_show()\n");

  symbols = symbols_ptr = get_symbol_table_data();

  /* count the elements in the table */
  while (symbols_ptr != NULL) {
    total++;
    symbols_ptr = g_slist_next(symbols_ptr);
  }
  /* Calculate table size; */
	/*
	//	if (total > 4) cols = 3;
	//	else
	*/
	cols = 2;
	rows = total / cols;

  /* Create the table */
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), _("Symbol groups"));
  gtk_window_set_modal(GTK_WINDOW(window), TRUE);

	/* FIXME */
#ifdef IMJA_TARGET_GTK
  if (GTK_IS_WINDOW(cn->toplevel_gtk)) {
		gtk_window_set_transient_for(GTK_WINDOW(window), GTK_WINDOW(cn->toplevel_gtk));
	}
#endif

	g_signal_connect(GTK_OBJECT(window), "key_press_event", G_CALLBACK(key_press_cb), NULL);

  symbol_grp_table = gtk_table_new(rows, cols, TRUE); 
  gtk_container_add(GTK_CONTAINER(window), symbol_grp_table);

  symbols_ptr = symbols;
	col = row = 0;
	while (symbols_ptr != NULL) {
		while (col < cols) {
			symbol_group_ptr = symbols_ptr->data;
			if (symbol_group_ptr != NULL) {
				GtkWidget *tmpwidget = gtk_button_new_with_label(symbol_group_ptr->data);
				g_object_set_data(G_OBJECT(tmpwidget), "im-ja-symbols", (gpointer) symbol_group_ptr);
				g_object_set_data(G_OBJECT(tmpwidget), "im-ja-symbol-grp-window", (gpointer) window);
				g_signal_connect(G_OBJECT(tmpwidget), "clicked", G_CALLBACK(show_symbols), cn);

				gtk_table_attach(GTK_TABLE(symbol_grp_table), tmpwidget, col, col + 1, row, row + 1,
												 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
												 (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
				symbols_ptr = g_slist_next(symbols_ptr);
				cnt++;
				col++;
				if (symbols_ptr == NULL) break;
			}
		}
		col = 0;
		row++;
	}

  gtk_widget_show_all(window);

}

static void translate_unicode_entry_cb(GtkWidget *entry, IMJAContext *cn) {
	gchar *code_strg, *utf8_strg;
	gint length;
	gunichar unichar;
	GtkWidget *window;
	gboolean input_ok = TRUE;
	gint i;

	code_strg = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
	length = strlen(code_strg);

	window = g_object_get_data(G_OBJECT(entry), "im-ja-unicode-entry-window");
	gtk_widget_destroy(window);
	
	if (length == 0) return;

  if (length != 4) input_ok = FALSE;
	else {
		unichar = 0;
		for (i = 0; i < 4; i++) {
			gint val = g_ascii_xdigit_value(code_strg[i]);
			if (val == -1) {
				input_ok = FALSE;
				break;
			}
			unichar += val << 4 * (3 - i);
		}
	}

	if (input_ok == TRUE) {
		if (g_unichar_validate(unichar) == FALSE) {
			im_ja_print_error(_("Invalid unicode character: %s"), code_strg);
		}
		else {
			utf8_strg = g_new0(gchar, 5);
			g_unichar_to_utf8(unichar, utf8_strg);
			im_ja_input_utf8(cn, utf8_strg);
		}
	}
	else {
		im_ja_print_error(_("Invalid 4 byte HEX input: %s"), code_strg);
	}
	g_free(code_strg);
}

static void translate_jiscode_entry_cb(GtkWidget *entry, IMJAContext *cn) {
	gchar *code_strg, *utf8_strg, *euc_str;
	gint length;
	GtkWidget *window;
	gboolean input_ok = TRUE;
	gint i;
	gint val = 0;
	wchar src_str[2];
	int chr = 0;

	code_strg = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
	length = strlen(code_strg);
	
	window = g_object_get_data(G_OBJECT(entry), "im-ja-jiscode-entry-window");
	gtk_widget_destroy(window);
	
	
	if (length == 0) return;
	
  if (length != 4) input_ok = FALSE;
	else {
		val = 0;
		chr = 0;
		for (i = 0; i < 4; i++) {
			gint val = g_ascii_xdigit_value(code_strg[i]);
			if (val == -1) {
				input_ok = FALSE;
				break;
			}
			chr = (chr << 4) + val;
		}
	}
	if (chr != -1) input_ok = TRUE;

	src_str[0] = chr | 0x8080;
	src_str[1] = 0;

	euc_str = wc2euc(src_str, strlen((gchar*)src_str));
	utf8_strg = euc2utf8(euc_str);
	g_free(euc_str);

	if (input_ok == TRUE) {
		const gchar *end;
		if (g_utf8_validate(utf8_strg, -1, &end) == FALSE) {
			im_ja_print_error(_("Invalid unicode character: %s"), code_strg);
		}
		else {
			im_ja_input_utf8(cn, utf8_strg);
		}
	}
	else {
		im_ja_print_error(_("Invalid 4 byte HEX input: %s"), code_strg);
	}
	g_free(code_strg);
}

void im_ja_unicode_entry_show(IMJAContext *cn) {
  GtkWidget *window;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *entry;

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), _("Unicode entry"));
  gtk_window_set_modal(GTK_WINDOW(window), TRUE);

	/* FIXME */
#ifdef IMJA_TARGET_GTK
  if (GTK_IS_WINDOW(cn->toplevel_gtk)) {
		gtk_window_set_transient_for(GTK_WINDOW(window), GTK_WINDOW(cn->toplevel_gtk));
	}
#endif
	g_signal_connect(GTK_OBJECT(window), "key_press_event", G_CALLBACK(key_press_cb), NULL);
	
	hbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(window), hbox);

	label = gtk_label_new(_("4 Byte HEX UniCode:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

	entry = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(entry), 4);
	gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 0);
	g_object_set_data(G_OBJECT(entry), "im-ja-unicode-entry-window", (gpointer) window);
	g_signal_connect(GTK_OBJECT(entry), "activate", G_CALLBACK(translate_unicode_entry_cb), cn);

	gtk_widget_show_all(window);

}

void im_ja_jiscode_entry_show(IMJAContext *cn) {
  GtkWidget *window;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *entry;

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), _("JIS Code entry"));
  gtk_window_set_modal(GTK_WINDOW(window), TRUE);

	/* FIXME */
#ifdef IMJA_TARGET_GTK
  if (GTK_IS_WINDOW(cn->toplevel_gtk)) {
		gtk_window_set_transient_for(GTK_WINDOW(window), GTK_WINDOW(cn->toplevel_gtk));
	}
#endif
	g_signal_connect(GTK_OBJECT(window), "key_press_event", G_CALLBACK(key_press_cb), NULL);
	
	hbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(window), hbox);

	label = gtk_label_new(_("4 Byte JIS Code:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

	entry = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(entry), 4);
	gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 0);
	g_object_set_data(G_OBJECT(entry), "im-ja-jiscode-entry-window", (gpointer) window);
	g_signal_connect(GTK_OBJECT(entry), "activate", G_CALLBACK(translate_jiscode_entry_cb), cn);

	gtk_widget_show_all(window);

}
