/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method Module for GTK-2.0
 *
 * Copyright (C) 2004 Botond Botyanszki <boti@rocketmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 */

#include <config.h>

#include <glib.h>
#include <string.h>

#include "../error.h"
#include "../common.h"
#include "../romakana.h"

#include "skklib.h"
#include "skkconv.h"


void skkconv_destroy_skkclause(SKKClause *clause, gpointer data) {
	IM_JA_DEBUG("skkconv_destroy_skkcand()\n");
	if (clause == NULL) IM_JA_DEBUG("**ERROR** can't destroy NULL clause!\n");
	if (clause->cand != NULL) freeCand(clause->cand);
	g_free(clause);
}

void skkconv_free_clauselist(SKKContext *skkctx) {
	g_list_foreach(skkctx->clauselist, (GFunc) skkconv_destroy_skkclause, NULL);
	g_list_free(skkctx->clauselist);
	skkctx->clauselist = NULL;
  skkctx->curr_clause = NULL;
}

void skkconv_reset_ctx(SKKContext *skkctx) {
	memset(skkctx->kana_buf, 0, BUFFERSIZE);
	skkctx->cursor_pos = 0;
	skkctx->conv_state = SKKCONV_UNCONVERTED;
	skkconv_free_clauselist(skkctx);
}

void skkconv_select_candidate(SKKContext *skkctx, gint direction) {
	SKKClause *clause;
	CandList cand;

	IM_JA_DEBUG("skkconv_select_candidate(%d)\n", direction);

	clause = (SKKClause *) skkctx->curr_clause->data;
	cand = clause->selected_cand;
	if (cand != NULL) {
		if (direction == SKKCONV_NEXT) {
			if (cand->nextcand == NULL) clause->selected_cand = clause->cand;
			else clause->selected_cand = clause->selected_cand->nextcand;
		}
		else { /* PREV */
			if (cand->prevcand == NULL) {
				while (cand->nextcand != NULL) cand = cand->nextcand;
				clause->selected_cand = cand;
			}
			else clause->selected_cand = clause->selected_cand->prevcand;
		}
	}
}

void skkconv_fix_selected_candidate(SKKContext *skkctx, gint selected) {
	SKKClause *clause;
	CandList cand;
	gint candnum = 0;

	IM_JA_DEBUG("skkconv_fix_selected_candidate(%d)\n", selected);
	/* selectCand(*first,cand) */

	clause = (SKKClause *) skkctx->curr_clause->data;
	cand = clause->cand;
	while (cand != NULL) {
		if (selected == candnum) {
			clause->selected_cand = cand;
			IM_JA_DEBUG("Select: %s\n", clause->selected_cand->candword);
			break;
		}
		candnum++;
		cand = cand->nextcand;
	}
}

void skkconv_unconvert_all(SKKContext *skkctx) {
	IM_JA_DEBUG("skkconv_unconvert_all()\n");

	skkctx->conv_state = SKKCONV_UNCONVERTED;
	skkconv_free_clauselist(skkctx);
	skkctx->cursor_pos = strlen(skkctx->kana_buf);
}

void skkconv_unconvert_clause(SKKClause *clause) {
	IM_JA_DEBUG("skkconv_unconvert_clause()\n");
	if (clause->conv_state == SKKCONV_UNCONVERTED) return;
	clause->conv_state = SKKCONV_UNCONVERTED;
	clause->selected_cand = clause->cand;
}

void skkconv_unconvert_current_clause(SKKContext *skkctx) {
	SKKClause *clause;

	clause = (SKKClause *) skkctx->curr_clause->data;
	skkconv_unconvert_clause(clause);
}

void skkconv_convert_all(SKKContext *skkctx) {
	SKKClause *clause;
	gchar *start, *end;

	IM_JA_DEBUG("skkconv_convert_all()\n");

	start = skkctx->kana_buf;
	end = start + strlen(start);

	while (TRUE) {
		clause = skkconv_convert_clause(start, end, TRUE);
		if (clause == NULL) break;
		else {
			skkctx->clauselist = g_list_append(skkctx->clauselist, clause);
			skkctx->conv_state = SKKCONV_CONVERTED;
			if (clause->kana_start < skkctx->kana_buf + strlen(skkctx->kana_buf)) {
				start = clause->kana_end;
			}
			else break;
		}
	}
	skkctx->curr_clause = skkctx->clauselist;
	IM_JA_DEBUG("skkconv_convert_all done\n");
}

SKKClause *skkconv_convert_clause(gchar *conversion_start, gchar *conversion_end, gboolean shrink) {
	gchar *eucstrg, *utfstrg, *kana, *kata;
	gchar *conv_start, *conv_end;
	CandList curr_cand, cand = NULL, cand_kana = NULL;
	CandList *first_cand_entry;
	SKKClause *skkclause = NULL;
	gboolean conversion_done = FALSE;
	conv_start = conversion_start;
	conv_end = conversion_end;

	/* IM_JA_DEBUG("skkconv_convert_clause()\n"); */

	if (conv_start == conv_end) return NULL;

	while (conversion_done != TRUE) {
		utfstrg = g_strdup(conv_start);
		utfstrg[conv_end - conv_start] = 0;

		eucstrg = utf82euc(utfstrg);
		g_free(utfstrg);

		curr_cand = getCandFromServer(eucstrg);
		if (curr_cand != NULL) {
			curr_cand = searchOkuri(curr_cand, eucstrg, &first_cand_entry);
			if (curr_cand->okuri != NULL) {
				IM_JA_DEBUG("Okuri: %s\n", curr_cand->okuri->candword);
			}
			skkclause = g_new0(SKKClause, 1);
			skkclause->kana_start = conv_start;
			skkclause->kana_end = conv_end;
			skkclause->cand = curr_cand;
			skkclause->selected_cand = curr_cand;
			skkclause->conv_state = SKKCONV_CONVERTED;
			/*
			while (curr_cand != NULL) {
				if (curr_cand->okuri != NULL) {
					okuri_cand = curr_cand->okuri;
					while (okuri_cand != NULL) {
						IM_JA_DEBUG("Okuri: %s\n", okuri_cand->candword);
						okuri_cand = okuri_cand->nextcand;
					}
				}
				else {
					IM_JA_DEBUG("Cand: %s\n", curr_cand->candword);
				}
				curr_cand = curr_cand->nextcand;
			}
			*/
			conversion_done = TRUE;
		}
		else {
			/* IM_JA_DEBUG("couldn't convert\n"); */
			if (shrink == TRUE) {
				/* IM_JA_DEBUG("couldn't convert [zero length]\n"); */
				if (conv_start == g_utf8_prev_char(conv_end)) { /* one char length */
					conversion_done = TRUE;
				}
				else conv_end = g_utf8_prev_char(conv_end); 
			}
			else {
				conv_start = conversion_start;
				conv_end = conversion_end;
				conversion_done = TRUE;
			}
			if (conv_end == conv_start) {
				/* IM_JA_DEBUG("couldn't convert [zero length]\n"); */
				conversion_done = TRUE;
				skkclause = NULL;
			}
			else { /* create an unconverted clause */
				/* IM_JA_DEBUG("couldn't convert\n"); */
				skkclause = g_new0(SKKClause, 1);
				skkclause->kana_start = conv_start;
				skkclause->kana_end = conv_end;
				skkclause->cand = NULL;
				skkclause->selected_cand = NULL;
				skkclause->conv_state = SKKCONV_UNCONVERTED;
			}
		}
		g_free(eucstrg);
	}

	if (skkclause != NULL) cand = skkclause->cand;
	kana = g_strdup(skkclause->kana_start);
	kana[skkclause->kana_end - skkclause->kana_start] = 0;
	/* Add katakana candidate */
	kata = hira2kata(kana);
	eucstrg = utf82euc(kata);
	cand_kana = _NEW2(CandList, strlen(eucstrg) + 1);
	g_strlcpy(cand_kana->candword, eucstrg, strlen(eucstrg) + 1);
	g_free(kata);
	g_free(eucstrg);
	if (cand != NULL) cand->prevcand = cand_kana;
	cand_kana->nextcand = cand;
	cand_kana->okuri = NULL;
	if (cand != NULL)	cand_kana->dicitem = cand->dicitem;
	cand = cand_kana;

	/* Add hiragana candidate */
	eucstrg = utf82euc(kana);
	cand_kana = _NEW2(CandList, strlen(eucstrg) + 1);
	g_strlcpy(cand_kana->candword, eucstrg, strlen(eucstrg) + 1);
	cand->prevcand = cand_kana;
	cand_kana->nextcand = cand;
	cand_kana->prevcand = NULL;
	cand_kana->okuri = NULL;
	cand_kana->dicitem = cand->dicitem;
	cand = cand_kana;
	g_free(kana);
	g_free(eucstrg);
	skkclause->cand = cand;
	if (skkclause->selected_cand == NULL) skkclause->selected_cand = cand;
	return skkclause;
}

void skkconv_insert_string(SKKContext *skkctx, gchar *strg) {
	gchar *tmpstrg;

	IM_JA_DEBUG("skkconv_insert_string: \"%s\"\n", strg);
	tmpstrg = g_strdup(skkctx->kana_buf + skkctx->cursor_pos);
	skkctx->kana_buf[skkctx->cursor_pos] = 0;
	g_strlcat(skkctx->kana_buf, strg, BUFFERSIZE);
	g_strlcat(skkctx->kana_buf, tmpstrg, BUFFERSIZE);
	g_free(tmpstrg);

}

void skkconv_roma2kana(SKKContext *skkctx, gint input_method) {
	gchar *tmpstr, *firstpart, *secondpart;

	firstpart = g_strdup(skkctx->kana_buf);
	secondpart = g_strdup(skkctx->kana_buf + skkctx->cursor_pos + 1);
	firstpart[skkctx->cursor_pos + 1] = 0;
	tmpstr = roma2kana(firstpart, input_method);
	g_free(firstpart);
	g_strlcpy(skkctx->kana_buf, tmpstr, BUFFERSIZE);
	g_strlcat(skkctx->kana_buf, secondpart, BUFFERSIZE);
	g_free(tmpstr);
	g_free(secondpart);
}

void skkconv_delete_char(SKKContext *skkctx, gint direction) {
	gchar *tmpstrg;

	if (direction == SKKCONV_BACKWARD) { /* Backspace */
		if (skkctx->cursor_pos == 0) return;
		tmpstrg = g_strdup(skkctx->kana_buf + skkctx->cursor_pos);
		skkctx->cursor_pos = g_utf8_prev_char(skkctx->kana_buf + skkctx->cursor_pos) - skkctx->kana_buf;
		*(skkctx->kana_buf + skkctx->cursor_pos) = 0;
		g_strlcat(skkctx->kana_buf, tmpstrg, BUFFERSIZE);
		g_free(tmpstrg);
	}
	else { /* Delete */
		if ((size_t) (skkctx->cursor_pos) == strlen(skkctx->kana_buf)) return;
		tmpstrg = g_strdup(g_utf8_next_char(skkctx->kana_buf + skkctx->cursor_pos));
		*(skkctx->kana_buf + skkctx->cursor_pos) = 0;
		g_strlcat(skkctx->kana_buf, tmpstrg, BUFFERSIZE);
		g_free(tmpstrg);
	}
}

void skkconv_select_clause(SKKContext *skkctx, gint direction) {
	GList *curr_clause;

	if (skkctx->conv_state == SKKCONV_UNCONVERTED) return;
	if (direction == SKKCONV_NEXT) {
		curr_clause = g_list_next(skkctx->curr_clause);
		if (curr_clause != NULL) skkctx->curr_clause = curr_clause;
	}
	else {
		curr_clause = g_list_previous(skkctx->curr_clause);
		if (curr_clause != NULL) skkctx->curr_clause = curr_clause;
	}
}

void skkconv_move_cursor(SKKContext *skkctx, gint direction) {
	gchar *buf = skkctx->kana_buf;

	IM_JA_DEBUG("skkconv_move_cursor(%d)\n", direction);
	if (direction == SKKCONV_FORWARD) {
		if ((size_t) (skkctx->cursor_pos) == strlen(buf)) {
			return;
		}
		else {
			skkctx->cursor_pos = g_utf8_next_char(buf + skkctx->cursor_pos) - buf;
		}
	}
	else {
		if (skkctx->cursor_pos == 0) {
			return;
		}
		else {
			skkctx->cursor_pos = g_utf8_prev_char(buf + skkctx->cursor_pos) - buf;
		}
	}				
}

void skkconv_resize_clause(SKKContext *skkctx, gint direction) {
	GList *clause_list;
	GList *curr_clause, *next_clause;
	SKKClause *curr_new = NULL, *next_new = NULL;
	gint len;

	IM_JA_DEBUG("skkconv_resize_clause(%d)\n", direction);

	skkconv_unconvert_current_clause(skkctx);
	clause_list = skkctx->clauselist;

	if (direction == SKKCONV_FORWARD) { /* == EXPAND */
		IM_JA_DEBUG("EXPAND!\n");
		/* can't expand if there is no next clause */
		if (g_list_next(skkctx->curr_clause) == NULL) {
			IM_JA_DEBUG("NO NEXT CLAUSE!\n");
			return;
		}
		curr_clause = skkctx->curr_clause;
		next_clause = g_list_next(skkctx->curr_clause);
		skkconv_unconvert_clause(next_clause->data);
		len = ((SKKClause *)next_clause->data)->kana_end - ((SKKClause *)next_clause->data)->kana_start;
		((SKKClause *)curr_clause->data)->kana_end = g_utf8_next_char(((SKKClause *)curr_clause->data)->kana_end);
		IM_JA_DEBUG("NEXT CLAUSE LENGTH: %d\n", (int) g_utf8_strlen(((SKKClause *)next_clause->data)->kana_start, len));
		if (g_utf8_strlen(((SKKClause *)next_clause->data)->kana_start, len) == 1) {
			/* Delete clause if it's one char only (merge into current) */
			IM_JA_DEBUG("ONE CHAR CLAUSE: deleting\n");
			skkconv_destroy_skkclause(((SKKClause *)next_clause->data), NULL);
			clause_list = g_list_delete_link(clause_list, next_clause);
			next_clause = NULL;
			next_new = NULL;
		}
		else {
			/* create a new next clause */
			((SKKClause *)next_clause->data)->kana_start = g_utf8_next_char(((SKKClause *)next_clause->data)->kana_start);			
			next_new = skkconv_convert_clause(((SKKClause *)next_clause->data)->kana_start,
																				((SKKClause *)next_clause->data)->kana_end, FALSE);
		}
		curr_new = skkconv_convert_clause(((SKKClause *)curr_clause->data)->kana_start,
																			((SKKClause *)curr_clause->data)->kana_end, FALSE);

		skkconv_destroy_skkclause(((SKKClause *)curr_clause->data), NULL);
		curr_clause->data = curr_new;
		if (next_new != NULL) {
			skkconv_destroy_skkclause(((SKKClause *)next_clause->data), NULL);
			next_clause->data = next_new; 
		}
	}
	else { /* BACKWARD == SHRINK */
		IM_JA_DEBUG("SHRINK!\n");
		curr_clause = skkctx->curr_clause;
		/* one character width, don't do anything */
		if (g_utf8_next_char(((SKKClause *)curr_clause->data)->kana_start) == ((SKKClause *)curr_clause->data)->kana_end) return;

		next_clause = g_list_next(skkctx->curr_clause);
		if (next_clause == NULL) {
			/* create a new next clause */
			next_new = skkconv_convert_clause(g_utf8_prev_char(((SKKClause *)curr_clause->data)->kana_end),
																				((SKKClause *)curr_clause->data)->kana_end, FALSE);
			clause_list = g_list_append(clause_list, next_new);
			((SKKClause *)curr_clause->data)->kana_end = g_utf8_prev_char(((SKKClause *)curr_clause->data)->kana_end);
			curr_new = skkconv_convert_clause(((SKKClause *)curr_clause->data)->kana_start,
																				((SKKClause *)curr_clause->data)->kana_end, FALSE);
			skkconv_destroy_skkclause(((SKKClause *)curr_clause->data), NULL);
			curr_clause->data = curr_new;
		}
		else {
			skkconv_unconvert_clause(((SKKClause *)next_clause->data));
			((SKKClause *)next_clause->data)->kana_start = g_utf8_prev_char(((SKKClause *)next_clause->data)->kana_start);			
			next_new = skkconv_convert_clause(((SKKClause *)next_clause->data)->kana_start,
																				((SKKClause *)next_clause->data)->kana_end, FALSE);
			skkconv_destroy_skkclause(((SKKClause *)next_clause->data), NULL);
			next_clause->data = next_new;

			((SKKClause *)curr_clause->data)->kana_end = g_utf8_prev_char(((SKKClause *)curr_clause->data)->kana_end);
			curr_new = skkconv_convert_clause(((SKKClause *)curr_clause->data)->kana_start,
																				((SKKClause *)curr_clause->data)->kana_end, FALSE);
			skkconv_destroy_skkclause(((SKKClause *)curr_clause->data), NULL);
			curr_clause->data = curr_new;
		}
	}
}
