/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method Module for GTK-2.0
 *
 * Copyright (C) 2004 Botond Botyanszki <boti@rocketmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 */

#include <config.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "../nls.h"
#include "../im-ja.h"
#include "../im-ja-impl.h"
#include "../error.h"
#include "../common.h"
#include "../conf.h"
#include "../romakana.h"
#include "../candwin.h"

#include "skk.h"
#include "skklib.h"
#include "skkconv.h"

/* * * * * * * * * * * */
extern IMJAConfig cfg;
/* * * * * * * * * * * */

void im_ja_skk_shutdown(IMJAContext *cn) {
	closeSKKserv();
}

gboolean im_ja_skk_init(IMJAContext *cn) {
	IM_JA_DEBUG("im_ja_skk_init\n");

	if (openSKKserv() == -1) return FALSE;
	cn->skk_ctx = g_new0(SKKContext, 1);
	skkconv_reset_ctx(cn->skk_ctx);

	cn->im_ja_conv_engine_reset_context = im_ja_skk_reset;
	cn->im_ja_conv_engine_filter_keypress = im_ja_skk_filter_keypress;
	cn->im_ja_conv_engine_shutdown = im_ja_skk_shutdown;
	cn->im_ja_conv_engine_select_candidate = im_ja_skk_select_candidate;
	cn->im_ja_conv_engine_update_preedit = im_ja_skk_update_preedit;

	return TRUE;
}

void im_ja_skk_unconvert(IMJAContext *cn) {
	SKKClause *clause;
	if (cn->skk_ctx->curr_clause == NULL) return;

	clause = (SKKClause *)cn->skk_ctx->curr_clause->data;
	if (clause == NULL) return;
	if (clause->conv_state == SKKCONV_CONVERTED) {
		skkconv_unconvert_current_clause(cn->skk_ctx);
	}
	else {
		skkconv_unconvert_all(cn->skk_ctx);
	}
}

void im_ja_skk_update_preedit(IMJAContext *cn) {
	GList *clauselist;
	gchar *tmputf;
	SKKClause *skkclause;

	cn->cursor_char_pos = 0;
	cn->preedit_reverse_start = 0;
	cn->preedit_reverse_end = 0;
	*cn->preedit_buf = 0;

	IM_JA_DEBUG("im_ja_skk_update_preedit(): %d\n", cn->skk_ctx->conv_state);
	if (cn->skk_ctx->conv_state == SKKCONV_UNCONVERTED) {
		g_strlcpy(cn->preedit_buf, cn->skk_ctx->kana_buf, BUFFERSIZE);
	}
	else {
		memset(cn->preedit_buf, 0, BUFFERSIZE);
		clauselist = cn->skk_ctx->clauselist;
		while (clauselist != NULL) {
			if (clauselist->data != NULL) {
				skkclause = (SKKClause *)clauselist->data;

				if (skkclause->conv_state == SKKCONV_CONVERTED) {
					if ((skkclause->selected_cand != NULL) && (skkclause->selected_cand->candword != NULL)) {
						tmputf = euc2utf8(skkclause->selected_cand->candword);
						if (cn->skk_ctx->curr_clause == clauselist) {
							/* if (cn->preedit_reverse_start != strlen(cn->preedit_buf)) */
							cn->update_candwin_pos = TRUE;
							cn->skk_ctx->cursor_pos = cn->preedit_reverse_start = strlen(cn->preedit_buf);
						}
						g_strlcat(cn->preedit_buf, tmputf, BUFFERSIZE);
						if (cn->skk_ctx->curr_clause == clauselist) {
							cn->preedit_reverse_end = strlen(cn->preedit_buf);
						}
						g_free(tmputf);
					}
				}

				else { /* Unconverted clause */
					/* IM_JA_DEBUG("UNCONVERTED CLAUSE!\n"); */
					tmputf = g_strdup(skkclause->kana_start);
					tmputf[skkclause->kana_end - skkclause->kana_start] = 0;
					if (cn->skk_ctx->curr_clause == clauselist) {
						cn->update_candwin_pos = TRUE;
						cn->skk_ctx->cursor_pos = cn->preedit_reverse_start = strlen(cn->preedit_buf);
					}
					g_strlcat(cn->preedit_buf, tmputf, BUFFERSIZE);
					if (cn->skk_ctx->curr_clause == clauselist) {
						cn->preedit_reverse_end = strlen(cn->preedit_buf);
					}
					g_free(tmputf);
				}
			}
			clauselist = g_list_next(clauselist);
		}
	}
	cn->cursor_char_pos = g_utf8_strlen(cn->preedit_buf, cn->skk_ctx->cursor_pos);
	/* IM_JA_DEBUG("CURSOR: %d\n", cn->skk_ctx->cursor_pos); */
	im_ja_preedit_changed(cn);
}

void im_ja_skk_reset(IMJAContext *cn) {
	IM_JA_DEBUG("im_ja_skk_reset\n");
	skkconv_reset_ctx(cn->skk_ctx);
}


void im_ja_skk_commit(IMJAContext *cn) {
	/* if (im_ja_wnn_convert_tailing_n(cn) == TRUE) im_ja_wnn_update_preedit(cn);*/
	im_ja_skk_reset(cn);
}

void im_ja_skk_convert(IMJAContext *cn) {
	IM_JA_DEBUG("im_ja_skk_convert()\n");
	if (cn->skk_ctx->conv_state == SKKCONV_UNCONVERTED) {
		skkconv_convert_all(cn->skk_ctx);
	}
	else {
		((SKKClause *)cn->skk_ctx->curr_clause->data)->conv_state = SKKCONV_CONVERTED;
	}
}

void im_ja_skk_show_candidates(IMJAContext *cn) {
	SKKClause *skkclause;
	CandList cand;
	gint candnum = 0, currcand = 0;
			
	if (cn->skk_ctx->curr_clause != NULL) {
		skkclause = (SKKClause *) cn->skk_ctx->curr_clause->data;
		cand = skkclause->cand;
		im_ja_free_candidate_list(cn);

		while (cand != NULL) {
			cn->candidate_list = g_list_append(cn->candidate_list, euc2utf8(cand->candword));
			if (skkclause->selected_cand == cand) currcand = candnum;
			candnum++;
			cand = cand->nextcand;
		}
		candidate_window_show(cn, currcand);
	}
	else IM_JA_DEBUG("NO CURRENT CANDIDATE!\n");
}

gboolean im_ja_skk_filter_keypress(IMJAContext *cn, GdkEventKey *key) {

	if (cn->input_method != IM_JA_HIRAGANA_INPUT) {
		return im_ja_kana_filter_keypress(cn, key);
	}
  
  if( key->type == GDK_KEY_RELEASE ) {
    return FALSE;
  };

	IM_JA_DEBUG("im_ja_skk_filter_keypress()\n");

  /* ENTER */
  if ((key->keyval == GDK_Return) || (ishotkey(key, COMMIT_PREEDIT, &cfg) == TRUE)) {
		if (strlen(cn->preedit_buf) > 0) { /* Commit preedit buffer */
			im_ja_commit(cn);
			return TRUE;
		}
		return FALSE;
  }

  /* BACKSPACE */
  if (key->keyval == GDK_BackSpace ) {
		if (strlen(cn->preedit_buf) > 0) { /* Delete preedit */
			if (cn->skk_ctx->conv_state == SKKCONV_CONVERTED) {
				im_ja_skk_unconvert(cn);
			}
			else {
				skkconv_delete_char(cn->skk_ctx, SKKCONV_BACKWARD);
			}
			im_ja_skk_update_preedit(cn);
			return TRUE;
		}
		return FALSE; /* Delete normal */
  }

  /* DELETE */
  if (key->keyval == GDK_Delete ) {
		if (strlen(cn->preedit_buf) > 0) { /* Delete preedit */
			if (cn->skk_ctx->conv_state == SKKCONV_CONVERTED) {
				im_ja_skk_unconvert(cn);
			}
			else {
				skkconv_delete_char(cn->skk_ctx, SKKCONV_FORWARD);
			}
			im_ja_skk_update_preedit(cn);
			return TRUE;
		}
    return FALSE; /* Delete normal */
  }
	
  /* SPACE */
  if (key->keyval == GDK_space) { 
    if (strlen(cn->preedit_buf) == 0) {
			g_strlcpy(cn->preedit_buf, " ", BUFFERSIZE);
			im_ja_commit(cn); /* Insert a space */
			return TRUE;         
    }
		if (cn->skk_ctx->conv_state == SKKCONV_UNCONVERTED) im_ja_skk_convert(cn);
		else {
			if (((SKKClause *)cn->skk_ctx->curr_clause->data)->conv_state == SKKCONV_UNCONVERTED) {
				im_ja_skk_convert(cn);
				skkconv_select_candidate(cn->skk_ctx, SKKCONV_NEXT);
			}
			else im_ja_skk_show_candidates(cn);
		}

		im_ja_skk_update_preedit(cn);
		return TRUE; 
  }

  /* Convert current bunsetsu, if converted show the next candidate */
  if (ishotkey(key, CONVERT, &cfg) == TRUE) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
		if (cn->skk_ctx->conv_state == SKKCONV_UNCONVERTED) im_ja_skk_convert(cn);
		else {
			if (((SKKClause *)cn->skk_ctx->curr_clause->data)->conv_state == SKKCONV_UNCONVERTED) {
				im_ja_skk_convert(cn);
			}
			skkconv_select_candidate(cn->skk_ctx, SKKCONV_NEXT);
		}
		im_ja_skk_update_preedit(cn);
    return TRUE;
  }

  /* Unconvert current bunsetsu */
  if (ishotkey(key, UNCONVERT, &cfg) == TRUE) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
		im_ja_skk_unconvert(cn);
		im_ja_skk_update_preedit(cn);
		return FALSE;
  }

  /* Show candidates, if not converted then convert also */
  if (ishotkey(key, LIST_CANDIDATES, &cfg) == TRUE) {
		if (cn->skk_ctx->conv_state == SKKCONV_UNCONVERTED) {		
			im_ja_skk_convert(cn);
			im_ja_skk_update_preedit(cn);
		}
		im_ja_skk_show_candidates(cn);
    return TRUE;
  }

  /* Expand current bunsetsu */
  if (ishotkey(key, EXPAND_BUNSETSU, &cfg) == TRUE) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
		if (cn->skk_ctx->conv_state == SKKCONV_CONVERTED) {
			skkconv_resize_clause(cn->skk_ctx, SKKCONV_FORWARD);
			im_ja_skk_update_preedit(cn);
			return TRUE; 
		}
  }

  /* Shrink current bunsetsu */
  if (ishotkey(key, SHRINK_BUNSETSU, &cfg) == TRUE) {
    if (strlen(cn->preedit_buf) == 0) return FALSE;
		if (cn->skk_ctx->conv_state == SKKCONV_CONVERTED) {
			skkconv_resize_clause(cn->skk_ctx, SKKCONV_BACKWARD);
			im_ja_skk_update_preedit(cn);
			return TRUE; 
		}
  }

  /* Select next bunsetsu */
  if (ishotkey(key, NEXT_BUNSETSU, &cfg) == TRUE) {
		if (strlen(cn->preedit_buf) == 0) return FALSE;
		if (cn->skk_ctx->conv_state == SKKCONV_CONVERTED) {
			skkconv_select_clause(cn->skk_ctx, SKKCONV_NEXT);
			im_ja_skk_update_preedit(cn);
			return TRUE; 
		}
  } 
 
  /* Select prev bunsetsu */
  if (ishotkey(key, PREV_BUNSETSU, &cfg) == TRUE) {
		if (strlen(cn->preedit_buf) == 0) return FALSE;
		if (cn->skk_ctx->conv_state == SKKCONV_CONVERTED) {
			skkconv_select_clause(cn->skk_ctx, SKKCONV_PREV);
			im_ja_skk_update_preedit(cn);
			return TRUE; 
		}
  }

  /* Select previous candidate */
  if (ishotkey(key, PREV_CANDIDATE, &cfg) == TRUE) {
		if (strlen(cn->preedit_buf) == 0) return FALSE;
		if (cn->skk_ctx->conv_state == SKKCONV_CONVERTED) {
			im_ja_skk_convert(cn);
			skkconv_select_candidate(cn->skk_ctx, SKKCONV_PREV);
			im_ja_skk_update_preedit(cn);
			return TRUE; 
		}
  }

  /* Select next candidate */
  if (ishotkey(key, NEXT_CANDIDATE, &cfg) == TRUE) {
		if (strlen(cn->preedit_buf) == 0) return FALSE;
		if (cn->skk_ctx->conv_state == SKKCONV_CONVERTED) {
			im_ja_skk_convert(cn);
			skkconv_select_candidate(cn->skk_ctx, SKKCONV_NEXT);
			im_ja_skk_update_preedit(cn);
			return TRUE; 
		}
  }

	/* Left arrow key: move cursor */
	if (key->keyval == GDK_Left) {
		if (strlen(cn->preedit_buf) == 0) return FALSE;
		if (cn->skk_ctx->conv_state == SKKCONV_UNCONVERTED) {
			skkconv_move_cursor(cn->skk_ctx, SKKCONV_BACKWARD);
		}
		im_ja_skk_update_preedit(cn);
		return TRUE;
	}

	/* Right arrow key: move cursor */
	if (key->keyval == GDK_Right) {
		if (strlen(cn->preedit_buf) == 0) return FALSE;
		if (cn->skk_ctx->conv_state == SKKCONV_UNCONVERTED) {
			skkconv_move_cursor(cn->skk_ctx, SKKCONV_FORWARD);
		}
		im_ja_skk_update_preedit(cn);
		return TRUE;
	}

  /* NORMAL CHAR KEYS */
	if (im_ja_is_printable_key(key) == TRUE) { 
    gchar utf8strg[7];
		gchar *tmpeuc;
		gint len;

		utf8strg[g_unichar_to_utf8(gdk_keyval_to_unicode(key->keyval), utf8strg)] = 0;
		tmpeuc = utf82euc(utf8strg);
		if ((unsigned int)tmpeuc[0] > 128) {
			g_free(tmpeuc);
			IM_JA_DEBUG("*ERROR* non-ascii input\n");
			im_ja_input_utf8(cn, utf8strg);
			return TRUE;
		}
		g_free(tmpeuc);

		if (cn->skk_ctx->conv_state == SKKCONV_CONVERTED) im_ja_commit(cn);

		len = strlen(cn->skk_ctx->kana_buf);
		skkconv_insert_string(cn->skk_ctx, utf8strg);
		skkconv_roma2kana(cn->skk_ctx, cn->input_method);
		cn->skk_ctx->cursor_pos += strlen(cn->skk_ctx->kana_buf) - len;

    im_ja_skk_update_preedit(cn);
		return TRUE;
  } 

  /* Unconvert current bunsetsu on ESC */
  if (key->keyval == GDK_Escape) {
		if (cn->skk_ctx->conv_state == SKKCONV_CONVERTED) {
			im_ja_skk_unconvert(cn);
		}
		else return FALSE; /* This will kill the preedit if not converted */
    im_ja_skk_update_preedit(cn);
    return TRUE;
  }
  
  return FALSE; /* We didn't handle the keypress */
}

void im_ja_skk_select_candidate(IMJAContext *cn, int selected) {
	IM_JA_DEBUG("im_ja_skk_select_candidate()\n");
	skkconv_fix_selected_candidate(cn->skk_ctx, selected);
}

