/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 *          
 */

#include <config.h>

#include <gtk/gtk.h>
#include <gtk/gtkimmodule.h>
#include <string.h>

#include "im-ja.h"
#include "im-ja-impl.h"
#include "common.h"
#include "error.h"
#include "conf.h"
#include "nls.h"
#include "preeditwin.h"
#include "preeditarea.h"
#include "actionmenu.h"

extern IMJAConfig cfg; 
GList *preedit_windows = NULL;

static void destroy_win(PreeditWin *preedit_win, gboolean remove_signal);


static void preedit_window_disable_focus_out(GtkWidget *widget, GdkEvent *event, IMJAContext* cn ) {
	cn->preedit_win->can_hide = FALSE;
}

static void preedit_window_press_cb(GtkWidget *button, GdkEventButton *event, IMJAContext* cn) {
	IM_JA_DEBUG("preedit_window_press_cb\n");
	if (event->button == 1) {
		im_ja_actionmenu_button_press_cb(button, event, &cn->preedit_win->menu, 
																		 IM_JA_STATUSWIN_MENU, cn);
	}
}

void preedit_window_show(IMJAContext* cn) {
	IM_JA_DEBUG("preedit_window_show\n");

	if ((cn->input_method == IM_JA_DIRECT_INPUT) || (cn->input_method == IM_JA_KANJIPAD_INPUT)) {
		preedit_window_hide(cn);
		return;
	}

#ifdef IMJA_TARGET_GTK
	if ((cn->toplevel_gtk != NULL) && GTK_WIDGET_HAS_GRAB(cn->toplevel_gtk) == TRUE) {
		IM_JA_DEBUG("GRABBED TOPLEVEL!\n");
	}

	/* FIXME: gtk bug */
	if (cn->client_gtk != NULL) {
		if (GTK_IS_ENTRY(cn->client_gtk) == TRUE) {
			if (gtk_editable_get_editable(GTK_EDITABLE(cn->client_gtk)) == FALSE) {
			IM_JA_DEBUG("non-editable GtkEntry.\n");
			return;
			}
		}
		if (GTK_IS_TEXT_VIEW(cn->client_gtk) == TRUE) {
			if (gtk_text_view_get_editable(GTK_TEXT_VIEW(cn->client_gtk)) == FALSE) {
				IM_JA_DEBUG("non-editable GtkTextView.\n");
				return;
			}
		}
	}
#endif

	/* cn->preedit_win = g_object_get_data(G_OBJECT(cn->toplevel_gdk), "im-ja-preedit-window"); */
	if (cn->preedit_win == NULL) {
		GtkWidget *border;
		GtkWidget *input_border;
		GtkWidget *hbox;
		/* IM_JA_DEBUG("preedit_window_create\n"); */
		cn->preedit_win = g_new0(PreeditWin, 1);
		cn->preedit_win->menu = NULL;

		/* g_object_set_data(G_OBJECT(cn->toplevel_gdk), "im-ja-preedit-window", cn->preedit_win); */
		preedit_windows = g_list_append(preedit_windows, cn->preedit_win);
		cn->preedit_win->can_hide = TRUE;
		cn->preedit_win->window = gtk_window_new(GTK_WINDOW_POPUP);
		im_ja_join_modal_window(cn, cn->preedit_win->window);
		/*gtk_window_set_decorated(GTK_WINDOW(cn->preedit_win->window), FALSE);*/
		gtk_window_set_title(GTK_WINDOW(cn->preedit_win->window), _("[preedit window]"));
		gtk_window_set_default_size(GTK_WINDOW(cn->preedit_win->window), 140, 25);
			
		cn->preedit_win->eventbox = gtk_event_box_new();
		gtk_container_add(GTK_CONTAINER(cn->preedit_win->window), cn->preedit_win->eventbox);
		gtk_widget_show(cn->preedit_win->eventbox);
		gtk_widget_realize(cn->preedit_win->eventbox);
		gdk_window_set_cursor(cn->preedit_win->eventbox->window, gdk_cursor_new(GDK_HAND2));

		/* If the preedit gets the focus, we should be still receiving keyboard input */
		g_signal_connect(G_OBJECT(cn->preedit_win->eventbox), "enter_notify_event",
										 G_CALLBACK(im_ja_grab_add), cn);
		g_signal_connect(G_OBJECT(cn->preedit_win->eventbox), "leave_notify_event",
										 G_CALLBACK(im_ja_grab_remove), cn);
		g_signal_connect(G_OBJECT(cn->preedit_win->eventbox), "button_press_event",
										 G_CALLBACK(preedit_window_press_cb), cn);
			
		border = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(border), GTK_SHADOW_OUT);
		gtk_container_add(GTK_CONTAINER(cn->preedit_win->eventbox), border);

		hbox = gtk_hbox_new(FALSE, 5);
		gtk_container_add(GTK_CONTAINER(border), hbox);

		input_border = gtk_frame_new(NULL);

		/* cn->preedit_win->text_view = gtk_label_new(cn->preedit_buf); */
		cn->preedit_win->text_view = preedit_area_new(cn->preedit_buf);

		gtk_misc_set_alignment(GTK_MISC(cn->preedit_win->text_view), 0.0, 0.5);
		gtk_container_add(GTK_CONTAINER(input_border), cn->preedit_win->text_view);
		gtk_box_pack_start(GTK_BOX(hbox), input_border, TRUE, TRUE, 0);
			
		cn->preedit_win->label = gtk_label_new(cfg.status_win_labels[cn->input_method]);
		gtk_box_pack_start(GTK_BOX(hbox), cn->preedit_win->label, FALSE, FALSE, 0);


		/* This hack is to disable flashy behaviour when window gets the focus*/
		g_signal_connect(G_OBJECT(cn->preedit_win->window), "enter_notify_event", 
										 G_CALLBACK(preedit_window_disable_focus_out), cn);
		g_signal_connect_swapped(G_OBJECT(cn->preedit_win->text_view), "realize",
														 G_CALLBACK(preedit_window_update_location), cn);
		/*
		cn->preedit_win->size_allocate_id = g_signal_connect_swapped(G_OBJECT(cn->toplevel_gtk), "size_allocate",
																																 G_CALLBACK(preedit_window_update_location), cn);
		g_signal_connect_swapped(G_OBJECT(cn->toplevel_gtk), "destroy",
														 G_CALLBACK(toplevel_destroyed_cb), cn->preedit_win);
		//cn->preedit_win->toplevel_gtk = cn->toplevel_gtk;
		*/
	}
	else {
		gtk_window_resize(GTK_WINDOW(cn->preedit_win->window), 140, 25);
		preedit_window_update(cn);
		gtk_label_set_text(GTK_LABEL(cn->preedit_win->label), cfg.status_win_labels[cn->input_method]);
		preedit_window_update_location(cn);
	}
#ifdef IMJA_TARGET_XIM
	preedit_window_hide_all();
#endif
	gtk_widget_show_all(cn->preedit_win->window);
	cn->preedit_win_on = TRUE;
}

void preedit_window_force_hide(IMJAContext *cn) {
	IM_JA_DEBUG("preedit_window_force_hide\n");

	if (cn->preedit_win != NULL) {
		gtk_widget_hide_all(cn->preedit_win->window);
	}
}

void preedit_window_hide(IMJAContext *cn) {
	IM_JA_DEBUG("preedit_window_hide\n");

	if (cn->preedit_win != NULL) {
		if (cn->preedit_win->menu != NULL) {
			/* don't hide if we have the menu pulled out */
			if ((GTK_WIDGET_HAS_GRAB(cn->preedit_win->menu) == TRUE)
					&& (cn->preedit_win_on == TRUE)) return;
		}
		if (gtk_widget_is_focus(cn->preedit_win->window) == TRUE) return;

		if (cn->preedit_win->can_hide == TRUE) {
			gtk_widget_hide_all(cn->preedit_win->window);
		}
	}
}

/*
static void toplevel_destroyed_cb(GtkWidget *toplevel, GdkEvent *event, PreeditWin *preedit_win) {
	IM_JA_DEBUG("toplevel_destroyed_cb()\n");
	destroy_win(preedit_win, FALSE);
}
*/

static void destroy_win(PreeditWin *preedit_win, gboolean remove_signal) {
	IM_JA_DEBUG("destroy_win() [%d]\n", (int) preedit_win);
	if (preedit_win == NULL) return;
	if (g_list_find(preedit_windows, preedit_win) == NULL) {
		IM_JA_DEBUG("*ERROR* preedit window is already destroyed\n");
		return;
	}
	preedit_windows = g_list_remove(preedit_windows, preedit_win);
	/*
	if (remove_signal == TRUE) {
		if (GTK_IS_WIDGET(preedit_win->toplevel_gtk)) {
			g_signal_handler_disconnect(preedit_win->toplevel_gtk, preedit_win->size_allocate_id);
		}
	}
	*/
	if (GTK_IS_WIDGET(preedit_win->window) == TRUE) gtk_widget_destroy(preedit_win->window);
	/* g_free(preedit_win); this causes a segfault in some cases */
}

void preedit_window_destroy(IMJAContext *cn) {
	IM_JA_DEBUG("preedit_window_destroy()\n");
	if (cn->preedit_win == NULL) return;
	destroy_win(cn->preedit_win, TRUE);
	cn->preedit_win = NULL;
}

void preedit_window_destroy_all() {
	IM_JA_DEBUG("preedit_window_destroy_all()\n");
	g_list_foreach(preedit_windows, (GFunc) destroy_win, (gpointer) TRUE);
	g_list_free(preedit_windows);
	preedit_windows = NULL;
}

void preedit_window_hide_all() {
	GList *windows = preedit_windows; 
	PreeditWin *preedit_win;

	IM_JA_DEBUG("preedit_window_hide_all()\n");
	
	while (windows != NULL) {
		preedit_win = (PreeditWin *) windows->data;
		if (preedit_win != NULL) {
			if (gtk_widget_is_focus(preedit_win->window) == TRUE) return;
			if (preedit_win->can_hide == TRUE) gtk_widget_hide_all(preedit_win->window);
		}
		windows = g_list_next(windows);
	}
}

void preedit_window_set_text(IMJAContext *cn, gchar *text) {
	GtkRequisition requisition;
	gint width, height, new_w;

	/* IM_JA_DEBUG("preedit_window_set_text\n"); */

	preedit_area_set_text(PREEDIT_AREA(cn->preedit_win->text_view), text);	

	gtk_widget_size_request(GTK_WIDGET(cn->preedit_win->window), &requisition);
	gtk_window_get_size(GTK_WINDOW(cn->preedit_win->window), &width, &height);

	if (requisition.width < width) {
		new_w = requisition.width > 140 ? requisition.width + 5 : 140;
		gtk_window_resize(GTK_WINDOW(cn->preedit_win->window), new_w, 25);
	}

}

void preedit_window_set_attributes(IMJAContext *cn, PangoAttrList *attrs) {
	/* IM_JA_DEBUG("preedit_window_set_attributes\n"); */
	preedit_area_set_attributes(PREEDIT_AREA(cn->preedit_win->text_view), attrs);
}

void preedit_window_set_cursor_pos(IMJAContext *cn, gint pos) {
	/* IM_JA_DEBUG("preedit_window_set_cursor_pos\n"); */
	preedit_area_set_cursor_pos(PREEDIT_AREA(cn->preedit_win->text_view), pos);
}

void preedit_window_update_location(IMJAContext *cn) {
	gint target_x = 0, target_y = 0;
	GdkRectangle toplevel_rect;
	GdkRectangle client_rect;

	/* IM_JA_DEBUG("im_ja_update_preedit_window_location\n"); */
	if (cn->preedit_win !=	NULL) {

		im_ja_get_toplevel_window_geometry(cn, &toplevel_rect);

		if ((cn->preedit_win_pos_offset_x == 0) && (cn->preedit_win_pos_offset_y == 0)) {
			im_ja_attach_bottom_left(cn, cn->preedit_win->window);
		}
		else {
			im_ja_get_client_window_geometry(cn, &client_rect);
			target_x = client_rect.x + cn->preedit_win_pos_offset_x;
			target_y = client_rect.y + cn->preedit_win_pos_offset_y;
			/* IM_JA_DEBUG("update preedit window location x: %d, y: %d\n", target_x, target_y); */

			im_ja_move_within_rect(cn, &target_x, &target_y, &client_rect);
			/* IM_JA_DEBUG("update preedit window location x: %d, y: %d\n", target_x, target_y); */
			gtk_window_move((GTK_WINDOW(cn->preedit_win->window)), target_x, target_y);
		}
	}
}

void preedit_window_update(IMJAContext *cn) {
	PangoAttrList *preedit_win_attrs;
	PangoAttribute *backgr;
	PangoAttribute *foregr;
	PangoAttribute *underline;
	gint preedit_cursor = 0;

	/* IM_JA_DEBUG("preedit_window_update\n"); */

	if ((cn->preedit_win == NULL) || (GTK_IS_WIDGET(cn->preedit_win->window) == FALSE)) return;

	if (strlen(cn->preedit_buf) == 0) {
		preedit_window_set_text(cn, "");
		preedit_window_set_cursor_pos(cn, 0);
	}

	preedit_win_attrs = pango_attr_list_new(); /* Create new attrib list for the preedit window */

	underline = pango_attr_underline_new(PANGO_UNDERLINE_SINGLE);
	underline->start_index = 0;
	underline->end_index = strlen(cn->preedit_buf);
	pango_attr_list_insert(preedit_win_attrs, underline);

	if (cfg.custom_preedit_n == TRUE) {
		foregr = pango_attr_foreground_new(cfg.preedit_colors[0].red, cfg.preedit_colors[0].green, cfg.preedit_colors[0].blue);
		backgr = pango_attr_background_new(cfg.preedit_colors[1].red, cfg.preedit_colors[1].green, cfg.preedit_colors[1].blue);
		backgr->start_index = 0;
		foregr->start_index = 0;
		backgr->end_index = strlen(cn->preedit_buf);
		foregr->end_index = strlen(cn->preedit_buf);
		pango_attr_list_insert(preedit_win_attrs, backgr);
		pango_attr_list_insert(preedit_win_attrs, foregr);
	}

	if (cn->preedit_reverse_start != cn->preedit_reverse_end) {
		/* Highlighted (==reverse) field */
		if (cfg.custom_preedit_hl == TRUE) {
			foregr = pango_attr_foreground_new(cfg.preedit_colors[2].red, cfg.preedit_colors[2].green, cfg.preedit_colors[2].blue);
			backgr = pango_attr_background_new(cfg.preedit_colors[3].red, cfg.preedit_colors[3].green, cfg.preedit_colors[3].blue);
		}
		else { /* Use the default */
			foregr = pango_attr_foreground_new(cn->original_colors[0].red, cn->original_colors[0].green, cn->original_colors[0].blue);
			backgr = pango_attr_background_new(cn->original_colors[1].red, cn->original_colors[1].green, cn->original_colors[1].blue);
		}
		backgr->start_index = cn->preedit_reverse_start;
		foregr->start_index = cn->preedit_reverse_start;
		backgr->end_index = cn->preedit_reverse_end;
		foregr->end_index = cn->preedit_reverse_end;
		
		pango_attr_list_insert(preedit_win_attrs, backgr);
		pango_attr_list_insert(preedit_win_attrs, foregr);
	}

	preedit_cursor = im_ja_get_cursor_pos_bytes(cn);
	preedit_window_set_cursor_pos(cn, preedit_cursor);
	preedit_window_set_text(cn, cn->preedit_buf);
	preedit_window_set_attributes(cn, preedit_win_attrs);

}
