/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 2; tab-width: 2 -*- */
/* about-box.c
 *
 * Copyright (C) 2001, 2002, 2003  Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Ettore Perazzoli <ettore@ximian.com>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "about-box.h"

#include <gtk/gtk.h>
#include <gdk/gdk.h>

#define PARENT_TYPE gtk_event_box_get_type ()
static GtkEventBoxClass *parent_class = NULL;

const gchar *about_text[] = {
	"IM-JA",
	"",
	"A JAPANESE INPUT MODULE FOR GTK2",
	"",
	"Copyright (C) 2002-2003 Botond Botyanszki",
	"Released under the GNU GPL",
	"",
	"",
	"",
	"CAST",
  "",
	"Directed by",
	"Botond Botyanszki <boti@rocketmail.com>",
	"",
	"Written by",
	"Botond Botyanszki <boti@rocketmail.com>",
	"",
	"Kanjipad enhancements and",
	"CannaRK implementation by",
	"Srin Tuar <srintuar26@earthlink.net>",
	"",
	"Useful patches and suggestions:",
	"Vadim Berezniker <vadim@berezniker.com>",
	"",
	"Kanjipad by",
	"Owen Taylor <otaylor@redhat.com>",
	"",
	"AboutBox by",
	"Ettore Perazzoli <ettore@ximian.com>",
	"",
	"",
	"Project Webpage is at",
	"http://im-ja.sourceforge.net"
};
#define NUM_TEXT_LINES (sizeof (about_text) / sizeof (gchar *))

struct _AboutBoxPrivate {
	GdkPixmap *pixmap;
	GdkPixmap *text_background_pixmap;
	GdkGC *clipped_gc;
	int text_y_offset;
	int timeout_id;
	const gchar **text;
};

#define ANIMATION_DELAY 100

#define WIDTH  400
#define HEIGHT 400

#define TEXT_Y_OFFSET 10
#define TEXT_X_OFFSET 10
#define TEXT_WIDTH    (WIDTH - 2 * TEXT_X_OFFSET)
#define TEXT_HEIGHT   350

#define IMAGE_PATH  PIXMAPDIR"/im-ja-about.jpg"

/* The callback.  */

static int
timeout_callback (void *data)
{
	AboutBox *about_box;
	AboutBoxPrivate *priv;
	GdkRectangle redraw_rect;
	GtkWidget *widget;
	PangoContext *context;
	PangoFontMetrics *metrics;
	PangoLayout *layout;
	int line_height;
	int first_line;
	int y;
	int i;

	about_box = ABOUT_BOX (data);
	priv = about_box->priv;

	widget = GTK_WIDGET (about_box);

	context = gtk_widget_get_pango_context (widget);
	metrics = pango_context_get_metrics (context, gtk_widget_get_style (GTK_WIDGET (about_box))->font_desc,
					     pango_context_get_language (context));
	line_height = PANGO_PIXELS (pango_font_metrics_get_ascent (metrics)
				    + pango_font_metrics_get_descent (metrics));
	/*	line_height += 5; */
	pango_font_metrics_unref (metrics);

	if (priv->text_y_offset < TEXT_HEIGHT) {
		y = TEXT_Y_OFFSET + (TEXT_HEIGHT - priv->text_y_offset);
		first_line = 0;
	} else {
		y = TEXT_Y_OFFSET - ((priv->text_y_offset - TEXT_HEIGHT) % line_height);
		first_line = (priv->text_y_offset - TEXT_HEIGHT) / line_height;
	}

	gdk_draw_drawable (priv->pixmap, priv->clipped_gc, priv->text_background_pixmap,
			 0, 0,
			 TEXT_X_OFFSET, TEXT_Y_OFFSET, TEXT_WIDTH, TEXT_HEIGHT);

	layout = pango_layout_new (context);

	for (i = 0; i < TEXT_HEIGHT / line_height + 3; i ++) {
		const char *line;
		int width;
		int x;

		if (first_line + i >= (int) NUM_TEXT_LINES)
			break;

		if (*priv->text[first_line + i] == '\0')	line = "";
		else line = priv->text[first_line + i];

		pango_layout_set_text (layout, line, -1);
		pango_layout_get_pixel_size (layout, &width, NULL);
		x = TEXT_X_OFFSET + (TEXT_WIDTH - width) / 2;
		gdk_draw_layout (priv->pixmap, priv->clipped_gc, x, y, layout);

		y += line_height;
	}

	redraw_rect.x      = TEXT_X_OFFSET;
	redraw_rect.y      = TEXT_Y_OFFSET;
	redraw_rect.width  = TEXT_WIDTH;
	redraw_rect.height = TEXT_HEIGHT;
	gdk_window_invalidate_rect (widget->window, &redraw_rect, FALSE);
	gdk_window_process_updates (widget->window, FALSE);

	priv->text_y_offset ++;
	if (priv->text_y_offset > (int) (line_height * NUM_TEXT_LINES + TEXT_HEIGHT)) {
		priv->text_y_offset = 0;
	}

	g_object_unref (layout);

	return TRUE;
}

/* GObject methods.  */

static void
impl_dispose (GObject *object)
{
	AboutBox *about_box;
	AboutBoxPrivate *priv;

	about_box = ABOUT_BOX (object);
	priv = about_box->priv;

	if (priv->pixmap != NULL) {
		g_object_unref (priv->pixmap);
		priv->pixmap = NULL;
	}

	if (priv->text_background_pixmap != NULL) {
		g_object_unref (priv->text_background_pixmap);
		priv->text_background_pixmap = NULL;
	}

	if (priv->clipped_gc != NULL) {
		g_object_unref (priv->clipped_gc);
		priv->clipped_gc = NULL;
	}

	(* G_OBJECT_CLASS (parent_class)->dispose) (object);
}

static void
impl_finalize (GObject *object)
{
	AboutBox *about_box;
	AboutBoxPrivate *priv;

	about_box = ABOUT_BOX (object);
	priv = about_box->priv;

	if (priv->timeout_id != -1)
		g_source_remove (priv->timeout_id);

	g_free (priv->text);

	g_free (priv);

	(* G_OBJECT_CLASS (parent_class)->finalize) (object);
}

/* GtkWidget methods.  */

static void
impl_size_request (GtkWidget *widget,
		   GtkRequisition *requisition)
{
	requisition->width = WIDTH;
	requisition->height = HEIGHT;
}

static void
impl_realize (GtkWidget *widget)
{
	AboutBox *about_box;
	AboutBoxPrivate *priv;
	GdkPixbuf *background_pixbuf;
	GdkRectangle clip_rectangle;

	(* GTK_WIDGET_CLASS (parent_class)->realize) (widget);

	about_box = ABOUT_BOX (widget);
	priv = about_box->priv;

	background_pixbuf = gdk_pixbuf_new_from_file (IMAGE_PATH, NULL);
	/*
	g_assert (background_pixbuf != NULL);
	g_assert (gdk_pixbuf_get_width (background_pixbuf) == WIDTH);
	g_assert (gdk_pixbuf_get_height (background_pixbuf) == HEIGHT);
	g_assert (priv->pixmap == NULL);
	*/
	priv->pixmap = gdk_pixmap_new (widget->window, WIDTH, HEIGHT, -1);

	gdk_draw_pixbuf (priv->pixmap, widget->style->black_gc, background_pixbuf,
				       0, 0, 0, 0, WIDTH, HEIGHT,
				       GDK_RGB_DITHER_MAX, 0, 0);

	/* g_assert (priv->clipped_gc == NULL); */
	priv->clipped_gc = gdk_gc_new (widget->window);
	gdk_gc_copy (priv->clipped_gc, widget->style->black_gc);

	clip_rectangle.x      = TEXT_X_OFFSET;
	clip_rectangle.y      = TEXT_Y_OFFSET;
	clip_rectangle.width  = TEXT_WIDTH;
	clip_rectangle.height = TEXT_HEIGHT;
	gdk_gc_set_clip_rectangle (priv->clipped_gc, & clip_rectangle);

	priv->text_background_pixmap = gdk_pixmap_new (widget->window, clip_rectangle.width, clip_rectangle.height, -1);
	gdk_draw_pixbuf (priv->text_background_pixmap, widget->style->black_gc, background_pixbuf,
																 TEXT_X_OFFSET, TEXT_Y_OFFSET,
																 0, 0, TEXT_WIDTH, TEXT_HEIGHT,
																 GDK_RGB_DITHER_MAX, 0, 0);

	/* g_assert (priv->timeout_id == -1); */
	priv->timeout_id = g_timeout_add (ANIMATION_DELAY, timeout_callback, about_box);

	g_object_unref (background_pixbuf);
}

static void
impl_unrealize (GtkWidget *widget)
{
	AboutBox *about_box;
	AboutBoxPrivate *priv;

	about_box = ABOUT_BOX (widget);
	priv = about_box->priv;

	(* GTK_WIDGET_CLASS (parent_class)->unrealize) (widget);

	if (priv->clipped_gc != NULL) g_object_unref (priv->clipped_gc);
	priv->clipped_gc = NULL;

	if (priv->pixmap != NULL) g_object_unref (priv->pixmap);
	priv->pixmap = NULL;

	if (priv->timeout_id != -1) {
		g_source_remove (priv->timeout_id);
		priv->timeout_id = -1;
	}
}

static int
impl_expose_event (GtkWidget *widget,
		   GdkEventExpose *event)
{
	AboutBoxPrivate *priv;

	if (! GTK_WIDGET_DRAWABLE (widget)) return FALSE;

	priv = ABOUT_BOX (widget)->priv;

	gdk_draw_drawable (widget->window, widget->style->black_gc,
										 priv->pixmap,
										 event->area.x, event->area.y,
										 event->area.x, event->area.y,
										 event->area.width, event->area.height);
	
	return TRUE;
}

static void
class_init (GObjectClass *object_class)
{
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_ref(PARENT_TYPE);

	object_class->dispose  = impl_dispose;
	object_class->finalize = impl_finalize;

	widget_class = GTK_WIDGET_CLASS (object_class);
	widget_class->size_request = impl_size_request;
	widget_class->realize      = impl_realize;
	widget_class->unrealize    = impl_unrealize;
	widget_class->expose_event = impl_expose_event;
}

static void
init (AboutBox *about_box)
{
	AboutBoxPrivate *priv;
	gint i;

	priv = g_new (AboutBoxPrivate, 1);
	priv->pixmap                 = NULL;
	priv->text_background_pixmap = NULL;
	priv->clipped_gc             = NULL;
	priv->timeout_id             = -1;
	priv->text_y_offset          = 0;

	priv->text = g_new0 (const gchar *, NUM_TEXT_LINES);
	for (i = 0; i < (int) NUM_TEXT_LINES; ++i) {
		priv->text[i] = about_text[i];
	}

	about_box->priv = priv;
}

GtkWidget *
about_box_new (void)
{
	AboutBox *about_box;

	about_box = g_object_new (about_box_get_type (), NULL);

	return GTK_WIDGET (about_box);
}


GType
about_box_get_type (void)
{
  static GType about_box_type = 0;
  
  if (!about_box_type)
    {
      static const GTypeInfo about_box_info =
      {
				sizeof (AboutBoxClass),
				NULL,           /* base_init */
				NULL,           /* base_finalize */
				(GClassInitFunc) class_init,
				NULL,           /* class_finalize */
				NULL,           /* class_data */
				sizeof (AboutBox),
				0,              /* n_preallocs */
				(GInstanceInitFunc) init,
				NULL,           /* value_table */
      };

      about_box_type = g_type_register_static(GTK_TYPE_EVENT_BOX, "AboutBox", &about_box_info, 0);
    }

  return about_box_type;
}
