/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/*
 * IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 *
 */

#include <gtk/gtk.h>
#include <sys/stat.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <gconf/gconf-client.h>

#include <config.h>

#include "helper-socket.h"
#include "helper-client.h"
#include "../error.h"
#include "../im-ja.h"
#include "../conf.h"
#include "../actionmenu.h"
#include "../eggtrayicon/eggtrayicon.h"
#include "../nls.h"

static void cleanup_on_signal(int sig);
static void init_trayicon();

static GHashTable *clients;
static gint sock_fd = -1;
static gchar *sock_path = NULL;
static EggTrayIcon *trayicon = NULL;
static GtkWidget *traybutton = NULL;
static GtkWidget *traymenu = NULL;
static gboolean daemonize = FALSE;
static gchar *progname;
static gint current_input_method = 0;
static guint notify_id = 0;

extern GConfClient *gconf_client;

IMJAConfig cfg;


void reload_settings_cb(GConfClient *client, guint cnxn_id, GConfEntry *entry, gpointer data) {
	im_ja_load_conf(&cfg);
	if ((trayicon != NULL) && (cfg.use_systray == FALSE)) {
		gtk_widget_destroy(GTK_WIDGET(trayicon));
		trayicon = NULL;
		traybutton = NULL;
	}
	else init_trayicon();
	
}

void broadcast_cmd_hashfunc(gpointer key, ClientIO *client, gchar *cmd) {
	helper_client_io_send(client, cmd);
}

void broadcast_command(gchar *cmd) {
	g_hash_table_foreach(clients, (GHFunc) broadcast_cmd_hashfunc, cmd);
}

void change_input_method_request(gpointer key, ClientIO *client, gint input_method) {
	helper_client_send_command(client, HELPER_MSG_SET_INPUT_METHOD, input_method);
}

void im_ja_run_configurator() {
	static gchar *command[] = {IM_JA_BINDIR"/im-ja-conf"};
	GError *g_error = NULL;
	if (g_spawn_async(NULL, command, NULL, G_SPAWN_STDOUT_TO_DEV_NULL | G_SPAWN_STDERR_TO_DEV_NULL,
										NULL, NULL, NULL, &g_error) == FALSE) {
		im_ja_print_error(g_error->message); 
	}
}

gboolean im_ja_execute_action(gpointer data, gint action_id, gboolean set_input_method) {
	IM_JA_DEBUG("im_ja_execute_action[helper]\n");
	if (set_input_method == TRUE) {
		g_hash_table_foreach(clients, (GHFunc) change_input_method_request, (gpointer) action_id);
		return TRUE;
	}
	switch (action_id) {
	case START_CONFIGURATOR:
		im_ja_run_configurator();
		return TRUE;
	}
	return FALSE;
}


void destroy_client_io(ClientIO *client) {
	IM_JA_DEBUG("destroy_client_io: %d\n", (int) client);
	if (client != NULL) {
		if (g_hash_table_remove(clients, client) == FALSE) {
			IM_JA_DEBUG("failed to remove client %d from hashtable\n", (int) client);
		}
		helper_client_io_close(client);
	}
	if (g_hash_table_size(clients) == 0) {
		IM_JA_DEBUG("No more clients connected, exiting.\n");
		cleanup_on_signal(0);
	}
}

void process_message(ClientIO *client, gchar *msg) {
	gchar *ptr;
	gint input_method = -1;
	if (msg == NULL) return;

	msg[strlen(msg) - 1] = 0; /* remove trailing newline */
	IM_JA_DEBUG("processing [%s]\n", msg);

	if (g_str_has_prefix(msg, HELPER_MSG_PING) == TRUE) {
		helper_client_io_send(client, HELPER_MSG_ACK);
	}
	else if (g_str_has_prefix(msg, HELPER_MSG_CHANGE_STATUS) == TRUE) {
		ptr = msg + strlen(HELPER_MSG_CHANGE_STATUS);
		input_method = atoi(ptr);
		IM_JA_DEBUG("Change to: %d\n", input_method);
		if (traybutton != NULL) {
			gtk_button_set_label(GTK_BUTTON(traybutton), cfg.status_win_labels[input_method]);
		}
		broadcast_command(msg);
	}
	else if (g_str_has_prefix(msg, HELPER_MSG_SET_INPUT_METHOD) == TRUE) {
		ptr = msg + strlen(HELPER_MSG_SET_INPUT_METHOD);
		input_method = atoi(ptr);
		IM_JA_DEBUG("HELPER_MSG_SET_INPUT_METHOD to: %d\n", input_method);
		current_input_method = input_method;
		im_ja_execute_action(NULL, input_method, TRUE);
	}
	else {
		IM_JA_DEBUG("Unprocessed helper message: %s\n", msg);
	}
}

gboolean input_handler(GIOChannel *source,
											 GIOCondition condition,
											 ClientIO *client) {

	GIOStatus status;
	gchar *line = NULL;
	GError *err = NULL;

	IM_JA_DEBUG("input_handler()\n");

	if (condition & G_IO_ERR) {
		IM_JA_DEBUG("IO Error\n");
		destroy_client_io(client);
		return FALSE;
	}

	if (condition & G_IO_IN) {
		IM_JA_DEBUG("data input:\n");

		status = g_io_channel_read_line(source, &line, NULL, NULL, &err);

		if (status & G_IO_STATUS_ERROR) {
			g_error("Error reading from client: %s\n", err->message);
			destroy_client_io(client);
			return FALSE;
		}

		else if (status & G_IO_STATUS_EOF) {
			IM_JA_DEBUG("EOF\n");
			destroy_client_io(client);
			return FALSE;
		}
		else if (status & G_IO_STATUS_NORMAL) {
			IM_JA_DEBUG("G_IO_STATUS_NORMAL\n");
			if ((line == NULL) || (strlen(line) == 0)) {
				IM_JA_DEBUG("connection closed\n");
				destroy_client_io(client);
				return FALSE;
			}
			else { /* OK */
				process_message(client, line);
				g_free(line);
				line = NULL;
				return TRUE;
			}
		}
		else if (status & G_IO_STATUS_AGAIN) {
			IM_JA_DEBUG("G_IO_STATUS_AGAIN\n");
			return TRUE;
		}
		return TRUE;
	}

	else if (condition & G_IO_ERR) {
		IM_JA_DEBUG("io error\n");
		destroy_client_io(client);
		return FALSE;
	}
	else if (condition & G_IO_HUP) {
		IM_JA_DEBUG("disconnection\n");
		destroy_client_io(client);
		return FALSE;
	}
	else if (condition & G_IO_NVAL) {
		IM_JA_DEBUG("invalid request, file descriptor is closed.\n");
		destroy_client_io(client);
		return FALSE;
	}

	return FALSE;

}

void add_client(gint socket) {
	ClientIO *client = helper_client_io_new_from_socket(socket);
	client->watch_id = g_io_add_watch(client->io,
																		G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
																		(GIOFunc) input_handler, client);

	g_hash_table_insert(clients, client, client);
	IM_JA_DEBUG("added client %d (io: %d) with watch id %d\n", (int) client, (int) client->io, client->watch_id);
}


gboolean helper_func(GIOChannel *source,
										 GIOCondition condition,
										 gpointer data) {

  if (condition & G_IO_IN) {
		gint client_sock = fd_accept(sock_fd);
		IM_JA_DEBUG("new connection\n");
		add_client(client_sock);
	}
	else {
		IM_JA_DEBUG("helper error\n");
		return FALSE;
	}

	return TRUE;
}


gboolean helper_running(gchar *socket_path) {
	struct stat sock_stat;

	if (stat(socket_path, &sock_stat) == 0) {
		gint serv_sock;
		IM_JA_DEBUG("%s already exists\n", socket_path);
		serv_sock = fd_connect_unix(socket_path);
		if (serv_sock < 0) {
			im_ja_print_error_cmdline("couldn't connect to socket %s, removing.\n", socket_path);
			if (unlink(socket_path) != 0) im_ja_print_error_cmdline("couldn't remove %s\n", socket_path);
		}
		else {
			ClientIO *server = helper_client_io_new_from_socket(serv_sock);
			IM_JA_DEBUG("pinging helper.\n");
			
			if (helper_client_io_send(server, HELPER_MSG_PING) == TRUE) {
				GIOStatus status;
				gchar *line = NULL;
				GError *err = NULL;

				IM_JA_DEBUG("sent PING\n");
				/* check response */
				status = g_io_channel_read_line(server->io, &line, NULL, NULL, &err);
				if (status & G_IO_STATUS_NORMAL) {
					IM_JA_DEBUG("read %s\n", line);
					if (g_str_has_prefix(line, HELPER_MSG_ACK)) return TRUE;
				}
				else {
					IM_JA_DEBUG("didn't receive ACK\n");
				}					
			}
		}
	}
	return FALSE;
}

static void cleanup_on_signal(int sig) {
	close(sock_fd);
	/* FIXME: close client connections */
	if (unlink(sock_path) != 0) g_error("couldn't remove %s\n", sock_path);

	im_ja_finalize_conf_handler(NULL); /* FIXME conf*/
	gtk_main_quit();
}

	
static void traybutton_press_cb(GtkWidget *button, gpointer data) {
	IM_JA_DEBUG("traybutton_press_cb\n");
	im_ja_actionmenu_button_press_cb(traybutton, NULL, &traymenu, 
																	 IM_JA_PANEL_MENU, NULL);
}

static void init_trayicon() {
	if ((trayicon == NULL) && (cfg.use_systray == TRUE)) {
		GtkWidget *traybox;
		GtkWidget *trayborder;

		trayicon = egg_tray_icon_new("im-ja-icon");
		traybox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(trayicon), traybox);
		trayborder = gtk_frame_new(NULL);
		gtk_box_pack_start(GTK_BOX(traybox), trayborder, TRUE, FALSE, 0);

		traybutton = gtk_button_new_with_label("?");
		g_signal_connect(G_OBJECT(traybutton), "clicked",
										 G_CALLBACK(traybutton_press_cb), NULL);

		gtk_widget_show(traybutton);
		gtk_container_add(GTK_CONTAINER(trayborder), traybutton);

		gtk_widget_show_all(GTK_WIDGET(trayicon));
	}
}


static void notify_parent(int ps, int status) {
	if (daemonize == FALSE) return;
	write(ps,&status,sizeof(status));
	close(ps);
}


static void daemonify_helper(int *ps) {
	int i, pipefd[2];
	pid_t	pid;

	if (daemonize == FALSE) return;

	chdir("/");

	for(i = 0; i <= getdtablesize(); i++)	close(i);
	
	if (open("/dev/null",O_RDONLY) != 0) {
		exit(1);
	}
	if (open("/dev/null",O_WRONLY) != 1) {
		exit(1);
	}
	if (open("/dev/null",O_WRONLY) != 2) {
		exit(1);
	}
	if (pipe(pipefd) != 0) {
		exit(1);
	}

	if ((pid=fork()) == -1) {
		exit(1);
	}

	if (pid != 0) { /* parent waits for notification and exits */
		if (read(pipefd[0], &i, sizeof(i)) != 4) {
	    exit(2);
		}
		exit(i);
	}

	if (setsid() == -1) {
		exit(3);
	}

	if ((pid=fork()) == -1) { /* forking a grandchild failed */
		notify_parent(pipefd[1], 1);
		exit(1);
	}

	if (pid != 0) exit(0); /* child process exits */

	close(pipefd[0]);
	*ps	= pipefd[1];
	/* grandchild returns */
}

static void print_usage() {
	im_ja_print_error_cmdline("Usage: %s [-d/--daemonize -t/--no-tray-icon]\n", progname);
	_exit(1);
}


int main(int argc, char *argv[]) {
	GIOChannel *serv_io;
	int ps;
	int i;
	GError *err = NULL;
	gboolean showtray = TRUE;

	progname = argv[0];

  for (i = 1; i < argc; i++) {
		if (!strcmp(argv[i], "--daemonize") || !strcmp(argv[i], "-d")) {
			daemonize = TRUE;
			continue;
		}
		if (!strcmp(argv[i], "--no-tray-icon") || !strcmp(argv[i], "-t")) {
			showtray = FALSE;
			continue;
		}
		print_usage();
	}


	daemonify_helper(&ps);

#ifdef ENABLE_NLS
  bindtextdomain(GETTEXT_PACKAGE, IM_JA_LOCALE_DIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
  textdomain(GETTEXT_PACKAGE);
#endif

	gtk_init(&argc, &argv);

	sock_path = get_sock_path();
	if (sock_path == NULL) g_error("couldn't get socket path\n");

	if (helper_running(sock_path) == TRUE) {
		IM_JA_DEBUG("im-ja-helper already running. Exiting.\n");
		notify_parent(ps, 1);
		exit(1);
	}

	sock_fd = fd_open_unix(sock_path);
	if (sock_fd < 0) g_error("couldn't create socket %s\n", sock_path);
	set_nonblocking_mode(sock_fd, TRUE);

	clients = g_hash_table_new_full(g_int_hash, g_int_equal, NULL, NULL);

	if (im_ja_init_conf_handler() == FALSE) {
		im_ja_print_error_cmdline(_("GConf initialization failed!"));
		notify_parent(ps, 1);
		exit(1);
	}
	im_ja_get_gconf_client();
	if (im_ja_load_conf(&cfg) == FALSE) {
		im_ja_print_error_cmdline(_("Couldn't load settings!"));
		notify_parent(ps, 1);
		exit(1);
	}

	gconf_client_add_dir(gconf_client, GCONF_NAMESPACE"other", GCONF_CLIENT_PRELOAD_NONE, NULL);
	notify_id = gconf_client_notify_add(gconf_client,
																			GCONF_NAMESPACE"other/use_systray",
																			(GConfClientNotifyFunc) reload_settings_cb,
																			NULL, NULL, &err);

	

  /* Add a watch for incoming client connections */
	serv_io = g_io_channel_unix_new(sock_fd);
  g_io_add_watch(serv_io,
                 G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
                 helper_func, NULL);


	notify_parent(ps, 0);

  signal(SIGINT, cleanup_on_signal);
  signal(SIGTERM, cleanup_on_signal);

	if (showtray == TRUE) {
		init_trayicon();
	}

	gtk_main();

	im_ja_finalize_conf_handler(&cfg);

	return 0;
}
