/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* vs: set ts=2 sw=2: */
/* IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Botond Botyanszki <boti@rocketmail.com>
 *          Srin Tuar <srintuar26@earthlink.net>
 */

#include <gdk/gdkevents.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include <string.h>

#include "common.h"
#include "conf.h"
#include "error.h"

gchar *euc2utf8(const gchar *str) {
  GError *error = NULL;
  gchar *result;
  
  result = g_convert (str, -1, "UTF-8", "EUC-JP", NULL, NULL, &error);
  if (!result)  {
		result = g_convert (str, -1, "UTF-8", "EUC-JISX0213", NULL, NULL, &error);
		if( !result ) {
			g_warning ("Error converting text from EUC-JP to UTF-8: %s\n", error->message);
			g_error_free (error);
			result = g_strdup("???");
		}
  }
  
  return result;
}

gchar *utf82euc(const gchar *str) {
  GError *error = NULL;
  gchar *result;
  
  result = g_convert (str, -1, "EUC-JP", "UTF-8", NULL, NULL, &error);
  if (!result)  {
    g_warning ("Error converting text from UTF-8 to EUC-JP: %s\n", error->message);
    g_error_free (error);
  }
  
  return result;
}

gchar *utf8to16(const gchar *str) {
  GError *error = NULL;
  gchar *result;
  
  result = g_convert (str, -1, "UTF-16", "UTF-8", NULL, NULL, &error);
  if (!result)  {
    g_warning ("Error converting text from UTF-8 to UTF-16: %s\n", error->message);
    g_error_free (error);
  }
  
  return result;
}

int euc2wc(gchar *eucstr) { /* probably reinvented the wheel here. */
	int wcchar;
 
	if (((unsigned char) (*eucstr)) > 128) {
		wcchar = ((int) *eucstr << 8) + ((int) *(eucstr + 1) & 0xff);
	}
	else {
		wcchar = *eucstr;
	}
	return wcchar;
}

gchar *wc2euc(wchar *wcstr, gint size) { /* probably reinvented the wheel here. */
	gchar *eucstr, *eucptr;
	int i;
	eucstr = g_new0(gchar, size * 2 + 1);
	eucptr = eucstr;
	i = 0;
	for (i = 0; i < size; i++) {
		if (*wcstr == 0) break;
		if (*(wcstr + i) > 128) {
			*eucptr = *(wcstr + i) >> 8;
			*(eucptr + 1) = *(wcstr + i) & 0xff;
			eucptr += 2;
		}
		else {
			*eucptr = *(wcstr + i);
			eucptr++;
		}
	}
	return eucstr;
}


gboolean isJPChar(gunichar c) {
  if (isKanaChar(c) == TRUE) return TRUE;
  if (isKanjiChar(c) == TRUE) return TRUE;
  if (isOtherChar(c) == TRUE) return TRUE;
  return FALSE;
}
gboolean isKanaChar(gunichar c) {
  if (isKatakanaChar(c) == TRUE) return TRUE;
  if (isHiraganaChar(c) == TRUE) return TRUE;
  return FALSE;
}
gboolean isKatakanaChar(gunichar c) {
  if ((c >= 0x30A0) && (c <= 0x30FF)) return TRUE; /* Full and half Katakana */
  if ((c >= 0xFF65) && (c <= 0xFF9F)) return TRUE; /* Narrow Katakana */
  return FALSE;
}
gboolean isHiraganaChar(gunichar c) {
  if ((c >= 0x3040) && (c <= 0x309F)) return TRUE; /* Hiragana */
  return FALSE;
}
gboolean isKanjiChar(gunichar c) {
  if ((c >= 0x3300) && (c <= 0x33FF)) return TRUE; /* cjk compatibility */
  if ((c >= 0x3400) && (c <= 0x4DBF)) return TRUE; /* cjk ext A */
  if ((c >= 0x4E00) && (c <= 0x9FAF)) return TRUE; /* cjk unified */
  if ((c >= 0x20000) && (c <= 0x2A6DF)) return TRUE; /* cjk ext B */
  if ((c >= 0x2F800) && (c <= 0x2FA1F)) return TRUE;  /* cjk supplement */
  return FALSE;
}
gboolean isOtherChar(gunichar c) {
  if ((c >= 0x2E80) && (c <= 0x2EFF)) return TRUE;  /* cjk radical */
  if ((c >= 0x2F00) && (c <= 0x2FDF)) return TRUE;  /* cjk kangxi radicals */
  if ((c >= 0x2FF0) && (c <= 0x2FFF)) return TRUE;  /* ideographic */
  if ((c >= 0x3000) && (c <= 0x303F)) return TRUE;  /* punctuation */
  if ((c >= 0x3200) && (c <= 0x32FF)) return TRUE;  /* enclosed letters */
  if ((c >= 0xFE30) && (c <= 0xFE4F)) return TRUE;  /* compatibility forms */
  if ((c >= 0xFF00) && (c <= 0xFF64)) return TRUE;  /* compatibility forms2 */
  if ((c >= 0xFFA0) && (c <= 0xFFEF)) return TRUE;  /* compatibility forms3 */
  return FALSE;
}

/* Convert Hiragana -> Katakana.*/
gchar *hira2kata(gchar *hirastr) {
  gchar *hiraptr;
  gchar *kata = g_new0(gchar, strlen(hirastr) + 6);
  gchar *kataptr = kata;
  int length;

  hiraptr = hirastr;
  while (*hiraptr != 0) {
    if (isHiraganaChar(g_utf8_get_char(hiraptr)) == TRUE) {
      g_unichar_to_utf8(g_utf8_get_char(hiraptr) + 96, kataptr);
    }
    else {
      length = g_utf8_next_char(hiraptr) - hiraptr;
      strncat(kataptr, hiraptr, length);
      kataptr[length + 1] = 0;
    }
    kataptr = g_utf8_next_char(kataptr);
    hiraptr = g_utf8_next_char(hiraptr);
    if (hiraptr == NULL) break;
  } 
  return kata;
}

/*
gint char_to_byte_pos(gchar *str, gint pos) {
	gint byte_pos = 0;
	gchar *ptr = str;
	gint len = strlen(str);
	gint cnt = 0;

	if (len == 0) return 0;

	while (byte_pos <= len) {
		if (cnt == pos) break;
		ptr = g_utf8_next_char(ptr);
		byte_pos = ptr - str;
		cnt++;
	}
	return byte_pos;
}
*/

/* Convert fullsize katakana to half width*/
/*
gchar *full2half(gchar *instr) {
  gchar *inptr = instr;
  gchar *out = g_new0(gchar, strlen(instr));
  gchar *outptr = out;
  int length;

  IM_JA_DEBUG("instr:%s\n", utf82euc(instr));

  while (*inptr != 0) {
    if (isKanaChar(g_utf8_get_char(inptr)) == TRUE) {
      IM_JA_DEBUG("hex:%x\n", (int)g_utf8_get_char(inptr));
      g_unichar_to_utf8(g_utf8_get_char(inptr) - 1, outptr);
    }
    else {
      length = g_utf8_next_char(inptr) - inptr;
      strncat(outptr, inptr, length);
      outptr[length + 1] = 0;
    }
    outptr = g_utf8_next_char(outptr);
    inptr = g_utf8_next_char(inptr);
    if (inptr == NULL) break;
  } 
  IM_JA_DEBUG("outstr:%s\n", utf82euc(out));
  return out;
}
*/

#define ALL_ACCELS_MASK (GDK_CONTROL_MASK | GDK_SHIFT_MASK | GDK_MOD1_MASK)
gboolean im_ja_is_printable_key(GdkEventKey *key) {
	/*
	guint keyval;
	gint eff_group, level;
	GdkModifierType consumed;

	gdk_keymap_translate_keyboard_state(gdk_keymap_get_default(), key->hardware_keycode,
																			key->state, key->group,
                                     &keyval, &eff_group, &level, &consumed);
	if (key->state & ~consumed & ALL_ACCELS_MASK) return FALSE;
	*/

	if (key->state & (gtk_accelerator_get_default_mod_mask() & ~GDK_SHIFT_MASK)) return FALSE;
	
	/* non-printable KP keys */
	if (key->keyval >= GDK_KP_Enter && key->keyval <= GDK_KP_Delete) {
		return FALSE;
	}

	IM_JA_DEBUG(" printable by gdk\n");
  /* NORMAL CHAR KEYS */
	if ((key->keyval >= GDK_exclam && key->keyval <= GDK_overline) ||
			(key->keyval >= GDK_KP_Space && key->keyval <= GDK_KP_9)) {
		return TRUE;
	}

	return FALSE;

}

gboolean ishotkey(GdkEventKey *key, int hotkey, IMJAConfig *cfg) {
  guint state = key->state;
  
  state &= ~GDK_LOCK_MASK; 
  state &= ~GDK_MOD2_MASK;
  state &= ~GDK_MOD5_MASK;
  
  if ((state == cfg->hotkey_states[hotkey]) &&
      (key->keyval == cfg->hotkey_values[hotkey])) return TRUE;
  
  return FALSE;

}

gboolean buffer_delchar(gchar* buf) {
  if( buf == NULL ) return FALSE;

  g_utf8_prev_char(buf + strlen(buf))[0] = 0;
  return TRUE;
}

/*
gchar *get_utf8_for_keyval(guint keyval) {
  gchar *utf8strg = g_new0(gchar, 7);
  int written = g_unichar_to_utf8(gdk_keyval_to_unicode(keyval), utf8strg);
  utf8strg[written] = 0;
  return utf8strg;
}
*/

int buffer_inschar(char* buf, int maxbuf, char *ins, int pos) {
	char *repo, *end;
	int clen = strlen(buf);
	int nlen = strlen(ins);

	if (clen+nlen >= maxbuf)	{
		nlen = maxbuf - clen - 1;
	}

	clen += nlen;

	repo = buf + clen;
	end = buf + pos + nlen;

	while (repo >= end) {
		*repo = *(repo-nlen);
		--repo;
	}

	repo = buf + pos;
	end = repo + nlen;

	while (repo < end) *repo++ = *ins++;
 
	return nlen;
}

int buffer_bkspchar(char* buf, int pos) {
	int delta=0;
	char *prev, *at;

	at = buf + pos;
	prev = g_utf8_find_prev_char(buf, at);

	if (prev) {
		delta = at - prev;
		do {
			*prev++ = *at++;
		}
		while (*prev);
	}

	return delta;
}

int buffer_deltchar(char* buf, int pos) {
	int delta=0;
	char *next, *at;

	at = buf + pos;
	next = g_utf8_find_next_char(at, 0);

	if (next) {
		delta = next - at;
		do {
			*at++ = *next++;
		}
		while (*at);
	}

	return delta;
}
