/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* vi: set ts=4 sw=4:
 * IM-JA Japanese Input Method Module for GTK-2.0
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Srin Tuar <srintuar26@earthlink.net>
 */

#include <string.h>
#include <stdio.h>

#include <canna/keydef.h>
#include <gdk/gdkkeysyms.h>

#define CANNA_NEW_WCHAR_AWARE
#include <canna/RK.h>

#include "../common.h"
#include "../error.h"
#include "../romakana.h"
#include "../candwin.h"
#include "../im-ja.h"
#include "../im-ja-impl.h"

#include "canna_rk.h"

/* manual prototypes FIXME: dont manually declare prototypes for externals
 * * * * * * * * * * * */

extern IMJAConfig cfg;

/*gboolean buffer_delchar(gchar* buf);*/
/* * * * * * * * * * * */

/* Internal use declarations
 * * * * * * * * * * * */

static inline gboolean in_preedit_mode(IMJAContext *cn);
static inline gboolean in_conversion_mode(canna_rk_context *ctx);

/*convert euc buffer to utf-8, set as conversion buffer*/
/** emits preedit_changed **/
static void set_preedit_from_conversion(IMJAContext *cn, canna_rk_context *ctx);

/*Get status from Canna API, may call "set_preedit_from_conversion"*/
/** emits preedit_changed **/
static void update_status(IMJAContext *cn, canna_rk_context *ctx);

static void update_clauses(IMJAContext *cn, canna_rk_context *ctx, int sel_clause );

/*convert preedit to euc, start conversion, call "set_preedit_from_conversion"*/
static void enter_conversion_mode(IMJAContext *cn, canna_rk_context *ctx);

/** these emit preedit_changed **/
/*get next candidate or open candidate window*/
static void spacebar_next_candidate(IMJAContext *cn, canna_rk_context *ctx);
static void list_candidates(IMJAContext *cn, canna_rk_context *ctx);
static void next_candidate(IMJAContext *cn, canna_rk_context *ctx);
static void prev_candidate(IMJAContext *cn, canna_rk_context *ctx);
static void goto_candidate(IMJAContext *cn, canna_rk_context *ctx, int candno );

/*
 End conversion, dont commit preedit yet though
 This function will change the preedit buffer, but it does not emit any signal
 the caller is responsible for deciding which signal to emit
*/
static void leave_conversion_mode(IMJAContext *cn, canna_rk_context *ctx);

/** emits preedit_changed **/
static void next_clause(IMJAContext *cn, canna_rk_context *ctx);
static void prev_clause(IMJAContext *cn, canna_rk_context *ctx);
static void goto_clause(IMJAContext *cn, canna_rk_context *ctx, int clause);

/** emits preedit_changed **/
static void shrink_clause(IMJAContext *cn, canna_rk_context *ctx);
static void grow_clause(IMJAContext *cn, canna_rk_context *ctx);

/** emits preedit_changed **/
static void unconvert_clause(IMJAContext *cn, canna_rk_context *ctx);

/** does not emit anything **/
static void unconvert_all_clauses(IMJAContext *cn, canna_rk_context *ctx);

/* * * * * * * * * * * */


static inline gboolean in_conversion_mode(canna_rk_context *ctx)
{
	return ctx->m_num_clauses >= 0;
}
static inline gboolean in_preedit_mode(IMJAContext *cn)
{
	return cn->preedit_buf[0] != 0;
}

void canna_rk_shutdown(IMJAContext *cn)
{
	IM_JA_DEBUG("FIXME: canna_rk_shutdown()\n");
}

void canna_rk_update_preedit(IMJAContext *cn)
{

}

void canna_rk_commit(IMJAContext *cn)
{
	canna_rk_context *ctx = (canna_rk_context*)cn->canna_rk_ctx;
	if( in_conversion_mode(ctx) )
	{
		leave_conversion_mode(cn,ctx);
	}
}

gboolean canna_rk_init(IMJAContext *cn)
{
	int dics = 0;
	int i;
	gchar *tmpstr;
	canna_rk_context *ctx;

	ctx = g_new0(canna_rk_context,1);
	cn->canna_rk_ctx = ctx;
	cn->cursor_ndx = 0;

	ctx->m_num_clauses = -1;
	ctx->m_consecutive_spacebar_count = 0;
	ctx->m_conv_euc[0]=0;
	ctx->m_canna_rk_mode = (RK_XFER << RK_XFERBITS) | RK_KFER;
	ctx->m_clauses = 0;
	ctx->m_current_clause = 0;
	
	/* Initialize Kanji conversion engine */
	if( 0 != RkInitialize(NULL) )
	{
		IM_JA_DEBUG("Canna RkInitialize failed\n" );
		return FALSE;
	}

	/* Create a kanji conversion context */
	ctx->m_canna_contextnum = RkCreateContext();
	IM_JA_DEBUG("%d\n", ctx->m_canna_contextnum);
	
	dics = RkGetDicList(ctx->m_canna_contextnum, ctx->m_diclist, BUFFERSIZE);

	if (dics == 0)
	{
		IM_JA_DEBUG("failed to get canna diclist\n");
		return FALSE;
	}
	else if (dics == -1)
	{
		IM_JA_DEBUG("invalid context number.\n");
		return FALSE;
	}
	else
	{
		tmpstr = ctx->m_diclist;
		for (i = 0; i < dics; i++) 
		{
			IM_JA_DEBUG("dics: %d, Diclist: %s\n", i, tmpstr);
			RkMountDic(ctx->m_canna_contextnum,tmpstr,0);
			if (i != dics - 1)
				tmpstr += strlen(tmpstr) + 1;
		}
	}

	cn->im_ja_conv_engine_reset_context = canna_rk_reset;
	cn->im_ja_conv_engine_filter_keypress = canna_rk_filter_keypress;
	cn->im_ja_conv_engine_shutdown = canna_rk_shutdown;
	cn->im_ja_conv_engine_select_candidate = canna_rk_select_candidate;
	cn->im_ja_conv_engine_update_preedit = canna_rk_update_preedit;
	cn->im_ja_conv_engine_commit = canna_rk_commit;

	return TRUE;
}

gboolean canna_rk_filter_keypress(IMJAContext *cn, GdkEventKey *key)
{
	canna_rk_context *ctx = (canna_rk_context*)cn->canna_rk_ctx;
	gchar *tmpstr, *tmpeuc;

	if (key->type == GDK_KEY_RELEASE)
		return FALSE;

	if( (key->keyval == GDK_space) || (ishotkey(key, CONVERT, &cfg) == TRUE) )
	{
		if( (key->keyval == GDK_space) && (strlen(cn->preedit_buf) == 0) ) 
    {
			g_strlcpy(cn->preedit_buf, " ", BUFFERSIZE);
			im_ja_commit(cn); /* Insert a space */
			return TRUE;         
    }
		if( !in_preedit_mode(cn) )
			return FALSE;

		if( in_conversion_mode(ctx) )
		{
			/*
				We are in conversion mode, space will pick the next
				candidate for the current clause, or it may bring up the
				conversion window.
			*/
			ctx->m_consecutive_spacebar_count ++;
			spacebar_next_candidate(cn,ctx);
		}
		else
		{
			char *tmpstr;

			/* do a lastpass-conversion */
			tmpstr = roma2kana_i_lastpass(cn->preedit_buf, &cn->cursor_ndx, cn->input_method);
			/* Start kanji conversion */
			if (g_utf8_validate(tmpstr, -1, NULL) != TRUE)
			{
				IM_JA_DEBUG("utf8_validate failed: lastpass %s\n", tmpstr);
			}
			else
			{
				g_strlcpy(cn->preedit_buf, tmpstr, BUFFERSIZE);
				g_free(tmpstr);
				enter_conversion_mode(cn,ctx);
				ctx->m_consecutive_spacebar_count = 0;
			}
		}

		return TRUE;
	}

	ctx->m_consecutive_spacebar_count = 0;

	if( ishotkey(key, LIST_CANDIDATES, &cfg) == TRUE )
	{
		if( !in_preedit_mode(cn) || !in_conversion_mode(ctx))
			return FALSE;

		list_candidates(cn,ctx);
		return TRUE;
	}

	if (key->keyval == GDK_BackSpace)
	{
		if( in_preedit_mode(cn) )
		{
			if( in_conversion_mode(ctx) )
			{
				unconvert_all_clauses(cn,ctx);
				im_ja_preedit_changed(cn);
				return TRUE;
			}
			
			cn->cursor_ndx -= buffer_bkspchar(cn->preedit_buf,cn->cursor_ndx);
			im_ja_preedit_changed(cn);

			return TRUE;
		}

		return FALSE;
	}

	if (key->keyval == GDK_Delete)
	{
		if( in_preedit_mode(cn) )
		{
			if( in_conversion_mode(ctx) )
			{
				unconvert_all_clauses(cn,ctx);
				im_ja_preedit_changed(cn);
				return TRUE;
			}
			
			/* cn->cursor_ndx doesnt change as a result of this */
			buffer_deltchar(cn->preedit_buf,cn->cursor_ndx);
			im_ja_preedit_changed(cn);

			return TRUE;
		}

		return FALSE;
	}

	if ((key->keyval == GDK_Return) || (key->keyval == GDK_KP_Enter) || (ishotkey(key, COMMIT_PREEDIT, &cfg) == TRUE))
	{
		if( !in_preedit_mode(cn) )
			return FALSE;

		if( in_conversion_mode(ctx) )
		{
			leave_conversion_mode(cn,ctx);
		}

		if( g_utf8_validate(cn->preedit_buf, -1, NULL) != TRUE )
		{
			IM_JA_DEBUG("Cannot commit. utf8_validate failed: %s\n", cn->preedit_buf);
		}
		else 
		{
			im_ja_kana_convert_tailing_n(cn);
			im_ja_commit(cn);
		}
		im_ja_context_reset(cn);
		return TRUE;
	} 

	if( in_preedit_mode(cn) )
	{
		if( in_conversion_mode(ctx) )
		{
			if( ishotkey(key, UNCONVERT, &cfg) == TRUE )
			{
				unconvert_clause(cn,ctx);
				return TRUE;
			}

			if( ishotkey(key, EXPAND_BUNSETSU, &cfg) == TRUE )
			{
				grow_clause(cn,ctx);
				return TRUE;
			}

			if( ishotkey(key, SHRINK_BUNSETSU, &cfg) == TRUE )
			{
				shrink_clause(cn,ctx);
				return TRUE;
			}

			if( ishotkey(key, NEXT_BUNSETSU, &cfg) == TRUE )
			{
				next_clause(cn,ctx);
				return TRUE;
			}

			if( ishotkey(key, PREV_BUNSETSU, &cfg) == TRUE )
			{
				prev_clause(cn,ctx);
				return TRUE;
			}

			if( ishotkey(key, PREV_CANDIDATE, &cfg) == TRUE )
			{
				prev_candidate(cn,ctx);
				return TRUE;
			}

			if( ishotkey(key, NEXT_CANDIDATE, &cfg) == TRUE )
			{
				next_candidate(cn,ctx);
				return TRUE;
			}
			if( key->keyval == GDK_Home )
			{
				goto_clause(cn,ctx,0);
				return TRUE;
			}
			if( key->keyval == GDK_End )
			{
				goto_clause(cn,ctx,ctx->m_num_clauses-1);
				return TRUE;
			}
			if( key->keyval == GDK_Escape )
			{
				unconvert_clause(cn,ctx);
				return TRUE;
			}
		}
		else
		{
			/* In preedit mode but not conversion */
			if( key->keyval == GDK_Left || key->keyval == GDK_KP_Left )
			{
				if( cn->cursor_ndx == 0 )
				{
					cn->cursor_ndx = strlen(cn->preedit_buf);
				}
				else
				{
					char *prev = g_utf8_find_prev_char(cn->preedit_buf,cn->preedit_buf + cn->cursor_ndx);
					if(prev)
						cn->cursor_ndx = prev - cn->preedit_buf;
				}
				im_ja_preedit_changed(cn);
				return TRUE;
			}
			if( key->keyval == GDK_Right || key->keyval == GDK_KP_Right )
			{
				if( cn->preedit_buf[cn->cursor_ndx] == 0 )
				{
					cn->cursor_ndx = 0;
				}
				else
				{
					char *next = g_utf8_find_next_char(cn->preedit_buf + cn->cursor_ndx,0);
					if(next)
						cn->cursor_ndx = next - cn->preedit_buf;
				}
				im_ja_preedit_changed(cn);
				return TRUE;
			}
			if( key->keyval == GDK_Home )
			{
				cn->cursor_ndx = 0;
				im_ja_preedit_changed(cn);
				return TRUE;
			}
			if( key->keyval == GDK_End )
			{
				cn->cursor_ndx = strlen(cn->preedit_buf);
				im_ja_preedit_changed(cn);
				return TRUE;
			}
		}
	}

	if (im_ja_is_printable_key(key) == TRUE)
	{
		gchar utf8strg[7];

		if( in_conversion_mode(ctx) )
		{
			/* exit conversion mode */
			canna_rk_commit(cn);
			im_ja_commit(cn);
			/* unconvert_all_clauses(cn,ctx); */
			
			/* fall through and proceed with regular logic below */
		}

		if( cn->cursor_ndx == -1 )
		{
			cn->cursor_ndx = strlen(cn->preedit_buf);
		}

		utf8strg[g_unichar_to_utf8(gdk_keyval_to_unicode(key->keyval), utf8strg)] = 0;

		IM_JA_DEBUG("inserting \"%s\" into \"%s\" at %d\n", 
				utf8strg, cn->preedit_buf, cn->cursor_ndx );

		tmpeuc = utf82euc(utf8strg);
		if ((unsigned int)tmpeuc[0] > 128) {
			g_free(tmpeuc);
			IM_JA_DEBUG("*ERROR* non-ascii input\n");
			im_ja_input_utf8(cn, utf8strg);
			return TRUE;
		}

		cn->cursor_ndx += 
			buffer_inschar(cn->preedit_buf, BUFFERSIZE, utf8strg, cn->cursor_ndx);

		IM_JA_DEBUG("converting \"%s\" to kana:", cn->preedit_buf);
		tmpstr = roma2kana_i(cn->preedit_buf, &cn->cursor_ndx, cn->input_method);
		IM_JA_DEBUG("\"%s\" \n", tmpstr);

		if (g_utf8_validate(tmpstr, -1, NULL) != TRUE)
		{
			IM_JA_DEBUG("utf8_validate failed: %s\n", tmpstr);
		}
		else
		{
			g_strlcpy(cn->preedit_buf, tmpstr, BUFFERSIZE);
			g_free(tmpstr);
			im_ja_preedit_changed(cn);
		}

		return TRUE;
	}

	return FALSE; /* Didn't handle the keypress */
}

void canna_rk_reset(IMJAContext *cn)
{
	canna_rk_context *ctx = (canna_rk_context*)cn->canna_rk_ctx;

	if( in_conversion_mode(ctx) )
	{
		ctx->m_num_clauses=-1;
		/* a parameter of 0 means not to learn from this conversion */
		RkEndBun(ctx->m_canna_contextnum,0);
	}

	cn->cursor_ndx = -1;
}

/* this should only be called from within preedit mode */
static void enter_conversion_mode(IMJAContext *cn, canna_rk_context *ctx)
{
	gchar *temp;

	/* Convert preedit to EUC-jp */
	temp = utf82euc(cn->preedit_buf);
	if( temp )
	{
		g_strlcpy(ctx->m_conv_euc, temp, BUFFERSIZE);
		g_free(temp);

		IM_JA_DEBUG("Entering conversion mode: %s\n", cn->preedit_buf);
		ctx->m_num_clauses =
			RkBgnBun(ctx->m_canna_contextnum, ctx->m_conv_euc, strlen(ctx->m_conv_euc), ctx->m_canna_rk_mode);

		if( ctx->m_num_clauses >= 0 )
		{
			/* We are now in conversion mode */
			update_clauses(cn,ctx,0);
		}
	}
}

static void update_status(IMJAContext *cn, canna_rk_context *ctx)
{
	RkStat stat;

	if( !in_conversion_mode(ctx) )
		return;

	if( 0 == RkGetStat(ctx->m_canna_contextnum, &stat ) )
	{
		/* We are now in conversion mode */
		ctx->m_current_clause_number = stat.bunnum;
		ctx->m_current_candidate_number = stat.candnum;
		ctx->m_max_current_candidate_number = stat.maxcand;
		IM_JA_DEBUG("RkGetStat success, stat.ylen=%d, stat.klen=%d, stat.tlen=%d\n",
				stat.ylen, stat.klen, stat.tlen );

		/* We are updating the status because something changed, so sync the display */
		
		/*
			FIXME: update candidate window position so that it will be placed
			under the current converting bunsetsu, and not at the start of the preedit.
		*/
		set_preedit_from_conversion(cn,ctx);
	}
	else
	{
		IM_JA_DEBUG("RkGetStat failed, leaving conversion mode\n");
		/* leave_conversion_mode(cn,ctx); */
		canna_rk_reset(cn);

		im_ja_preedit_changed(cn);
	}
}

static void update_clauses(IMJAContext *cn, canna_rk_context *ctx, int sel_clause )
{
	int at;

	if(sel_clause >= ctx->m_num_clauses)
		sel_clause=0;
	
	/* Delete any leftover clauses */
	if( ctx->m_clauses != NULL)
	{
		GList *iter = ctx->m_clauses;
		while( iter )
		{
			canna_rk_clause *prk = (canna_rk_clause*)iter->data;
			if(prk->m_utf8_kanji)
			{
				g_free(prk->m_utf8_kanji);
				prk->m_utf8_kanji=0;
			}
			
			g_free(iter->data);
			iter->data=0;
			iter = g_list_next(iter);
		}
		
		g_list_free(ctx->m_clauses);
		ctx->m_clauses = NULL;
	}
	
	if( ctx->m_num_clauses < 1 )
	{
		IM_JA_DEBUG("Lost Conversion mode!!!\n" );
		ctx->m_num_clauses=-1; /* FIXME im_ja_context_reset() here ?? */
		RkEndBun(ctx->m_canna_contextnum,0);
		cn->preedit_buf[0]=0;
		cn->cursor_ndx=0;
		cn->preedit_reverse_start = cn->preedit_reverse_end = 0;
		im_ja_preedit_changed(cn);
		return;
	}
	
	IM_JA_DEBUG("Detecting clauses (should be %d):\n", ctx->m_num_clauses );
	/* Iterate through clauses */
	for(at=ctx->m_num_clauses-1; at >= 0 ; --at )
	{
		canna_rk_clause *newclause;
		unsigned char euc_buf[BUFFERSIZE];
		int euclen; 
		int retval;
		gboolean done = FALSE;

		retval = RkGoTo(ctx->m_canna_contextnum,at);
		
		while (!done) {
			euclen = RkGetKanji(ctx->m_canna_contextnum,euc_buf,BUFFERSIZE);
			IM_JA_DEBUG(" [RkGetKanji] candnum: %d, length %d, result: %s\n", at, euclen, euc_buf);
			/*if(euclen<1)
				euclen = RkGetYomi(ctx->m_canna_contextnum,euc_buf,BUFFERSIZE); */

			newclause = g_new(canna_rk_clause,1);
			newclause->m_utf8_kanji = euc2utf8((gchar*)euc_buf);
			/*IM_JA_DEBUG("\t%s (euclen=%d)\n", newclause->m_utf8_kanji, euclen );*/
			if (newclause->m_utf8_kanji == NULL) {
				retval = RkNext(ctx->m_canna_contextnum);
				if (retval == 0) {
					done = TRUE;
					newclause->m_utf8_kanji = g_strdup("???");
				}
			}
			else {
				done = TRUE;
				ctx->m_clauses = g_list_prepend(ctx->m_clauses,newclause);
			}
		}

		RkGoTo(ctx->m_canna_contextnum,sel_clause);
	}

	update_status(cn,ctx);
}

/* this should only be called from within conversion mode */
static void set_preedit_from_conversion(IMJAContext *cn, canna_rk_context *ctx)
{
	int at, count;
	GList *iter;

	cn->preedit_buf[0]=0;
	cn->preedit_reverse_start = cn->preedit_reverse_end = 0;
	cn->cursor_ndx=0;

	at=count=0;
	iter = ctx->m_clauses;
	while( iter )
	{
		canna_rk_clause *prk = (canna_rk_clause*)iter->data;

		if( at == ctx->m_current_clause_number )
		{
			cn->preedit_reverse_start = count;
			cn->cursor_ndx=count;
			ctx->m_current_clause=iter->data;
		}

		count = g_strlcat(cn->preedit_buf, prk->m_utf8_kanji, BUFFERSIZE);

		if( at == ctx->m_current_clause_number )
		{
			cn->preedit_reverse_end = count;
			/*cn->cursor_ndx=count;*/

			IM_JA_DEBUG("Highligted kanji \"%s\" %d-%d\n",
					prk->m_utf8_kanji,cn->preedit_reverse_start,cn->preedit_reverse_end);
		}

		++at;
		iter = g_list_next(iter);
	}

	im_ja_preedit_changed(cn);
}

/* this should only be called from within conversion mode */
static void leave_conversion_mode(IMJAContext *cn, canna_rk_context *ctx)
{
	int at;
	GList *iter;

	/* Harvest final state */
	cn->preedit_buf[0]=0;
	cn->preedit_reverse_start = cn->preedit_reverse_end = 0;

	at=0;
	iter = ctx->m_clauses;
	while( iter )
	{
		canna_rk_clause *prk = (canna_rk_clause*)iter->data;
		g_strlcat(cn->preedit_buf, prk->m_utf8_kanji, BUFFERSIZE);

		++at;
		iter = g_list_next(iter);
	}

	cn->cursor_ndx=-1;
	ctx->m_num_clauses=-1;
	/* a parameter of 1 means to learn from this conversion */
	RkEndBun(ctx->m_canna_contextnum,1);
}

/* this should only be called from within conversion mode */
static void list_candidates(IMJAContext *cn, canna_rk_context *ctx)
{
	unsigned char cand_buf[BUFFERSIZE];
	int ncand,at;
	unsigned char *pcand;
	gchar *tmpcand;
	
	/* Release old candidates */
	im_ja_free_candidate_list(cn);

	/* Get list of candidates */
	ncand = RkGetKanjiList(ctx->m_canna_contextnum, cand_buf, BUFFERSIZE);
	pcand=cand_buf;

	IM_JA_DEBUG("Building candidate window: (%d)\n", ncand );
	for( at=0; at<ncand; ++at)
	{
		int len = strlen((char *)pcand);
		IM_JA_DEBUG(" [RkGetKanjiList] candnum: %d, length %d, result: %s\n", at, len, (gchar *)pcand);
		tmpcand = euc2utf8((gchar *)pcand);
		/*IM_JA_DEBUG("\t(len:%d)%s\n", len, tmpcand);*/
		cn->candidate_list = g_list_append(cn->candidate_list, tmpcand);
		pcand+=len+1;
	}
	/*
		TODO: ON CANCEL we need to redraw from the current clauses, because
		the candidate window corrupts our preedit string
	*/
	candidate_window_show(cn, ctx->m_current_candidate_number);
}
static void spacebar_next_candidate(IMJAContext *cn, canna_rk_context *ctx)
{
	/*
		We are in conversion mode, and now wish to switch our current candidate
		with the next possible one.
	*/
	/* if threshhold exceeded, spawn candidate window */
	if(ctx->m_consecutive_spacebar_count >= 2)
	{
		list_candidates(cn,ctx);
	}
	else
		next_candidate(cn,ctx);
}
static void next_candidate(IMJAContext *cn, canna_rk_context *ctx)
{
	int candno = ctx->m_current_candidate_number + 1;

	if(candno >= ctx->m_max_current_candidate_number )
		candno = 0;

	goto_candidate(cn,ctx,candno);
}
static void prev_candidate(IMJAContext *cn, canna_rk_context *ctx)
{
	int candno = ctx->m_current_candidate_number - 1;

	if(candno < 0 )
		candno += ctx->m_max_current_candidate_number;

	goto_candidate(cn,ctx,candno);
}
static void goto_candidate(IMJAContext *cn, canna_rk_context *ctx, int candno)
{
	unsigned char euc_buf[BUFFERSIZE];
	
	ctx->m_current_candidate_number = RkXfer(ctx->m_canna_contextnum,candno);
	IM_JA_DEBUG("Selecting candidate number %d\n", ctx->m_current_candidate_number );

	g_free(ctx->m_current_clause->m_utf8_kanji);
	RkGetKanji(ctx->m_canna_contextnum,euc_buf,BUFFERSIZE);
	ctx->m_current_clause->m_utf8_kanji = euc2utf8((gchar *)euc_buf);
	if (ctx->m_current_clause->m_utf8_kanji == NULL) ctx->m_current_clause->m_utf8_kanji = g_strdup("???");

	set_preedit_from_conversion(cn,ctx);
}
void canna_rk_select_candidate(IMJAContext *cn, int candno)
{
	canna_rk_context *ctx = (canna_rk_context*)cn->canna_rk_ctx;
	if(candno >= 0 && candno < ctx->m_max_current_candidate_number)
	{
		goto_candidate(cn,ctx,candno);
	}
}

static void next_clause(IMJAContext *cn, canna_rk_context *ctx)
{
	int newclause = ctx->m_current_clause_number + 1;
	if( newclause >= ctx->m_num_clauses)
		newclause = 0;
	goto_clause(cn,ctx,newclause);
}
static void prev_clause(IMJAContext *cn, canna_rk_context *ctx)
{
	int newclause = ctx->m_current_clause_number - 1;
	if( newclause < 0 )
		newclause += ctx->m_num_clauses;
	goto_clause(cn,ctx,newclause);
}
static void goto_clause(IMJAContext *cn, canna_rk_context *ctx, int clause)
{
	cn->update_candwin_pos = TRUE;
	ctx->m_current_clause_number = RkGoTo(ctx->m_canna_contextnum,clause);
	IM_JA_DEBUG("Selecting clause number %d\n", ctx->m_current_clause_number );
	update_status(cn,ctx);
}
static void shrink_clause(IMJAContext *cn, canna_rk_context *ctx)
{
	IM_JA_DEBUG("Shrinking current clause\n" );
	RkNfer(ctx->m_canna_contextnum);
	ctx->m_num_clauses = RkShorten(ctx->m_canna_contextnum);
	update_clauses(cn,ctx,ctx->m_current_clause_number);
}
static void grow_clause(IMJAContext *cn, canna_rk_context *ctx)
{
	IM_JA_DEBUG("Growing current clause\n" );
	RkNfer(ctx->m_canna_contextnum);
	ctx->m_num_clauses = RkEnlarge(ctx->m_canna_contextnum);
	update_clauses(cn,ctx,ctx->m_current_clause_number);
}

static void unconvert_clause(IMJAContext *cn, canna_rk_context *ctx)
{
	unsigned char euc_buf[BUFFERSIZE];
	gchar *old;

	IM_JA_DEBUG("Unconverting current clause\n" );
	RkNfer(ctx->m_canna_contextnum);
	
	old = ctx->m_current_clause->m_utf8_kanji;
	RkGetKanji(ctx->m_canna_contextnum,euc_buf,BUFFERSIZE);
	ctx->m_current_clause->m_utf8_kanji = euc2utf8((gchar *)euc_buf);

	if( 0 == strcmp(old,ctx->m_current_clause->m_utf8_kanji) )
	{
		/* 
			 The current clause was already unconverted,
			 lets unconvert the whole thing and exit conversion mode
		*/
		unconvert_all_clauses(cn,ctx);
		im_ja_preedit_changed(cn);
	}
	else
	{
		/* call update_status to discover what our current candidate number is now */
		update_status(cn,ctx);
	}
	
	g_free(old);
}

static void unconvert_all_clauses(IMJAContext *cn, canna_rk_context *ctx)
{
	unsigned char euc_buf[BUFFERSIZE];
	gchar *tmp;
	int euclen;
	int at,count;

	cn->preedit_buf[0]=0;
	cn->preedit_reverse_start = cn->preedit_reverse_end = 0;

	for(at=0; at < ctx->m_num_clauses ; ++at )
	{
		RkGoTo(ctx->m_canna_contextnum,at);
		
		euclen = RkGetYomi(ctx->m_canna_contextnum,euc_buf,BUFFERSIZE);

		tmp = euc2utf8((gchar *)euc_buf);
		
		count = g_strlcat(cn->preedit_buf, tmp, BUFFERSIZE);

		if(ctx->m_current_clause_number == at)
		{
			/* current cursor pos = count */
			cn->cursor_ndx=count;
		}

		g_free(tmp);
	}

	/* We now return to pre-edit mode, which is essentially a hiragana mode */
	ctx->m_num_clauses=-1;

	/* a parameter of 0 means to not learn from this conversion */
	RkEndBun(ctx->m_canna_contextnum,0);
}

