/* $Id: ifp.c,v 1.44 2005/12/11 01:41:13 yamajun Exp $ */

static char const rcsid[] = "$Id: ifp.c,v 1.44 2005/12/11 01:41:13 yamajun Exp $";

#include <stdio.h>
#include <string.h>
#include <usb.h>

#include <limits.h>

#include "config.h"		// auto-generated by configure script.
#include "ifp_routines.h"

void help(char *prog);
void version(char *prog);

extern int IFP_BULK_TO;
extern int IFP_BULK_FROM;
extern int isOldIFP;

/* 
 * Note: argv for mc
 * Don't use argv[2] for mc plugin 
 *
 * command:	"ifp"	    "copyin"	"hoge.mp3"  "/foo.mp3"
 * real ifp:	argv[0],    argv[1],	argv[2],    argv[3], ... argv[argc-1];
 * for mc:	argv[0],    argv[1],		    argv[3], ... argv[argc-1];
 */

int main(int argc, char **argv)
{
    struct usb_bus *busses;
    struct usb_bus *bus;
    struct usb_device *tmp_dev;
    struct usb_device *dev = NULL;
    struct usb_endpoint_descriptor ep;
    usb_dev_handle *dh;

    FILE *fp;
    int is_mc;
    int retval = 0;
    char buf[255];

    /* check for mc (run from mc's extfs directory ? */
    if (strstr(argv[0],"/mc/extfs/"))
	is_mc = IS_MC; else is_mc = IS_IFP;

    /* no command, or help options(help | -h | -help | --help). */
    if (argc < 2 || strstr(argv[1], "help") || !strcmp("-h", argv[1])) {
	help(argv[0]);
	return 0;
    // "version" command
    } else if ( argv[1] == strstr(argv[1], "version") ) {
	version(argv[0]);
	return 0;
    }

    usb_init();

    if(usb_find_busses() < 0) {
	fprintf(stderr, "Could not find any USB busses.\n");
	return 1;
    }

    if (usb_find_devices() < 0) {
	fprintf(stderr, "USB devices not found(nor hubs!).\n");
	fprintf(stderr, "You must run as root.\n");
	fprintf(stderr, "Note: This program cannot work with UMS firmware.\n");
	return 1;
    }

    /* C calling convention, it's not nice to use global stuff */
    busses = usb_get_busses();

    for (bus = busses; bus; bus = bus->next) {
	for (tmp_dev = bus->devices; tmp_dev; tmp_dev = tmp_dev->next) {
	    // reject UMS firmware.
	    if (tmp_dev->descriptor.idVendor == iRiver_Vendor &&
		((tmp_dev->descriptor.idProduct & 0xFF00) != 0x1100) ) {

		ep = tmp_dev->config[0].interface[0].altsetting[0].endpoint[0];
		if ((ep.bEndpointAddress & 0x80) == 0x00) {
		    IFP_BULK_TO = ep.bEndpointAddress;
		} else {
		    IFP_BULK_FROM = ep.bEndpointAddress;
		}

		ep = tmp_dev->config[0].interface[0].altsetting[0].endpoint[1];
		if ((ep.bEndpointAddress & 0x80) == 0x00) {
		    IFP_BULK_TO = ep.bEndpointAddress;
		} else {
		    IFP_BULK_FROM = ep.bEndpointAddress;
		}

		if (IFP_BULK_TO == 0x03 && IFP_BULK_FROM == 0x83) {
		    isOldIFP = 1;
		}

		dev = tmp_dev;
		goto IFPFOUND;

	    }
	}
    }

    if (dev == NULL) {
	fprintf(stderr, "iRiver iFP device not found.\n");
	fprintf(stderr, "Note: Please check USB connection.\n");
	return 1;
    }

    IFPFOUND:
    if ( (dh = usb_open(dev)) == NULL) {
	fprintf(stderr,"Unable to open iFP device.\n");
	return 1;
    }

    /* "must be called" written in the libusb documentation */
    if (usb_claim_interface(dh, dev->config->interface->altsetting->bInterfaceNumber)) {
	fprintf(stderr, "Device is busy.  (I was unable to claim its interface.)\n");
	usb_close(dh);
	return 1;
    }

    if (!ifp_check_connect(dh)) {
	fprintf(stderr, "Unable to connect to iFP.\n");
	fprintf(stderr, "You must run as root.\n");
	usb_close(dh);
	return 1;
    }

    /* parse command */
    if (strcmp(argv[1], "ls") == 0) {
	if (argc < 3) {
	    retval = ifp_list_dir_printf(dh, "/", 0, 0);
	} else {
	    retval = ifp_list_dir_printf(dh, argv[2], 0, 0);
	}

    } else if (strcmp(argv[1], "df") == 0) {
	printf("Capacity: %d bytes\n", ifp_get_capacity(dh));
	printf("Free:     %d bytes\n", ifp_get_free(dh));

    } else if (strcmp(argv[1], "put") == 0) {
	retval = ifp_put(dh, argc - 1, argv + 1);

    } else if (strcmp(argv[1], "upload") == 0) {
	retval = ifp_upload(dh, argc - 1, argv + 1);

    } else if (strcmp(argv[1], "get") == 0) {
	retval = ifp_get(dh, argc - 1, argv + 1);

    } else if (strcmp(argv[1], "download") == 0) {
	retval = ifp_download(dh, argc - 1, argv + 1);

    } else if (strcmp(argv[1], "rm") == 0) {
	retval = ifp_rm(dh, is_mc, argc - 1, argv + 1);

    } else if (strcmp(argv[1], "rmdir") == 0) {
	retval = ifp_rmdir(dh, is_mc, argc - 1, argv + 1);

    } else if (strcmp(argv[1], "mkdir") == 0) {
	retval = ifp_mkdir(dh, is_mc, argc - 1, argv + 1);

    } else if (strcmp(argv[1], "battery") == 0) {
	printf("Battery status (really not sure!): %d\n",
		ifp_battery_condition(dh));

    } else if (strcmp(argv[1], "typestring") == 0) {
	retval = ifp_type_string(dh, buf);
	printf("iFP type string: %s\n", buf);

    } else if (strcmp(argv[1], "firmversion") == 0) {
	printf("Firmware version (really not sure!): %.2f\n",
		((float)ifp_firmware_version(dh))/100.00 );

    } else if (strcmp(argv[1], "firmupdate") == 0) {
	retval = ifp_firmware_update(dh, argc - 1, argv + 1);

    } else if (strcmp(argv[1], "format") == 0) {
	retval = ifp_format(dh);

    } else if (strcmp(argv[1], "getpreset") == 0) {
	retval = ifp_get_tuner_preset(dh, argc - 1, argv + 1);

    } else if (strcmp(argv[1], "setpreset") == 0) {
	retval = ifp_set_tuner_preset(dh, argc - 1, argv + 1);

    // Functions for mc support
    } else if (strcmp(argv[1], "copyin") == 0) {
	if (argc < 5) {
	    retval = -1;
	} else {
	    // suppress to upload file with name of MC_INFO
	    if (!strstr(argv[3], MC_INFO)) { 
		if ( (fp = fopen(argv[4], "r")) == NULL) {
		    perror(argv[4]);
		    retval = -1;
		} else {
		    retval = ifp_upload_file(dh, IS_MC, argv[3], fp);
		    fclose(fp);
		}
		if (retval <= -2) fprintf(stderr, "Not enough space on iFP device.\n");
	    }
	}

    } else if (strcmp(argv[1], "copyout") == 0) {
	if (argc < 5) {
	    retval = -1;
	} else {
	    if (strcmp(argv[3], MC_INFO) == 0) {
		if ( (fp = fopen(argv[4], "w")) == NULL) {
		    perror(argv[4]);
		    retval = -1;
		} else {
		    info(dh, fp, 1);
		    fclose(fp);
		}

	    } else {
		// file to file only
	        retval = ifp_download_file(dh, IS_MC, argv[3], argv[4]);
	    }
	}

    } else if (strcmp(argv[1], "list") == 0) {
	// faked ls -l
	/* faked file containing generated information about iFP */
	printf("-r-xr-xr-x  1 root  root        0 Jan  1 1980 " MC_INFO "\n");
	/* ls listing (all files and directories) */
	retval = ifp_list_dir_printf(dh, "/", 1, 1);

    } else if (strcmp(argv[1], "run") == 0) {
	if (argc >= 4) {
	    if (strstr(argv[3], MC_INFO)) {
		info(dh, stdout, 0);
		printf("### Press enter ###\n");
		getchar();
	    }
	}

    } else {
	help(argv[0]);
    }

    /* release claimed interface */
    usb_release_interface(dh, dev->config->interface->altsetting->bInterfaceNumber);

//#ifdef linux 
//    usb_reset(dh);	// Don't use for Mac OS X.
//#endif

    usb_close(dh);
    return (retval < 0) ? 1 : 0;

}

void help(char *prog) {
    printf("usage: %s [-h] command [params ...]\n",prog);
    printf("commands:\n");
    printf("    ls [directory]\n");
    printf("    df\n");
    printf("    upload localfile ifptarget\n");
    printf("    upload localdir  ifpdir\n");
    printf("    put localfile\n");
    printf("    put localdir\n");
    printf("    download ifpfile localtarget\n");
    printf("    download ifpdir  localdir\n");
    printf("    get ifpfile\n");
    printf("    get ifpdir\n");
    printf("    rm [-r] file\n");
    printf("    rmdir dir\n");
    printf("    mkdir dir\n");
    printf("    battery\n");
    printf("    typestring\n");
    printf("    firmversion\n");
    printf("    format\n");
    printf("    firmupdate /path/to/FIRMWARE.HEX\n");
    printf("    help\n");
    printf("    version\n");
    printf("\n");
    printf("Note: This program cannot work with UMS firmware.\n");
}

void version(char *prog) {
#ifdef HAVE_ICONV
    printf("%s: File manager for iRiver iFP music player. (version %s with iconv)\n", prog, VERSION);
#else
    printf("%s: File manager for iRiver iFP music player. (version %s)\n", prog, VERSION);
#endif	// HAVE_ICONV
    printf("This is free software which licensed under the GNU Public License.\n");
    printf("This is absolutely no warranty.\n");
}

