//
// Copyright (c) 2018 Ted Unangst <tedu@tedunangst.com>
//
// Permission to use, copy, modify, and distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

package main

import (
	"bytes"
	"crypto/sha512"
	"fmt"
	"html"
	"html/template"
	"io"
	"io/ioutil"
	"net"
	"net/http"
	"os"
	"os/exec"
	"regexp"
	"sort"
	"strings"
	"sync"
	"time"

	"github.com/gorilla/mux"
	"humungus.tedunangst.com/r/gerc"
	_ "humungus.tedunangst.com/r/go-sqlite3"
	"humungus.tedunangst.com/r/webs/cache"
	"humungus.tedunangst.com/r/webs/rss"
	"humungus.tedunangst.com/r/webs/templates"
)

var savedstyleparam string

func getstyleparam() string {
	if savedstyleparam != "" {
		return savedstyleparam
	}
	data, _ := ioutil.ReadFile("views/style.css")
	hasher := sha512.New()
	hasher.Write(data)
	return fmt.Sprintf("?v=%.8x", hasher.Sum(nil))
}

func openListener() (net.Listener, error) {
	var listenAddr string
	err := getconfig("listenaddr", &listenAddr)
	if err != nil {
		return nil, err
	}
	if listenAddr == "" {
		return nil, fmt.Errorf("must have listenaddr")
	}
	dlog.Printf("starting web server on %s", listenAddr)
	proto := "tcp"
	if listenAddr[0] == '/' {
		proto = "unix"
		err := os.Remove(listenAddr)
		if err != nil && !os.IsNotExist(err) {
			elog.Printf("unable to unlink socket: %s", err)
		}
	}
	listener, err := net.Listen(proto, listenAddr)
	if err != nil {
		return nil, err
	}
	if proto == "unix" {
		os.Chmod(listenAddr, 0777)
	}
	return listener, nil
}

func getInfo(r *http.Request) map[string]interface{} {
	templinfo := make(map[string]interface{})
	templinfo["StyleParam"] = getstyleparam()
	templinfo["Title"] = serverName
	return templinfo
}

func errorPage(w http.ResponseWriter, r *http.Request, msg string) {
	w.Header().Set("Cache-Control", "max-age=60")
	w.WriteHeader(500)
	templinfo := getInfo(r)
	templinfo["ErrorMessage"] = msg
	err := views.Execute(w, "error.html", templinfo)
	if err != nil {
		elog.Printf("template execution: %s", err)
	}
}

var re_allcaps = regexp.MustCompile("^[A-Z]+$")

func filelisting(repo *gerc.Repo, reponame string, path string) []string {
	files, _ := repo.GetFiles(gerc.FilesArgs{Filenames: []string{path}})
	dirname := ""
	prefix := path
	lines := make([]string, 0, len(files))
	for _, file := range files {
		line := file.Name
		if line == ".hgtags" {
			continue
		}
		line = strings.TrimPrefix(line, prefix)
		if idx := strings.IndexByte(line, '/'); idx != -1 {
			if line[0:idx+1] == dirname {
				continue
			}
			dirname = line[0 : idx+1]
			lines = append(lines, path+dirname)
			continue
		}
		lines = append(lines, path+line)
	}
	sort.Slice(lines, func(i, j int) bool {
		dirA := lines[i][len(lines[i])-1] == '/'
		dirB := lines[j][len(lines[j])-1] == '/'
		if dirA != dirB {
			return dirA
		}
		dotA := lines[i][0] == '.'
		dotB := lines[j][0] == '.'
		if dotA != dotB {
			return dotB
		}
		capsA := re_allcaps.MatchString(lines[i])
		capsB := re_allcaps.MatchString(lines[j])
		if capsA != capsB {
			return capsA
		}
		return lines[i] < lines[j]
	})

	return lines
}

type Rev struct {
	Num int
	Hex string
}

type Change struct {
	Changeset Rev
	Parents   []Rev
	Branch    string
	Tag       string
	User      string
	Date      time.Time
	Summary   string
	Message   []string
}

func changelisting(conn *gerc.Repo, changeset string, filename string, limit int) []*Change {
	if changeset == "" {
		changeset = "tip"
	}
	if strings.IndexByte(changeset, ':') == -1 {
		changeset += ":0"
	}
	if limit > 0 {
		changeset = fmt.Sprintf("limit(%s, %d)", changeset, limit)
	}

	gchanges, err := conn.GetChanges(gerc.ChangesArgs{Revisions: changeset, Filename: filename})
	if err != nil {
		elog.Printf("error getting changes: %s\n", err)
		return nil
	}
	changes := make([]*Change, 0, len(gchanges))
	for _, g := range gchanges {
		var parents []Rev
		if g.P1rev != g.Linkrev-1 {
			parents = append(parents, Rev{g.P1rev, fmt.Sprintf("%.6x", g.P1node)})
		}
		if g.P2rev != -1 {
			parents = append(parents, Rev{g.P2rev, fmt.Sprintf("%.6x", g.P2node)})
		}
		tag := ""
		if len(g.Tags) > 0 {
			tag = strings.Join(g.Tags, " ")
		}
		changes = append(changes, &Change{
			Changeset: Rev{g.Linkrev, fmt.Sprintf("%.6x", g.NodeID)},
			Parents:   parents,
			//Branch:    branch,
			Tag:     tag,
			User:    g.User,
			Date:    g.Date,
			Summary: g.Summary,
			Message: g.Message,
		})
	}
	return changes
}

func recentchangelisting(conn *gerc.Repo) []*Change {
	changes := changelisting(conn, "date(yesterday):", "", 0)
	if len(changes) < 10 {
		changes = changelisting(conn, "", "", 10)
	} else {
		// reverse it
		for i, j := 0, len(changes)-1; i < j; i, j = i+1, j-1 {
			changes[i], changes[j] = changes[j], changes[i]
		}
	}
	return changes
}

func getreadme(conn *gerc.Repo) string {
	readmenames := []string{"README", "README.txt", "README.md"}
	files, err := conn.GetFiles(gerc.FilesArgs{Filenames: readmenames})
	if err != nil || len(files) == 0 {
		return ""
	}
	data, err := conn.GetFileData(gerc.FileDataArgs{Filename: files[0].Name})
	if err != nil || len(data) == 0 {
		return ""
	}
	return string(data)
}

func getscreenshot(conn *gerc.Repo) string {
	names := []string{"screenshot.png", "screenshot.jpg", "docs/screenshot.png", "docs/screenshot.jpg"}
	files, err := conn.GetFiles(gerc.FilesArgs{Filenames: names})
	if err != nil || len(files) == 0 {
		return ""
	}
	return files[0].Name
}

var views *templates.Template

func showrepo(w http.ResponseWriter, r *http.Request) {
	if strings.HasPrefix(r.Header.Get("User-Agent"), "mercurial/proto-1.0") {
		hgpassthru(w, r)
		return
	}
	reponame := mux.Vars(r)["reponame"]
	conn := getgerc(reponame)
	if conn == nil {
		http.NotFound(w, r)
		return
	}
	defer putgerc(conn)

	if enableFedi && isAPrequest(r) {
		j := jsonRepository(reponame)
		w.Header().Set("Content-Type", apContentType)
		j.Write(w)
		return
	}

	changes := recentchangelisting(conn)
	if changes == nil {
		errorPage(w, r, "unable to get recent changes")
		return
	}

	w.Header().Set("Cache-Control", "max-age=30")

	templinfo := getInfo(r)
	templinfo["Title"] = fmt.Sprintf("%s - %s", reponame, serverName)
	templinfo["ServerName"] = serverName
	templinfo["Repo"] = reponame
	templinfo["Readme"] = getreadme(conn)
	templinfo["Screenshot"] = getscreenshot(conn)
	templinfo["Changes"] = changes
	templinfo["HasDownloads"] = hasdownloads(reponame)
	templinfo["HasManual"] = len(manualfiles(conn, reponame)) > 0
	err := views.Execute(w, "overview.html", templinfo)
	if err != nil {
		elog.Printf("template execution: %s", err)
	}
}

func showfiles(w http.ResponseWriter, r *http.Request) {
	reponame := mux.Vars(r)["reponame"]
	filename := mux.Vars(r)["filename"]
	changeset := mux.Vars(r)["changeset"]
	if changeset == "" {
		changeset = "tip"
	}
	repo := getgerc(reponame)
	if repo == nil {
		http.NotFound(w, r)
		return
	}
	defer putgerc(repo)

	path := strings.Split(filename, "/")
	basename := path[len(path)-1]
	path = path[:len(path)-1]

	w.Header().Set("Cache-Control", "max-age=30")

	templinfo := getInfo(r)
	templinfo["Title"] = fmt.Sprintf("%s/%s - %s", reponame, filename, serverName)
	templinfo["Repo"] = reponame
	templinfo["Changeset"] = changeset
	templinfo["Path"] = path
	templinfo["Basename"] = basename
	templinfo["Files"] = filelisting(repo, reponame, filename)
	err := views.Execute(w, "files.html", templinfo)
	if err != nil {
		elog.Printf("template execution: %s", err)
	}
}

var dlfilescache = make(map[string][]os.FileInfo)
var dlfilescachemtx sync.Mutex
var dlfilescachetime time.Time

var dlcache = cache.New(cache.Options{Filler: func(reponame string) ([]os.FileInfo, bool) {
	dirname := "downloads/" + reponame
	dir, err := os.Open(dirname)
	var files []os.FileInfo
	if err == nil {
		defer dir.Close()
		names, _ := dir.Readdirnames(0)
		files = make([]os.FileInfo, 0, len(names))
		for _, name := range names {
			fi, err := os.Stat(dirname + "/" + name)
			if err == nil {
				files = append(files, fi)
			}
		}
	}
	sort.Slice(files, func(i, j int) bool { return files[i].ModTime().After(files[j].ModTime()) })
	dlfilescache[reponame] = files
	return files, true
}, Duration: time.Minute})

func finddownloads(reponame string) []os.FileInfo {
	var files []os.FileInfo
	dlcache.Get(reponame, &files)
	return files
}

func hasdownloads(reponame string) bool {
	return len(finddownloads(reponame)) > 0
}

func showdownloads(w http.ResponseWriter, r *http.Request) {
	reponame := mux.Vars(r)["reponame"]
	conn := getgerc(reponame)
	if conn == nil {
		http.NotFound(w, r)
		return
	}
	defer putgerc(conn)

	files := finddownloads(reponame)

	w.Header().Set("Cache-Control", "max-age=30")

	templinfo := getInfo(r)
	templinfo["Title"] = fmt.Sprintf("%s downloads - %s", reponame, serverName)
	templinfo["Repo"] = reponame
	templinfo["Files"] = files
	err := views.Execute(w, "downloads.html", templinfo)
	if err != nil {
		elog.Printf("template execution: %s", err)
	}
}

func showdownloadrss(w http.ResponseWriter, r *http.Request, reponame string) {
	files := finddownloads(reponame)

	feed := rss.Feed{
		Title:       reponame + " downloads",
		Description: "downloads for " + reponame,
		Link:        fmt.Sprintf("https://%s/r/%s/d", serverName, reponame),
	}

	feed.Items = make([]*rss.Item, 0, len(files))
	for _, f := range files {
		desc := fmt.Sprintf("<p>file: %s (%d bytes) - %s\n", f.Name(), f.Size(), f.ModTime().UTC().Format("2006-01-02 15:04:05"))
		feed.Items = append(feed.Items, &rss.Item{
			Title:       f.Name(),
			Description: rss.CData{desc},
			Link:        fmt.Sprintf("https://%s/r/%s/d/%s", serverName, reponame, f.Name()),
			PubDate:     f.ModTime().UTC().Format(time.RFC1123),
		})
	}

	w.Header().Set("Cache-Control", "max-age=300")

	err := feed.Write(w)
	if err != nil {
		elog.Printf("rss error: %s", err)
	}
}

func downloadfile(w http.ResponseWriter, r *http.Request) {
	if isAPrequest(r) {
		http.Error(w, "there are no activities here", http.StatusNotAcceptable)
		return
	}
	reponame := mux.Vars(r)["reponame"]
	filename := mux.Vars(r)["filename"]
	conn := getgerc(reponame)
	if conn == nil {
		http.NotFound(w, r)
		return
	}
	defer putgerc(conn)

	if filename == "rss" {
		showdownloadrss(w, r, reponame)
		return
	}

	dlog.Printf("download of file %s", filename)

	fd, err := os.Open("downloads/" + reponame + "/" + filename)
	if err != nil {
		http.NotFound(w, r)
		return
	}
	defer fd.Close()

	w.Header().Set("Content-Type", "application/binary")
	w.Header().Set("X-Content-Type-Options", "nosniff")
	w.Header().Set("Cache-Control", "max-age=432000")
	w.Header().Set("Content-Disposition", fmt.Sprintf(`attachment; filename="%s"`, filename))

	io.Copy(w, fd)
}

func showannotate(w http.ResponseWriter, r *http.Request) {
	reponame := mux.Vars(r)["reponame"]
	filename := mux.Vars(r)["filename"]
	changeset := mux.Vars(r)["changeset"]
	conn := getgerc(reponame)
	if conn == nil {
		http.NotFound(w, r)
		return
	}
	defer putgerc(conn)

	var contents template.HTML
	data, err := conn.GetFileData(gerc.FileDataArgs{Filename: filename, Revision: changeset})
	if err != nil {
		elog.Printf("error getting file data: %s\n", err)
		errorPage(w, r, "unable to get file data")
		return
	}
	ct := http.DetectContentType(data)
	if !strings.HasPrefix(ct, "text/") {
		contents = template.HTML("unsupported file type: " + ct + "\n")
	} else {
		rawlines, err := conn.Annotate(gerc.AnnotateArgs{Filename: filename, Revisions: changeset})
		if err != nil {
			elog.Printf("error annotating: %s", err)
			errorPage(w, r, "unable to annotate file")
			return
		}
		numlines := len(rawlines)
		clines := make([][]byte, numlines)
		codelines := make([][]byte, numlines, numlines+1)
		for i, line := range rawlines {
			clines[i] = []byte(line.Annotation)
			codelines[i] = []byte(line.Line)
		}
		codelines = append(codelines, []byte(""))
		codelines = bytes.Split(highlight(bytes.Join(codelines, []byte("\n")), filename), []byte("\n"))
		for i := 0; i < len(clines); i++ {
			codelines[i] = []byte(fmt.Sprintf(
				`<a href="/r/%s/v/%s">%s</a>:	%s`,
				reponame, clines[i], clines[i], codelines[i]))
		}
		contents = template.HTML(bytes.Join(codelines, []byte("\n")))
	}

	path := strings.Split(filename, "/")
	basename := path[len(path)-1]
	path = path[:len(path)-1]

	w.Header().Set("Cache-Control", "max-age=30")

	templinfo := getInfo(r)
	templinfo["Title"] = fmt.Sprintf("%s/%s - %s", reponame, filename, serverName)
	templinfo["Repo"] = reponame
	templinfo["Changeset"] = changeset
	templinfo["Filename"] = filename
	templinfo["Path"] = path
	templinfo["Basename"] = basename
	templinfo["Contents"] = contents
	templinfo["IsAnnotated"] = true
	err = views.Execute(w, "file.html", templinfo)
	if err != nil {
		elog.Printf("template execution: %s", err)
	}
}

func showfileraw(w http.ResponseWriter, r *http.Request) {
	if isAPrequest(r) {
		http.Error(w, "there are no activities here", http.StatusNotAcceptable)
		return
	}
	reponame := mux.Vars(r)["reponame"]
	filename := mux.Vars(r)["filename"]
	changeset := mux.Vars(r)["changeset"]
	conn := getgerc(reponame)
	if conn == nil {
		http.NotFound(w, r)
		return
	}
	defer putgerc(conn)

	data, err := conn.GetFileData(gerc.FileDataArgs{Filename: filename, Revision: changeset})
	if err != nil {
		elog.Printf("error getting file data: %s\n", err)
		errorPage(w, r, "unable to get file data")
		return
	}

	ct := http.DetectContentType(data)
	if strings.HasPrefix(ct, "text/") {
		w.Header().Set("Content-Type", "text/plain")
	} else if strings.HasPrefix(ct, "image/") {
		w.Header().Set("Content-Type", ct)
	} else {
		w.Header().Set("Content-Type", "application/binary")
	}
	w.Header().Set("X-Content-Type-Options", "nosniff")
	w.Header().Set("Cache-Control", "max-age=30")

	w.Write(data)
}

func showfile(w http.ResponseWriter, r *http.Request) {
	reponame := mux.Vars(r)["reponame"]
	filename := mux.Vars(r)["filename"]
	changeset := mux.Vars(r)["changeset"]
	if filename == "" || strings.HasSuffix(filename, "/") {
		showfiles(w, r)
		return
	}
	conn := getgerc(reponame)
	if conn == nil {
		http.NotFound(w, r)
		return
	}
	defer putgerc(conn)

	var contents template.HTML
	data, err := conn.GetFileData(gerc.FileDataArgs{Filename: filename, Revision: changeset})
	if err != nil {
		elog.Printf("error getting file data: %s\n", err)
		errorPage(w, r, "unable to get file data")
		return
	}
	ct := http.DetectContentType(data)
	if strings.HasPrefix(ct, "text/") {
		contents = template.HTML(highlight(data, filename))
	} else if strings.HasPrefix(ct, "image/") {
		contents = template.HTML(fmt.Sprintf(
			`<img src="/r/%s/v/%s/d/%s">`,
			reponame, changeset, filename))
	} else {
		contents = template.HTML("unsupported file type: " + ct + "\n")
	}

	path := strings.Split(filename, "/")
	basename := path[len(path)-1]
	path = path[:len(path)-1]

	w.Header().Set("Cache-Control", "max-age=30")

	templinfo := getInfo(r)
	templinfo["Title"] = fmt.Sprintf("%s/%s - %s", reponame, filename, serverName)
	templinfo["Repo"] = reponame
	templinfo["Changeset"] = changeset
	templinfo["Filename"] = filename
	templinfo["Path"] = path
	templinfo["Basename"] = basename
	templinfo["Contents"] = contents
	err = views.Execute(w, "file.html", templinfo)
	if err != nil {
		elog.Printf("template execution: %s", err)
	}
}

func showchanges(w http.ResponseWriter, r *http.Request) {
	reponame := mux.Vars(r)["reponame"]
	changeset := mux.Vars(r)["changeset"]
	filename := mux.Vars(r)["filename"]
	conn := getgerc(reponame)
	if conn == nil {
		http.NotFound(w, r)
		return
	}
	defer putgerc(conn)

	start := r.FormValue("start")
	end := r.FormValue("end")
	if start != "" {
		s := start
		e := end
		if (s < e || e == "tip") && s != "0" && s != "tip" {
			s += "~-2"
		}
		if (e < s || s == "tip") && e != "0" && e != "tip" {
			e += "~-2"
		}
		changeset = s + ":" + e
	} else {
		start = "tip"
		end = "0"
	}

	tags := conn.GetTags()

	path := strings.Split(filename, "/")
	basename := path[len(path)-1]
	path = path[:len(path)-1]

	changes := changelisting(conn, changeset, filename, 0)
	if changes == nil {
		errorPage(w, r, "unable to get changes")
		return
	}

	w.Header().Set("Cache-Control", "max-age=30")

	templinfo := getInfo(r)
	templinfo["Title"] = fmt.Sprintf("%s/%s history - %s", reponame, filename, serverName)
	templinfo["Repo"] = reponame
	templinfo["Changeset"] = changeset
	templinfo["Filename"] = filename
	templinfo["Path"] = path
	templinfo["Basename"] = basename
	templinfo["Changes"] = changes
	templinfo["Tags"] = tags
	templinfo["StartTag"] = start
	templinfo["EndTag"] = end
	err := views.Execute(w, "changes.html", templinfo)
	if err != nil {
		elog.Printf("template execution: %s", err)
	}
}

func showdiff(w http.ResponseWriter, r *http.Request) {
	reponame := mux.Vars(r)["reponame"]
	changeset := mux.Vars(r)["changeset"]
	conn := getgerc(reponame)
	if conn == nil {
		http.NotFound(w, r)
		return
	}
	defer putgerc(conn)

	changes, err := conn.GetChanges(gerc.ChangesArgs{Revisions: changeset, WithDiff: true})
	if err != nil {
		elog.Printf("error getting changeset: %s\n", err)
		errorPage(w, r, "unable to get changeset")
		return
	}

	if len(changes) > 1 {
		elog.Printf("too many changesets")
		errorPage(w, r, "too many chanesets in result")
		return
	}

	var diff bytes.Buffer
	var info strings.Builder
	for _, change := range changes {
		templates.Fprintf(&info, "changeset: %d:%.6x\n", change.Linkrev, change.NodeID)
		templates.Fprintf(&info, "user: %s\n", change.User)
		templates.Fprintf(&info, "files:")
		for _, f := range change.Files {
			templates.Fprintf(&info, " <a href=\"./%.8x/f/%s\">%s</a>", change.NodeID, f, f)
		}
		templates.Fprintf(&info, "\n")
		templates.Fprintf(&info, "description:\n %s", strings.Join(change.Message, "\n"))
		templates.Fprintf(&info, "\n")
		diff.WriteString(change.Diff)
	}

	w.Header().Set("Cache-Control", "max-age=3600")

	templinfo := getInfo(r)
	templinfo["Title"] = fmt.Sprintf("%s/%s - %s", reponame, changeset, serverName)
	templinfo["Repo"] = reponame
	templinfo["Changeset"] = changeset
	templinfo["Info"] = template.HTML(info.String())
	templinfo["Diff"] = template.HTML(highlight(diff.Bytes(), "hg.diff"))
	err = views.Execute(w, "diff.html", templinfo)
	if err != nil {
		elog.Printf("template execution: %s", err)
	}
}

func runmandoc(reponame string, src []byte) (template.HTML, error) {
	manopt := fmt.Sprintf("fragment,man=/r/%s/m/%%N.%%S", reponame)
	proc := exec.Command("mandoc", "-T", "html", "-O", manopt)
	wpipe, _ := proc.StdinPipe()
	rpipe, _ := proc.StdoutPipe()
	err := proc.Start()
	if err != nil {
		return "", err
	}
	defer proc.Wait()
	wpipe.Write(src)
	wpipe.Close()
	defer rpipe.Close()
	res, err := ioutil.ReadAll(rpipe)
	if err != nil {
		return "", err
	}
	re_imglinker := regexp.MustCompile(`<a class="Lk" href="([[:alnum:]._-]*?)">((?s)[[:alnum:][:space:]]*?)</a>`)
	repl := fmt.Sprintf(`<a class="Lk" href="/r/%s/v/tip/d/docs/$1"><img src="/r/%s/v/tip/d/docs/$1" alt="$2"></a>`, reponame, reponame)
	res = re_imglinker.ReplaceAll(res, []byte(repl))
	return template.HTML(res), nil
}

func manualfiles(conn *gerc.Repo, reponame string) []gerc.ManifestFile {
	suffixes := []string{"1", "8", "3"}
	var srcnames []string
	for _, s := range suffixes {
		srcnames = append(srcnames, fmt.Sprintf("docs/%s.%s", "intro", s))
	}
	for _, s := range suffixes {
		srcnames = append(srcnames, fmt.Sprintf("docs/%s.%s", reponame, s))
	}
	files, _ := conn.GetFiles(gerc.FilesArgs{Filenames: srcnames})
	return files
}

func showmanual(w http.ResponseWriter, r *http.Request) {
	reponame := mux.Vars(r)["reponame"]
	filename := mux.Vars(r)["filename"]
	conn := getgerc(reponame)
	if conn == nil {
		http.NotFound(w, r)
		return
	}
	defer putgerc(conn)

	if filename == "" {
		files := manualfiles(conn, reponame)
		if len(files) > 0 {
			filename = files[0].Name
		}
	} else {
		filename = "docs/" + filename
	}
	//mansrc, err := ioutil.ReadFile("repos/" + reponame + "/" + filename)
	mansrc, err := conn.GetFileData(gerc.FileDataArgs{Filename: filename})
	if err != nil {
		elog.Printf("error getting mansrc: %s", err)
		errorPage(w, r, "no documentation available")
		return
	}
	man, err := runmandoc(reponame, mansrc)
	if err != nil {
		elog.Printf("error running mandoc: %s", err)
	}

	w.Header().Set("Cache-Control", "max-age=30")

	templinfo := getInfo(r)
	templinfo["Title"] = fmt.Sprintf("%s/%s - %s", reponame, filename, serverName)
	templinfo["Repo"] = reponame
	templinfo["Manual"] = man
	err = views.Execute(w, "manual.html", templinfo)
	if err != nil {
		elog.Printf("template execution: %s", err)
	}
}

func showreporss(w http.ResponseWriter, r *http.Request) {
	reponame := mux.Vars(r)["reponame"]
	conn := getgerc(reponame)
	if conn == nil {
		http.NotFound(w, r)
		return
	}
	defer putgerc(conn)

	changes := recentchangelisting(conn)
	if changes == nil {
		errorPage(w, r, "unable to get recent changes")
		return
	}

	feed := rss.Feed{
		Title:       reponame + " changes",
		Description: "most recent changes for " + reponame,
		Link:        fmt.Sprintf("https://%s/r/%s", serverName, reponame),
	}

	feed.Items = make([]*rss.Item, 0, len(changes))
	for _, c := range changes {
		desc := fmt.Sprintf("<p>Changeset %s by %s:\n<p>%s\n",
			c.Changeset.Hex, c.User, strings.Replace(html.EscapeString(strings.Join(c.Message, "\n")), "\n", "<br>\n", -1))
		feed.Items = append(feed.Items, &rss.Item{
			Title:       c.Summary,
			Description: rss.CData{desc},
			Link:        fmt.Sprintf("https://%s/r/%s/v/%s", serverName, reponame, c.Changeset.Hex),
			PubDate:     c.Date.Format(time.RFC1123),
		})
	}

	w.Header().Set("Cache-Control", "max-age=300")

	err := feed.Write(w)
	if err != nil {
		elog.Printf("rss error: %s", err)
	}
}

func showchangeset(w http.ResponseWriter, r *http.Request) {
	reponame := mux.Vars(r)["reponame"]
	changeset := mux.Vars(r)["changeset"]
	conn := getgerc(reponame)
	if conn == nil {
		http.NotFound(w, r)
		return
	}
	defer putgerc(conn)

	changes, err := conn.GetChanges(gerc.ChangesArgs{Revisions: changeset, WithDiff: true})
	if err != nil {
		elog.Printf("error getting changeset: %s\n", err)
		errorPage(w, r, "unable to get changeset")
		return
	}

	if len(changes) > 1 {
		elog.Printf("too many changesets")
		errorPage(w, r, "too many chanesets in result")
		return
	}
	if len(changes) == 0 {
		http.NotFound(w, r)
		return
	}
	if enableFedi && isAPrequest(r) {
		j := jsonCommit(reponame, changes[0])
		w.Header().Set("Content-Type", apContentType)
		j.Write(w)
		return
	}

	var diff bytes.Buffer
	var info strings.Builder
	for _, change := range changes {
		templates.Fprintf(&info, "changeset: %d:%.6x\n", change.Linkrev, change.NodeID)
		templates.Fprintf(&info, "user: %s\n", change.User)
		templates.Fprintf(&info, "files:")
		for _, f := range change.Files {
			templates.Fprintf(&info, " <a href=\"./%.8x/f/%s\">%s</a>", change.NodeID, f, f)
		}
		templates.Fprintf(&info, "\n")
		templates.Fprintf(&info, "description:\n %s", strings.Join(change.Message, "\n"))
		templates.Fprintf(&info, "\n")
		diff.WriteString(change.Diff)
	}

	w.Header().Set("Cache-Control", "max-age=30")

	templinfo := getInfo(r)
	templinfo["Title"] = fmt.Sprintf("%s/%s - %s", reponame, changeset, serverName)
	templinfo["Repo"] = reponame
	templinfo["Changeset"] = changeset
	templinfo["Info"] = template.HTML(info.String())
	templinfo["Diff"] = template.HTML(highlight(diff.Bytes(), "hg.diff"))
	err = views.Execute(w, "changeset.html", templinfo)
	if err != nil {
		elog.Printf("template execution: %s", err)
	}
}
func rootpage(w http.ResponseWriter, r *http.Request) {
	allrepos := loadrepos()

	w.Header().Set("Cache-Control", "max-age=30")

	templinfo := getInfo(r)
	templinfo["AllRepos"] = allrepos
	err := views.Execute(w, "repos.html", templinfo)
	if err != nil {
		elog.Printf("template execution: %s", err)
	}
}

func servecss(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Cache-Control", "max-age=7776000")
	http.ServeFile(w, r, "views"+r.URL.Path)
}

func servettf(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Cache-Control", "max-age=7776000")
	http.ServeFile(w, r, "views"+r.URL.Path)
}

func robotstxt(w http.ResponseWriter, r *http.Request) {
	allrepos := loadrepos()
	io.WriteString(w, "User-agent: *\n")
	for _, r := range allrepos {
		fmt.Fprintf(w, "Disallow: /r/%s/v/\n", r.Name)
	}
}

func serve() {
	listener, err := openListener()
	if err != nil {
		elog.Fatal(err)
	}

	securitizeweb()

	go reaper()

	initrepos()

	views = templates.Load(develMode,
		"views/head.html",
		"views/path.html",
		"views/changeset.html",
		"views/diff.html",
		"views/changes.html",
		"views/downloads.html",
		"views/manual.html",
		"views/file.html",
		"views/files.html",
		"views/overview.html",
		"views/repos.html",
		"views/error.html",
	)
	if !develMode {
		savedstyleparam = getstyleparam()
	}

	go adminlistener()

	ssh := false
	getconfig("sshenabled", &ssh)
	if ssh {
		go sshserver()
	}
	if enableFedi {
		go deliveryManager()
	}

	mux := mux.NewRouter()
	getters := mux.Methods("GET").Subrouter()
	posters := mux.Methods("POST").Subrouter()

	getters.HandleFunc("/style.css", servecss)
	getters.HandleFunc("/htmx.min.js", servecss)
	getters.HandleFunc("/fantasque.ttf", servettf)
	getters.HandleFunc("/robots.txt", robotstxt)
	getters.HandleFunc("/", rootpage)
	getters.HandleFunc("/r/{reponame}", showrepo)
	getters.HandleFunc("/r/{reponame}/d", showdownloads)
	getters.HandleFunc("/r/{reponame}/d/{filename:[[:alnum:]_. -]*}", downloadfile)
	getters.HandleFunc("/r/{reponame}/f", showfiles)
	getters.HandleFunc("/r/{reponame}/h", showchanges)
	getters.HandleFunc("/r/{reponame}/m", showmanual)
	getters.HandleFunc("/r/{reponame}/m/{filename:[[:alnum:]_. -]*}", showmanual)
	getters.HandleFunc("/r/{reponame}/rss", showreporss)
	getters.HandleFunc("/r/{reponame}/diff/{changeset:[^/]*}", showdiff)
	getters.HandleFunc("/r/{reponame}/v/{changeset:[^/]*}", showchangeset)
	getters.HandleFunc("/r/{reponame}/v/{changeset:[^/]*}/a/{filename:[[:alnum:]_. /-]*}", showannotate)
	getters.HandleFunc("/r/{reponame}/v/{changeset:[^/]*}/d/{filename:[[:alnum:]_. /-]*}", showfileraw)
	getters.HandleFunc("/r/{reponame}/v/{changeset:[^/]*}/f/{filename:[[:alnum:]_. /-]*}", showfile)
	getters.HandleFunc("/r/{reponame}/v/{changeset:[^/]*}/h/{filename:[[:alnum:]_. /-]*}", showchanges)

	if enableFedi {
		posters.HandleFunc("/inbox", webinbox)
		posters.HandleFunc("/r/{reponame}/inbox", webinbox)
		getters.HandleFunc("/u/{username}", showuser)
	}

	err = http.Serve(listener, mux)
	if err != nil {
		elog.Fatalf("error serving: %s", err)
	}
}
