package main

import (
	"crypto/rand"
	"crypto/rsa"
	"database/sql"
	"fmt"
	"os"
	"strings"

	"humungus.tedunangst.com/r/webs/httpsig"
)

func generateKeys() (string, string, error) {
	k, err := rsa.GenerateKey(rand.Reader, 2048)
	if err != nil {
		return "", "", err
	}
	pubkey, err := httpsig.EncodeKey(&k.PublicKey)
	if err != nil {
		return "", "", err
	}
	seckey, err := httpsig.EncodeKey(k)
	if err != nil {
		return "", "", err
	}
	return pubkey, seckey, nil
}

func updateKeys(reponame string) error {
	apid := serverURL("/r/%s", reponame)

	pubkey, seckey, err := generateKeys()
	if err == nil {
		err = dbSavePubkey(apid+"#key", apid, pubkey)
	}
	if err == nil {
		err = dbSaveSeckey(apid, seckey)
	}
	if err != nil {
		return err
	}
	return nil
}

func upgradedb() {
	db := opendatabase()
	dbversion := 0
	getconfig("dbversion", &dbversion)
	getconfig("servername", &serverName)
	serverPrefix = fmt.Sprintf("https://%s/", serverName)
	var tx *sql.Tx
	try := func(s string, args ...interface{}) *sql.Rows {
		var err error
		var rows *sql.Rows
		if strings.HasPrefix(s, "select ") {
			if tx != nil {
				rows, err = tx.Query(s, args...)
			} else {
				rows, err = db.Query(s, args...)
			}
		} else {
			if tx != nil {
				_, err = tx.Exec(s, args...)
			} else {
				_, err = db.Exec(s, args...)
			}
		}
		if err != nil {
			elog.Fatalf("can't run %s: %s", s, err)
		}
		return rows
	}
	setversion := func(version int) {
		try("update config set value = ? where key = 'dbversion'", version)
	}
	switch dbversion {
	case 0:
		try("create table pubkeys (pubkeyid integer primary key, url text, owner text, pubkey text, dt text)")
		try("create table seckeys (seckeyid integer primary key, owner text, seckey text)")
		try("create table follows (followid integer primary key, folxid text, who text, what text)")
		try("create table deliveries (deliveryid integer primary key, dt text, tries integer, rcpt text, data blob)")
		try("create index idxpubkeys_url on pubkeys(url)")
		try("create index idxseckeys_owner on seckeys(owner)")
		try("create index idxfollows_xid on follows(folxid)")
		try("create index idxfollows_who on follows(who)")
		try("create index idxfollows_what on follows(what)")
		try("create index idxdeliveries_rcpt on deliveries(rcpt)")
		prepareStatements(db)
		allrepos := loadrepos()
		for _, repo := range allrepos {
			updateKeys(repo.Name)
		}
		setversion(1)
		fallthrough
	case 1:
		try("create table persons (personid integer primary key, pname text, display text, papid text, personmeta text)")
		try("create index idxpersons_name on persons(pname)")
		try("create index idxpersons_apid on persons(papid)")
		setversion(1)
		fallthrough
	case 2:
	default:
		elog.Fatal("unknown version: %d", dbversion)
	}
	os.Exit(0)
}
