//
// Copyright (c) 2018 Ted Unangst <tedu@tedunangst.com>
//
// Permission to use, copy, modify, and distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

package main

import (
	"bufio"
	"bytes"
	"crypto/sha512"
	"flag"
	"fmt"
	"io"
	golog "log"
	"net"
	"os"
	"strconv"
	"sync"

	"humungus.tedunangst.com/r/gerc"
	_ "humungus.tedunangst.com/r/go-sqlite3"
	"humungus.tedunangst.com/r/webs/cache"
	"humungus.tedunangst.com/r/webs/log"
	"humungus.tedunangst.com/r/webs/synlight"
)

var serverName string
var serverPrefix string
var develMode bool

type hlargs struct {
	Code     []byte
	Filename string
}

var htmlfragCodeline = []byte("\n<span class=codeline></span>")

var lighter = synlight.New(synlight.Options{Format: synlight.HTML})

func dohighlight(args *hlargs) ([]byte, bool) {
	var buf bytes.Buffer
	lighter.Highlight(args.Code, args.Filename, &buf)

	hh := buf.Bytes()
	if len(hh) > 0 && hh[len(hh)-1] == '\n' {
		hh = hh[:len(hh)-1]
	}
	if args.Filename != "hg.diff" && len(hh) > 0 {
		hh = append(htmlfragCodeline[1:],
			bytes.Replace(hh, []byte("\n"), htmlfragCodeline, -1)...)
	}

	return hh, true
}

var hlcache = cache.New(cache.Options{
	Filler:    dohighlight,
	SizeLimit: 40 * 1024 * 1024,
	Reducer: func(args *hlargs) interface{} {
		hasher := sha512.New()
		io.WriteString(hasher, args.Filename)
		io.WriteString(hasher, "\x00")
		hasher.Write(args.Code)
		return string(hasher.Sum(nil))
	},
	Sizer: func(res []byte) int {
		return len(res)
	},
})

func highlight(code []byte, filename string) []byte {
	var res []byte
	hlcache.Get(&hlargs{Code: code, Filename: filename}, &res)
	return res
}

type RepoRow struct {
	ID          int
	Name        string
	Description string
	Order       int
}

func loadrepos() []*RepoRow {
	db := opendatabase()
	rows, _ := db.Query("select repoid, name, description, orderid from repos order by orderid asc")
	defer rows.Close()
	var repos []*RepoRow
	for rows.Next() {
		var repo RepoRow
		rows.Scan(&repo.ID, &repo.Name, &repo.Description, &repo.Order)
		repos = append(repos, &repo)
	}
	return repos
}

func initrepos() {
	dlog.Printf("init repos")

	gercpoolmtx.Lock()
	defer gercpoolmtx.Unlock()
	commitmapmtx.Lock()
	defer commitmapmtx.Unlock()
	for _, pool := range gercpools {
		pool.flush()
	}
	gercpools = make(map[string]gercPool)
	commitmap = make(map[string]map[string]bool)
	allrepos := loadrepos()
	for _, repo := range allrepos {
		gercpools[repo.Name] = gercPool{}
	}
	go initcommitmaps(allrepos)
}

func initcommitmaps(allrepos []*RepoRow) {
	commitmapmtx.Lock()
	defer commitmapmtx.Unlock()
	for _, repo := range allrepos {
		initcommitmap(repo.Name)
	}
}
func initcommitmap(reponame string) {
	m := make(map[string]bool)
	commitmap[reponame] = m
	conn := getgerc(reponame)
	defer putgerc(conn)
	changes, _ := conn.GetChanges(gerc.ChangesArgs{Revisions: "tip:0"})
	for _, change := range changes {
		m[hexchange(change)] = true
	}
}

func hexchange(change *gerc.Change) string {
	return fmt.Sprintf("%.6x", change.NodeID)
}

func recache(name string) {
	gercpoolmtx.Lock()
	defer gercpoolmtx.Unlock()
	pool := gercpools[name]
	pool.flush()
	gercpools[name] = gercPool{}
	unbusy(name)
	go checkNewCommits(name)
}

var commitmap map[string]map[string]bool
var commitmapmtx sync.Mutex

func checkNewCommits(reponame string) {
	commitmapmtx.Lock()
	defer commitmapmtx.Unlock()
	m := commitmap[reponame]
	conn := getgerc(reponame)
	defer putgerc(conn)
	changes, _ := conn.GetChanges(gerc.ChangesArgs{Revisions: "tip:0"})
	for _, change := range changes {
		hex := hexchange(change)
		if !m[hex] {
			c2, _ := conn.GetChanges(gerc.ChangesArgs{Revisions: hex, WithDiff: true})
			apNewCommit(reponame, c2[0])
			m[hex] = true
		}
	}
}

func adminlistener() {
	sockname := "humungus.sock"
	err := os.Remove(sockname)
	if err != nil && !os.IsNotExist(err) {
		elog.Printf("unable to unlink admin socket: %s", err)
		return
	}
	sock, err := net.Listen("unix", sockname)
	if err != nil {
		elog.Printf("unable to create admin socket: %s", err)
		return
	}
	for {
		fd, err := sock.Accept()
		if err != nil {
			continue
		}
		scanner := bufio.NewScanner(fd)
		for scanner.Scan() {
			cmd := scanner.Text()
			dlog.Printf("got cmd %s\n", cmd)
			switch cmd {
			case "init":
				initrepos()
			case "stop":
				os.Exit(0)
			}
		}
		fd.Close()
	}
}

var elog, ilog, dlog *golog.Logger

func main() {
	flag.Parse()
	log.Init(log.Options{Progname: "humungus"})
	elog = log.E
	ilog = log.I
	dlog = log.D

	args := flag.Args()
	if len(args) == 0 {
		args = []string{"serve"}
	}
	cmd := args[0]
	args = args[1:]

	switch cmd {
	case "init":
		initdb()
	case "upgrade":
		upgradedb()
	case "version":
		fmt.Println("version")
		os.Exit(0)
	}

	preparedatabase()
	getconfig("devel", &develMode)
	getconfig("enablefedi", &enableFedi)
	getconfig("servername", &serverName)
	serverPrefix = fmt.Sprintf("https://%s/", serverName)

	switch cmd {
	case "admin":
		adminscreen()
	case "control":
		controlcommand(args)
	case "devel":
		if len(args) != 1 {
			elog.Fatal("usage: devel (on|off)")
			return
		}
		switch args[0] {
		case "on":
			setconfig("devel", 1)
		case "off":
			setconfig("devel", 0)
		default:
			elog.Fatal("usage: devel (on|off)")
		}
	case "setconfig":
		if len(args) != 2 {
			errx("need an argument: setconfig key val")
		}
		var val interface{}
		var err error
		if val, err = strconv.Atoi(args[1]); err != nil {
			val = args[1]
		}
		setconfig(args[0], val)
	case "serve":
		serve()
	default:
		fmt.Printf("unknown command\n")
		os.Exit(1)
	}
}

func errx(msg string, args ...interface{}) {
	fmt.Fprintf(os.Stderr, msg+"\n", args...)
	os.Exit(1)
}
