//
// Copyright (c) 2023 Ted Unangst <tedu@tedunangst.com>
//
// Permission to use, copy, modify, and distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

package main

import (
	"bytes"
	"context"
	"crypto/tls"
	"fmt"
	"io"
	"net/http"
	"regexp"
	"sync"
	"time"

	"humungus.tedunangst.com/r/webs/httpsig"
	"humungus.tedunangst.com/r/webs/junk"
)

var userAgent = "humungus/1.0; "

var fastTimeout time.Duration = 5 * time.Second
var slowTimeout time.Duration = 30 * time.Second
var onceSetFetch sync.Once

var httpTransport = http.Transport{
	MaxIdleConns:    120,
	MaxConnsPerHost: 4,
}

var httpClient = http.Client{
	Transport: &httpTransport,
}

func disableTLS() {
	httpTransport.TLSClientConfig = &tls.Config{
		InsecureSkipVerify: true,
	}
}

func fetchActivity(url string, timeout time.Duration) (junk.Junk, error) {
	if hastombstone(url) {
		dlog.Printf("skipping dead url: %s", url)
		return nil, fmt.Errorf("url has a tombstone")
	}
	dlog.Printf("fetching activity: %s", url)
	client := httpClient
	client.CheckRedirect = func(req *http.Request, via []*http.Request) error {
		if len(via) >= 5 {
			return fmt.Errorf("stopped after 5 redirects")
		}
		return nil
	}
	j, err := junkGet(url, junk.GetArgs{
		Accept:  apAcceptType,
		Agent:   userAgent,
		Timeout: timeout,
		Client:  &client,
		Limit:   1 * 1024 * 1024,
	})
	return j, err
}

func junkGet(url string, args junk.GetArgs) (junk.Junk, error) {
	client := http.DefaultClient
	if args.Client != nil {
		client = args.Client
	}
	req, err := http.NewRequest("GET", url, nil)
	if err != nil {
		return nil, err
	}
	if args.Accept != "" {
		req.Header.Set("Accept", args.Accept)
	}
	if args.Agent != "" {
		req.Header.Set("User-Agent", args.Agent)
	}
	if args.Fixup != nil {
		err = args.Fixup(req)
		if err != nil {
			return nil, err
		}
	}
	if args.Timeout != 0 {
		ctx, cancel := context.WithTimeout(context.Background(), args.Timeout)
		defer cancel()
		req = req.WithContext(ctx)
	}
	resp, err := client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()

	switch resp.StatusCode {
	case 200:
	case 201:
	case 202:
	default:
		return nil, fmt.Errorf("http get status: %d", resp.StatusCode)
	}
	ct := resp.Header.Get("Content-Type")
	if !isAPtype(ct) {
		return nil, fmt.Errorf("incompatible content type %s", ct)
	}
	var r io.Reader = resp.Body
	if args.Limit > 0 {
		r = io.LimitReader(r, args.Limit)
	}
	return junk.Read(r)
}

func PostMsg(keyname string, key httpsig.PrivateKey, url string, msg []byte) error {
	dlog.Printf("posting to %s", url)
	req, err := http.NewRequest("POST", url, bytes.NewReader(msg))
	if err != nil {
		return err
	}
	req.Header.Set("User-Agent", userAgent)
	req.Header.Set("Content-Type", apContentType)
	httpsig.SignRequest(keyname, key, req, msg)
	ctx, cancel := context.WithTimeout(context.Background(), 2*slowTimeout)
	defer cancel()
	req = req.WithContext(ctx)
	resp, err := httpClient.Do(req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	switch resp.StatusCode {
	case 200:
	case 201:
	case 202:
	default:
		var buf [80]byte
		n, _ := resp.Body.Read(buf[:])
		dlog.Printf("post failure message: %s", buf[:n])
		return fmt.Errorf("http post status: %d", resp.StatusCode)
	}
	ilog.Printf("successful post: %s %d", url, resp.StatusCode)
	return nil
}

var re_keyholder = regexp.MustCompile(`keyId="([^"]+)"`)

func requestActor(r *http.Request) string {
	if sig := r.Header.Get("Signature"); sig != "" {
		if m := re_keyholder.FindStringSubmatch(sig); len(m) == 2 {
			return m[1]
		}
	}
	return ""
}
