//
// Copyright (c) 2023 Ted Unangst <tedu@tedunangst.com>
//
// Permission to use, copy, modify, and distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

package main

import (
	"bytes"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strings"
	"sync"
	"time"

	"github.com/gorilla/mux"
	"humungus.tedunangst.com/r/gerc"
	"humungus.tedunangst.com/r/webs/gencache"
	"humungus.tedunangst.com/r/webs/httpsig"
	"humungus.tedunangst.com/r/webs/junk"
)

var enableFedi bool
var requestWG sync.WaitGroup

func showuser(w http.ResponseWriter, r *http.Request) {
	username := mux.Vars(r)["username"]
	j := jsonPerson(username)
	//w.Header().Set("Cache-Control", "max-age=30")
	j.Write(w)
}

func webinbox(w http.ResponseWriter, r *http.Request) {
	if !enableFedi {
		http.NotFound(w, r)
		return
	}
	if !isAPpost(r) {
		http.Error(w, "speak activity please", http.StatusNotAcceptable)
		return
	}

	if reponame := mux.Vars(r)["reponame"]; reponame != "" {
		conn := getgerc(reponame)
		if conn == nil {
			http.NotFound(w, r)
			return
		}
		defer putgerc(conn)
	}

	var buf bytes.Buffer
	io.CopyN(&buf, r.Body, 1*1024*1024)
	payload := buf.Bytes()
	j, err := junk.FromBytes(payload)
	if err != nil {
		ilog.Printf("bad payload: %s", err)
		return
	}

	if ignoreJunk(j) {
		return
	}
	keyname, err := httpsig.VerifyRequest(r, payload, findpubkey)
	if err != nil {
		if shouldReverify(keyname) {
			keyname, err = httpsig.VerifyRequest(r, payload, findpubkey)
		}
	}
	if err != nil {
		ilog.Printf("signature failed: %s", err)
		w.WriteHeader(http.StatusUnauthorized)
		io.WriteString(w, "valid signature required\n")
		return
	}
	actor, _ := j.GetString("actor")
	if originate(actor) != originate(keyname) {
		dlog.Printf("impersonated object %s from %s", actor, keyname)
		return
	}
	go inboxProcess(actor, r.URL.Path, j)
}

func inboxProcess(actor, path string, j junk.Junk) {
	id, _ := j.GetString("id")
	typ := getType(j)
	dlog.Printf("got an inbox: %s to %s from %s %s", typ, path, actor, id)
	switch typ {
	case "Follow":
		apHandleFollow(j)
	case "Undo":
		apHandleUndo(j)
	default:
		dlog.Printf("unknown activity in inbox: %s %s", typ, j.ToString())
	}
}

func weboutbox(w http.ResponseWriter, r *http.Request) {
	if !enableFedi {
		http.NotFound(w, r)
		return
	}
	http.NotFound(w, r)
}

func apHandleUndo(j junk.Junk) {
	actor, _ := j.GetString("actor")
	obj, _ := j.GetMap("object")
	if len(obj) == 0 {
		dlog.Printf("not sure what to undo: %s", j.ToString())
		return
	}
	typ := getType(obj)
	switch typ {
	case "Follow":
		target, _ := obj.GetString("object")
		if target != "" {
			err := dbDeleteFollow(actor, target)
			if err != nil {
				elog.Printf("error deleting follow: %s", err)
			}
			return
		}
		dlog.Printf("can't undo follow: %s", obj.ToString())
	default:
		dlog.Printf("can't undo yet: %s", obj.ToString())
	}
}

func apHandleFollow(j junk.Junk) {
	folxid, _ := j.GetString("id")
	who, _ := j.GetString("actor")
	what, _ := j.GetString("object")
	if !isMine(what) {
		return
	}
	reponame := strings.TrimPrefix(what, serverURL("/r/"))
	conn := getgerc(reponame)
	if conn == nil {
		return
	}
	defer putgerc(conn)
	err := dbSaveFollow(folxid, who, what)
	if err != nil {
		elog.Printf("error saving follow: %s", err)
		return
	}
	apSendAccept(folxid, who, what)
}

func jsonFollow(folxid, who, what string) junk.Junk {
	j := newActivity("Follow", folxid)
	j["actor"] = who
	j["to"] = what
	j["object"] = what
	return j
}

func findInbox(rcpt string, shared bool) string {
	if person := findperson(rcpt); person != nil {
		if shared && person.ShareBox != "" {
			return person.ShareBox
		}
		return person.Inbox
	}
	if person := personForAPid(rcpt); person != nil {
		if shared && person.ShareBox != "" {
			return person.ShareBox
		}
		return person.Inbox
	}
	return ""
}

func collectBoxes(rcpts []string) map[string]bool {
	boxes := make(map[string]bool)
	for _, rcpt := range rcpts {
		if rcpt == apPublic {
			continue
		}
		if isMine(rcpt) {
			continue
		}
		inbox := findInbox(rcpt, true)
		if inbox != "" {
			boxes["%"+inbox] = true
			continue
		}
		boxes[rcpt] = true
	}
	return boxes
}

func apSendAccept(folxid, who, what string) {
	j := newActivity("Accept", what+"/accept/"+generateXid())
	j["actor"] = what
	j["to"] = who
	j["object"] = jsonFollow(folxid, who, what)

	go deliver(what, who, j.ToBytes())
}

func apSendToRcpts(from string, rcpts []string, msg []byte) {
	boxes := collectBoxes(rcpts)
	for inbox := range boxes {
		go deliver(from, inbox, msg)
	}
}

func ignoreJunk(j junk.Junk) bool {
	t := getType(j)
	if t == "Delete" {
		a, _ := j.GetString("actor")
		o, _ := j.GetString("object")
		if a == o && dbFindPerson(a) == nil {
			return true
		}
	}
	if t == "Announce" {
		if obj, ok := j.GetMap("object"); ok {
			j = obj
			t = getType(j)
		}
	}
	if t == "Undo" {
		if obj, ok := j.GetMap("object"); ok {
			j = obj
			t = getType(j)
		}
	}
	if t == "Like" || t == "Dislike" {
		return true
	}
	return false
}

var pubkeyCache = gencache.New(gencache.Options[string, httpsig.PublicKey]{
	Fill: func(keyname string) (httpsig.PublicKey, bool) {
		data := dbFindPubkey(keyname)
		if data == "" {
			j, err := fetchActivity(keyname, fastTimeout)
			if err != nil {
				dlog.Printf("unable to fetch needed pubkey: %s", err)
				return httpsig.PublicKey{}, false
			}
			typ := getType(j)
			if typ == "Person" || typ == "Service" {
				apHandlePerson(originate(keyname), j)
			}
			keyobj, ok := j.GetMap("publicKey")
			if ok {
				savepubkey(originate(keyname), keyobj, "")
			}
			data = dbFindPubkey(keyname)
		}
		if data == "" {
			dlog.Printf("no key found for %s", keyname)
			return httpsig.PublicKey{}, false
		}
		_, key, err := httpsig.DecodeKey(data)
		if err != nil {
			dlog.Printf("error decoding pubkey for %s: %s", keyname, err)
			return httpsig.PublicKey{}, false
		}
		return key, true
	},
	Limit: 600})

func findpubkey(keyname string) (httpsig.PublicKey, error) {
	var err error
	key, ok := pubkeyCache.Get(keyname)
	if !ok {
		err = fmt.Errorf("unable to find pubkey: %s", keyname)
	}
	return key, err
}

func savepubkey(origin string, keyobj junk.Junk, owner string) {
	pubkey, ok := keyobj.GetString("publicKeyPem")
	keyid, _ := keyobj.GetString("id")
	u, err := url.Parse(keyid)
	if err != nil {
		dlog.Printf("error parsing keyid: %s", err)
		return
	}
	u.Path = "/" + strings.TrimLeft(u.Path, "/")
	keyid = u.String()
	if owner == "" {
		owner, _ = keyobj.GetString("owner")
	}
	if owner == "" {
		dlog.Printf("no owner for key: %s", keyid)
		return
	}
	if originate(keyid) != origin || originate(owner) != origin {
		dlog.Printf("impersonated key %s for %s from %s", keyid, owner, origin)
		ok = false
	}
	if ok {
		err := dbSavePubkey(keyid, owner, pubkey)
		if err != nil {
			elog.Printf("error saving pubkey: %s", err)
		}
	}
}

func shouldReverify(keyname string) bool {
	if isMine(keyname) {
		return false
	}
	db := opendatabase()
	when := time.Now().Add(-60 * time.Minute).UTC().Format(dbtimeformat)
	res, err := db.Exec("delete from pubkeys where url = ? and dt < ?", keyname, when)
	if err != nil {
		elog.Printf("error clearing old pubkey: %s", err)
		return false
	}
	num, _ := res.RowsAffected()
	if num > 0 {
		dlog.Printf("deleted old pubkey %s", keyname)
		pubkeyCache.Clear(keyname)
		return true
	}
	return false
}

type keyInfo struct {
	name string
	key  httpsig.PrivateKey
}

var seckeyCache = gencache.New(gencache.Options[string, keyInfo]{
	Fill: func(from string) (keyInfo, bool) {
		row := stmtFindSeckey.QueryRow(from)
		var seckey string
		err := row.Scan(&seckey)
		if err != nil {
			elog.Printf("lost seckey for %s: %s", from, err)
			return keyInfo{}, false
		}
		key, _, err := httpsig.DecodeKey(seckey)
		if err != nil {
			elog.Printf("error decoding seckey: %s", err)
			return keyInfo{}, false
		}
		return keyInfo{from + "#key", key}, true
	},
	Limit: 300})

func getseckey(from string) (string, httpsig.PrivateKey) {
	ki, _ := seckeyCache.Get(from)
	return ki.name, ki.key
}

func apNewCommit(reponame string, change *gerc.Change) {
	apid := serverURL("/r/%s", reponame)
	rcpts := findfollowers(apid)
	jc := jsonCommit(reponame, change)
	j := jsonCreate(apid, "", jc)
	apSendToRcpts(apid, rcpts, j.ToBytes())
}
