package main

/*
#include <termios.h>

void
termecho(int on)
{
        struct termios t;
        tcgetattr(1, &t);
        if (on)
                t.c_lflag |= ECHO;
        else
                t.c_lflag &= ~ECHO;
        tcsetattr(1, TCSADRAIN, &t);
}
*/
import "C"

import (
	"bufio"
	"crypto/rand"
	"database/sql"
	"fmt"
	notrand "math/rand"
	"os"
	"os/signal"
	"strings"
	"time"
)

var dbVersion = 1

var alreadyopendb *sql.DB

func init() {
	notrand.Seed(time.Now().Unix())
}

func somedays() string {
	secs := 432000 + notrand.Int63n(432000)
	return fmt.Sprintf("%d", secs)
}

func opendatabase() *sql.DB {
	if alreadyopendb != nil {
		return alreadyopendb
	}
	dbname := "." + "/humungus.db"
	_, err := os.Stat(dbname)
	if err != nil {
		elog.Fatalf("unable to open database: %s", err)
	}
	db, err := sql.Open("sqlite3", dbname)
	if err != nil {
		elog.Fatalf("unable to open database: %s", err)
	}
	stmtConfig, err = db.Prepare("select value from config where key = ?")
	if err != nil {
		elog.Fatal(err)
	}
	alreadyopendb = db
	return db
}

func preparedatabase() {
	db := opendatabase()
	vers := 0
	getconfig("dbversion", &vers)
	if vers != dbVersion {
		elog.Fatal("incorrect database version. run upgrade.")
	}
	prepareStatements(db)
}

func askandread(q string, r *bufio.Reader) (string, error) {
	fmt.Print(q)
	ans, err := r.ReadString('\n')
	if err != nil {
		return "", err
	}
	ans = ans[:len(ans)-1]
	if len(ans) < 1 {
		return "", fmt.Errorf("that's way too short")
	}
	return ans, nil
}

func initdb() {
	dbname := "." + "/humungus.db"
	_, err := os.Stat(dbname)
	if err == nil {
		elog.Fatalf("%s already exists", dbname)
	}
	db, err := sql.Open("sqlite3", dbname)
	if err != nil {
		elog.Fatal(err)
	}
	alreadyopendb = db
	withcleanup(func() {
		for _, line := range strings.Split(sqlSchema, ";") {
			_, err = db.Exec(line)
			if err != nil {
				elog.Print(err)
				return
			}
		}
		prepareStatements(db)
		r := bufio.NewReader(os.Stdin)
		addr, err := askandread("listen address: ", r)
		if err != nil {
			elog.Print(err)
			return
		}
		setconfig("listenaddr", addr)
		addr, err = askandread("server name: ", r)
		if err != nil {
			elog.Print(err)
			return
		}
		setconfig("servername", addr)
		setconfig("dbversion", dbVersion)
		setconfig("debug", 0)
		db.Close()
		fmt.Printf("done.\n")
		os.Exit(0)
	}, func() {
		os.Remove(dbname)
	})
}

func withcleanup(fn func(), cleanup func()) {
	defer func() {
		if cleanup != nil {
			cleanup()
		}
		os.Exit(1)
	}()
	c := make(chan os.Signal)
	signal.Notify(c, os.Interrupt)
	go func() {
		<-c
		C.termecho(1)
		fmt.Printf("\n")
		if cleanup != nil {
			cleanup()
		}
		os.Exit(1)
	}()

	fn()
}

func byteletters(b []byte) string {
	letters := "BCDFGHJKLMNPQRSTVWXYZbcdfghjklmnpqrstvwxyz1234567891234567891234"
	for i, c := range b {
		b[i] = letters[c&63]
	}
	return string(b)
}

func genxid() string {
	var b [18]byte
	rand.Read(b[:])
	return byteletters(b[:])
}

func getconfig(key string, value interface{}) error {
	m, ok := value.(*map[string]bool)
	if ok {
		rows, err := stmtConfig.Query(key)
		if err != nil {
			return err
		}
		defer rows.Close()
		for rows.Next() {
			var s string
			err = rows.Scan(&s)
			if err != nil {
				return err
			}
			(*m)[s] = true
		}
		return nil
	}
	row := stmtConfig.QueryRow(key)
	err := row.Scan(value)
	if err == sql.ErrNoRows {
		err = nil
	}
	return err
}

func setconfig(key string, val interface{}) error {
	db := opendatabase()
	db.Exec("delete from config where key = ?", key)
	_, err := db.Exec("insert into config (key, value) values (?, ?)", key, val)
	return err
}
