//
// Copyright (c) 2024 Ted Unangst <tedu@tedunangst.com>
//
// Permission to use, copy, modify, and distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

package main

import (
	"bytes"
	"crypto/rand"
	"database/sql"
	_ "embed"
	"encoding/json"
	"fmt"
	"regexp"
	"strings"
	"time"

	"humungus.tedunangst.com/r/webs/gencache"
)

//go:embed schema.sql
var sqlSchema string

var dbtimeformat = "2006-01-02 15:04:05"

var stmtConfig *sql.Stmt
var stmtRepos *sql.Stmt

var stmtGetDeliveries, stmtAddDelivery, stmtLoadDelivery, stmtZapDelivery, stmtDelinquentCheck, stmtDelinquentUpdate *sql.Stmt
var stmtFindSeckey, stmtSavePubkey, stmtFindPubkey *sql.Stmt
var stmtSaveSeckey, stmtSaveFollow, stmtDeleteFollow, stmtFindFollow, stmtFindFollowers *sql.Stmt
var stmtFindPerson, stmtFindPersonByAPid, stmtGetPerson, stmtSavePerson, stmtUpdatePerson *sql.Stmt

func prepareStatements(db *sql.DB) {
	prep := func(s string) *sql.Stmt {
		stmt, err := db.Prepare(s)
		if err != nil {
			elog.Panicf("error %s: %s", err, s)
		}
		return stmt
	}
	stmtRepos = prep("select repoid, name, description, orderid from repos order by orderid asc")
	stmtAddDelivery = prep("insert into deliveries (dt, tries, rcpt, data) values (?, ?, ?, ?)")
	stmtGetDeliveries = prep("select deliveryid, dt from deliveries")
	stmtLoadDelivery = prep("select tries, rcpt, data from deliveries where deliveryid = ?")
	stmtZapDelivery = prep("delete from deliveries where deliveryid = ?")

	stmtDelinquentCheck = prep("select deliveryid, data from deliveries where rcpt = ?")
	stmtDelinquentUpdate = prep("update deliveries set data = ? where deliveryid = ?")
	stmtSavePubkey = prep("insert into pubkeys (url, owner, pubkey, dt) values (?, ?, ?, ?)")
	stmtFindPubkey = prep("select pubkey from pubkeys where url = ?")
	stmtSaveSeckey = prep("insert into seckeys (owner, seckey) values (?, ?)")
	stmtFindSeckey = prep("select seckey from seckeys where owner = ?")
	stmtSaveFollow = prep("insert into follows (folxid, who, what) values (?, ?, ?)")
	stmtDeleteFollow = prep("delete from follows where who = ? and what = ?")
	stmtFindFollow = prep("select folxid from follows where who = ? and what = ?")
	stmtFindFollowers = prep("select who from follows where what = ?")

	personcols := "personid, pname, display, papid, personmeta"
	stmtFindPerson = prep("select " + personcols + " from persons where pname = ?")
	stmtFindPersonByAPid = prep("select " + personcols + " from persons where papid = ?")
	stmtGetPerson = prep("select " + personcols + " from persons where personid = ?")
	stmtSavePerson = prep("insert into persons (" + personcols[10:] + ") values (?, ?, ?, ?)")
	stmtUpdatePerson = prep("update persons set (" + personcols[10:] + ") = (?, ?, ?, ?) where personid = ?")

}

func dbSaveFollow(folxid, who, what string) error {
	_, err := stmtSaveFollow.Exec(folxid, who, what)
	return err
}

func dbDeleteFollow(who, what string) error {
	_, err := stmtDeleteFollow.Exec(who, what)
	return err
}

func dbClearFollows(rcpt string) {
}

func dbSaveSeckey(owner, seckey string) error {
	_, err := stmtSaveSeckey.Exec(owner, seckey)
	return err
}

func dbSavePubkey(url, owner, pubkey string) error {
	dt := time.Now().UTC().Format(dbtimeformat)
	_, err := stmtSavePubkey.Exec(url, owner, pubkey, dt)
	return err
}

func dbFindPubkey(url string) string {
	row := stmtFindPubkey.QueryRow(url)
	var pubkey string
	err := row.Scan(&pubkey)
	if err != nil && err != sql.ErrNoRows {
		elog.Printf("error scanning pubkey: %s", err)
	}
	return pubkey
}

func hastombstone(url string) bool {
	return false
}

var re_mapuser = regexp.MustCompile(`<(.*)>`)

func mapuser(user string) string {
	m := re_mapuser.FindStringSubmatch(user)
	if len(m) == 2 {
		user = fmt.Sprintf("%su/%s", serverPrefix, m[1])
	}
	return user
}

func xcelerate(b []byte) string {
	letters := "BCDFGHJKLMNPQRSTVWXYZbcdfghjklmnpqrstvwxyz1234567891234567891234"
	for i, c := range b {
		b[i] = letters[c&63]
	}
	s := string(b)
	return s
}

func generateXid() string {
	var b [18]byte
	rand.Read(b[:])
	return xcelerate(b[:])
}

func findfollowers(what string) []string {
	whos := make([]string, 0, 32)
	rows, err := stmtFindFollowers.Query(what)
	return scanstrings(whos, rows, err)
}

func scanstrings(arr []string, rows *sql.Rows, err error) []string {
	if err != nil {
		elog.Printf("error getting strings: %s", err)
		return arr
	}
	defer rows.Close()
	for rows.Next() {
		var s string
		err = rows.Scan(&s)
		if err != nil {
			elog.Printf("error scanning string: %s", err)
			continue
		}
		arr = append(arr, s)
	}
	return arr
}

var personInvalidator gencache.Invalidator[string]
var personCache = gencache.New(gencache.Options[string, *Person]{
	Fill: func(ref string) (*Person, bool) {
		person := dbFindPerson(ref)
		return person, true
	},
	Invalidator: &personInvalidator, Limit: 600})

func findperson(ref string) *Person {
	person, _ := personCache.Get(ref)
	return person
}

func dbFindPerson(ref string) *Person {
	var person *Person
	if strings.HasPrefix(ref, "https://") {
		row := stmtFindPersonByAPid.QueryRow(ref)
		person = scanperson(row)
	} else {
		row := stmtFindPerson.QueryRow(ref)
		person = scanperson(row)
	}
	return person
}

func dbSavePerson(person *Person) error {
	if person.APid == "" {
		person.APid = fmt.Sprintf("https://%s/u/%s", serverName, person.Name)
	}
	meta := personMeta{
		About:    person.About,
		Format:   person.Format,
		Inbox:    person.Inbox,
		ShareBox: person.ShareBox,
	}
	j, err := jsonify(&meta)
	if err == nil {
		if person.ID == 0 {
			var res sql.Result
			res, err = stmtSavePerson.Exec(person.Name, person.Display, person.APid, j)
			if err == nil {
				personid, _ := res.LastInsertId()
				person.ID = PersonID(personid)
			}
		} else {
			_, err = stmtUpdatePerson.Exec(person.Name, person.Display, person.APid, j, person.ID)
		}
	}
	personInvalidator.Clear(person.Name)
	personInvalidator.Clear(person.APid)
	return err
}

func scanperson(row *sql.Row) *Person {
	person := new(Person)
	var j string
	err := row.Scan(&person.ID, &person.Name, &person.Display, &person.APid, &j)
	if err == sql.ErrNoRows {
		return nil
	}
	if err == nil {
		var meta personMeta
		err = unjsonify(j, &meta)
		if err == nil {
			person.About = meta.About
			person.Format = meta.Format
			person.Inbox = meta.Inbox
			person.ShareBox = meta.ShareBox
		}
	}
	if err != nil {
		elog.Printf("error scanning person: %s", err)
		return nil
	}
	return person
}

func jsonify(what interface{}) (string, error) {
	var buf bytes.Buffer
	e := json.NewEncoder(&buf)
	e.SetEscapeHTML(false)
	e.SetIndent("", "")
	err := e.Encode(what)
	return buf.String(), err
}

func unjsonify(s string, dest interface{}) error {
	d := json.NewDecoder(strings.NewReader(s))
	err := d.Decode(dest)
	return err
}

type personMeta struct {
	About    string `json:",omitempty"`
	Format   string `json:",omitempty"`
	Inbox    string `json:",omitempty"`
	ShareBox string `json:",omitempty"`
}
