//
// Copyright (c) 2018 Ted Unangst <tedu@tedunangst.com>
//
// Permission to use, copy, modify, and distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

package main

import (
	"io"
	"os/exec"
	"sync"

	"humungus.tedunangst.com/r/gerc"
	"humungus.tedunangst.com/r/webs/gate"
)

var pullgate = gate.NewLimiter(6)

// A connection to an hg repo
type WireConn struct {
	Name  string
	Wpipe io.WriteCloser
	Rpipe io.ReadCloser
	Proc  *exec.Cmd
}

func newWireConn(reponame string) *WireConn {
	dlog.Printf("starting wire server for %s", reponame)
	proc := exec.Command("hg", "-R", reponame, "serve", "--stdio")
	proc.Dir = "repos"
	wpipe, _ := proc.StdinPipe()
	rpipe, _ := proc.StdoutPipe()
	err := proc.Start()
	if err != nil {
		elog.Printf("error running cmd: %s", err)
		return nil
	}

	conn := &WireConn{
		Name:  reponame,
		Wpipe: wpipe,
		Rpipe: rpipe,
		Proc:  proc,
	}

	return conn
}

func (conn *WireConn) Finish() {
	if conn.Wpipe != nil {
		conn.Wpipe.Close()
	}
	conn.Rpipe.Close()
	conn.Proc.Wait()
}

func getwireconn(reponame string) *WireConn {
	gercpoolmtx.Lock()
	defer gercpoolmtx.Unlock()
	pool := gercpools[reponame]
	if pool == nil {
		return nil
	}
	pullgate.Start()
	return newWireConn(reponame)
}

func putbackconn(conn *WireConn) {
	pullgate.Finish()
	conn.Finish()
}

var gercpoolmtx sync.Mutex

type gercPool []*gerc.Repo

var gercpools = make(map[string]gercPool)
var revgerc = make(map[*gerc.Repo]string)

func getgerc(reponame string) *gerc.Repo {
	gercpoolmtx.Lock()
	defer gercpoolmtx.Unlock()
	pool := gercpools[reponame]
	if pool == nil {
		return nil
	}
	plen := len(pool)
	if plen == 0 {
		repo, _ := gerc.Open("repos/" + reponame)
		revgerc[repo] = reponame
		return repo
	}

	repo := pool[plen-1]
	gercpools[reponame] = pool[0 : plen-1]
	return repo
}

func putgerc(repo *gerc.Repo) {
	gercpoolmtx.Lock()
	defer gercpoolmtx.Unlock()
	reponame := revgerc[repo]
	if reponame == "" {
		return
	}
	pool := gercpools[reponame]
	gercpools[reponame] = append(pool, repo)
}

func (gp gercPool) flush() {
	for _, repo := range gp {
		repo.Close()
		delete(revgerc, repo)
	}
}
