//
// Copyright (c) 2019 Ted Unangst <tedu@tedunangst.com>
//
// Permission to use, copy, modify, and distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

package main

import (
	"fmt"
	"log"
	"net"
	"os"
	"sort"
	"strconv"
	"strings"

	"github.com/gdamore/tcell/v2"
	"github.com/rivo/tview"
)

func adminscreen() {
	allrepos := loadrepos()
	allusers := getusers()

	var deletedrepos []*RepoRow
	var deletedusers []*User

	var logbuf strings.Builder
	log.SetOutput(&logbuf)
	defer func() {
		fmt.Fprintf(os.Stderr, logbuf.String())
	}()

	app := tview.NewApplication()
	var maindriver func(event *tcell.EventKey) *tcell.EventKey
	var showuserlist func()

	var st tcell.Style
	st = st.Foreground(tcell.ColorBlack).Background(tcell.ColorGreen)
	table := tview.NewTable().SetFixed(1, 0).SetSelectable(true, false).
		SetSelectedStyle(st)

	mainframe := tview.NewFrame(table)
	mainframe.AddText(tview.Escape("humungus - [a] add [d] delete [e] edit [s] save and quit [^C] abort"),
		true, 0, tcell.ColorGreen)
	mainframe.AddText(tview.Escape("           [o] server options [u] user list [r] repo list"),
		true, 0, tcell.ColorGreen)
	mainframe.SetBorders(1, 0, 1, 0, 4, 0)

	pad := func(s string) string {
		return fmt.Sprintf("  %-20s", s)
	}

	dupecell := func(base *tview.TableCell) *tview.TableCell {
		rv := new(tview.TableCell)
		*rv = *base
		return rv
	}

	showtable := func() {
		table.Clear()
		sort.Slice(allrepos, func(i, j int) bool {
			return allrepos[i].Order < allrepos[j].Order
		})

		row := 0
		{
			col := 0
			headcell := tview.TableCell{
				Color:         tcell.ColorWhite,
				NotSelectable: true,
			}
			cell := dupecell(&headcell)
			cell.Text = "order"
			table.SetCell(row, col, cell)
			col++
			cell = dupecell(&headcell)
			cell.Text = "  name"
			table.SetCell(row, col, cell)
			col++
			cell = dupecell(&headcell)
			cell.Text = "description"
			table.SetCell(row, col, cell)
			col++

			row++
		}

		for _, repo := range allrepos {
			col := 0
			headcell := tview.TableCell{
				Color: tcell.ColorWhite,
			}
			cell := dupecell(&headcell)
			cell.Text = fmt.Sprintf("%d", repo.Order)
			cell.Align = tview.AlignRight
			table.SetCell(row, col, cell)
			col++
			cell = dupecell(&headcell)
			cell.Text = pad(repo.Name)
			table.SetCell(row, col, cell)
			col++
			cell = dupecell(&headcell)
			cell.Text = repo.Description
			table.SetCell(row, col, cell)
			col++

			row++
		}

		app.SetInputCapture(maindriver)
		app.SetRoot(mainframe, true)
	}

	arrowadapter := func(event *tcell.EventKey) *tcell.EventKey {
		switch event.Key() {
		case tcell.KeyDown:
			return tcell.NewEventKey(tcell.KeyTab, '\t', tcell.ModNone)
		case tcell.KeyUp:
			return tcell.NewEventKey(tcell.KeyBacktab, '\t', tcell.ModNone)
		}
		return event
	}

	namebox := tview.NewInputField().SetLabel("name").SetFieldWidth(20)
	descbox := tview.NewInputField().SetLabel("description").SetFieldWidth(60)
	permbox := tview.NewInputField().SetLabel("permissions").SetFieldWidth(60)
	keybox := tview.NewInputField().SetLabel("pubkey").SetFieldWidth(60)
	orderbox := tview.NewInputField().SetLabel("order").SetFieldWidth(10)

	hadchanges := false

	showform := func(which int, savefn func()) {
		editform := tview.NewForm()
		editform.AddButton("save", nil)
		editform.AddButton("cancel", nil)
		savebutton := editform.GetButton(0)
		savebutton.SetSelectedFunc(savefn)
		editform.SetFieldTextColor(tcell.ColorBlack)
		editform.SetFieldBackgroundColor(tcell.ColorGreen)
		editform.SetLabelColor(tcell.ColorWhite)
		editform.SetButtonTextColor(tcell.ColorGreen)
		editform.SetButtonBackgroundColor(tcell.ColorBlack)
		editform.GetButton(1).SetSelectedFunc(showtable)
		editform.SetCancelFunc(showtable)

		editform.AddFormItem(namebox)
		switch which {
		case 0:
			editform.AddFormItem(descbox)
			editform.AddFormItem(orderbox)
		case 1:
			editform.AddFormItem(permbox)
			editform.AddFormItem(keybox)
		}
		app.SetInputCapture(arrowadapter)
		app.SetRoot(editform, true)
	}

	editrepo := func(r *RepoRow) {
		namebox.SetText(r.Name)
		descbox.SetText(r.Description)
		orderbox.SetText(fmt.Sprintf("%d", r.Order))
		savefn := func() {
			r.Name = namebox.GetText()
			r.Description = descbox.GetText()
			r.Order, _ = strconv.Atoi(orderbox.GetText())
			hadchanges = true
			showtable()
		}
		showform(0, savefn)
	}

	edituser := func(u *User) {
		namebox.SetText(u.Name)
		permbox.SetText(u.Perms)
		keybox.SetText(u.Pubkey)
		savefn := func() {
			u.Name = namebox.GetText()
			u.Perms = permbox.GetText()
			u.Pubkey = keybox.GetText()
			hadchanges = true
			showuserlist()
		}
		showform(1, savefn)
	}

	addrepo := func() {
		namebox.SetText("")
		descbox.SetText("")
		orderbox.SetText("10")
		savefn := func() {
			r := new(RepoRow)
			r.Name = namebox.GetText()
			r.Description = descbox.GetText()
			r.Order, _ = strconv.Atoi(orderbox.GetText())
			allrepos = append(allrepos, r)
			hadchanges = true
			showtable()
		}
		showform(0, savefn)
	}
	adduser := func() {
		namebox.SetText("")
		permbox.SetText("")
		keybox.SetText("")
		savefn := func() {
			user := new(User)
			user.Name = namebox.GetText()
			user.Perms = permbox.GetText()
			user.Pubkey = keybox.GetText()
			allusers = append(allusers, user)
			hadchanges = true
			showuserlist()
		}
		showform(1, savefn)
	}
	saveandquit := func() {
		app.Stop()
		if !hadchanges {
			return
		}
		db := opendatabase()

		tx, err := db.Begin()
		if err != nil {
			fmt.Printf("error starting tx: %s\n", err)
			return
		}
		for _, r := range allrepos {
			if r.ID == 0 {
				_, err = tx.Exec("insert into repos (name, description, orderid) values (?, ?, ?)",
					r.Name, r.Description, r.Order)
			} else {
				_, err = tx.Exec("update repos set name = ?, description = ?, orderid = ? where repoid = ?",
					r.Name, r.Description, r.Order, r.ID)
			}
			if err != nil {
				fmt.Printf("error saving row: %s\n", err)
				return
			}
		}
		for _, u := range allusers {
			if u.ID == 0 {
				_, err = tx.Exec("insert into users (name, pubkey, perms) values (?, ?, ?)",
					u.Name, u.Pubkey, u.Perms)
			} else {
				_, err = tx.Exec("update users set name = ?, pubkey = ?, perms = ? where userid = ?",
					u.Name, u.Pubkey, u.Perms, u.ID)
			}
			if err != nil {
				fmt.Printf("error saving user: %s\n", err)
				return
			}
		}
		for _, r := range deletedrepos {
			_, err := tx.Exec("delete from repos where repoid = ?", r.ID)
			if err != nil {
				fmt.Printf("error deleting row: %s\n", err)
				return
			}
		}
		for _, u := range deletedusers {
			_, err := tx.Exec("delete from users where userid = ?", u.ID)
			if err != nil {
				fmt.Printf("error deleting user: %s\n", err)
				return
			}
		}
		err = tx.Commit()
		if err != nil {
			fmt.Printf("error committing tx: %s\n", err)
			return
		}
		sendcmd("init")
	}

	editserveroptions := func() {
		ssh := false
		getconfig("sshenabled", &ssh)
		var addr string
		getconfig("sshlistenaddr", &addr)
		editform := tview.NewForm()
		en := "no"
		if ssh {
			en = "yes"
		}

		sshbox := tview.NewInputField().SetLabel("ssh enabled").SetFieldWidth(10).SetText(en)
		addrbox := tview.NewInputField().SetLabel("ssh address").SetFieldWidth(40).SetText(addr)
		editform.AddButton("save", nil)
		editform.AddButton("cancel", nil)
		savebutton := editform.GetButton(0)
		editform.SetFieldTextColor(tcell.ColorBlack)
		editform.SetFieldBackgroundColor(tcell.ColorGreen)
		editform.SetLabelColor(tcell.ColorWhite)
		editform.SetButtonTextColor(tcell.ColorGreen)
		editform.SetButtonBackgroundColor(tcell.ColorBlack)
		editform.GetButton(1).SetSelectedFunc(showtable)
		editform.SetCancelFunc(showtable)

		editform.AddFormItem(sshbox)
		editform.AddFormItem(addrbox)
		savefn := func() {
			if sshbox.GetText() == "yes" {
				setconfig("sshlistenaddr", addrbox.GetText())
				var key []byte
				getconfig("sshprivatekey", &key)
				if len(key) == 0 {
					sshgenhostkey()
				}
				setconfig("sshenabled", true)
			} else {
				setconfig("sshenabled", false)
			}
			showtable()
		}
		savebutton.SetSelectedFunc(savefn)
		app.SetInputCapture(arrowadapter)
		app.SetRoot(editform, true)
	}
	userdriver := func(event *tcell.EventKey) *tcell.EventKey {
		switch event.Rune() {
		case 'a':
			adduser()
		case 'd':
			hadchanges = true
			row, _ := table.GetSelection()
			table.RemoveRow(row)
			if row == table.GetRowCount() {
				table.Select(row-1, 0)
			}
			row--
			if row >= 0 && row < len(allusers) {
				user := allusers[row]
				if row == len(allusers)-1 {
					allusers = allusers[:row]
				} else {
					allusers = append(allusers[:row], allusers[row+1:]...)
				}
				deletedusers = append(deletedusers, user)
			}
		case 'e':
			row, _ := table.GetSelection()
			row--
			if row >= 0 && row < len(allusers) {
				u := allusers[row]
				edituser(u)
			}
		case 'r':
			showtable()
		case 's':
			saveandquit()
			return nil
		}
		return event
	}

	showuserlist = func() {
		table.Clear()

		row := 0
		{
			col := 0
			headcell := tview.TableCell{
				Color:         tcell.ColorWhite,
				NotSelectable: true,
			}
			cell := dupecell(&headcell)
			cell.Text = "name        "
			table.SetCell(row, col, cell)
			col++
			cell = dupecell(&headcell)
			cell.Text = "perms"
			table.SetCell(row, col, cell)
			col++

			row++
		}

		for _, user := range allusers {
			col := 0
			headcell := tview.TableCell{
				Color: tcell.ColorWhite,
			}
			cell := dupecell(&headcell)
			cell.Text = user.Name
			table.SetCell(row, col, cell)
			col++
			cell = dupecell(&headcell)
			cell.Text = user.Perms
			table.SetCell(row, col, cell)

			row++
		}
		table.Select(0, 0)
		app.SetInputCapture(userdriver)
		app.SetRoot(mainframe, true)
	}

	maindriver = func(event *tcell.EventKey) *tcell.EventKey {
		switch event.Rune() {
		case 'a':
			addrepo()
		case 'd':
			hadchanges = true
			row, _ := table.GetSelection()
			table.RemoveRow(row)
			if row == table.GetRowCount() {
				table.Select(row-1, 0)
			}
			row--
			if row >= 0 && row < len(allrepos) {
				r := allrepos[row]
				if row == len(allrepos)-1 {
					allrepos = allrepos[:row]
				} else {
					allrepos = append(allrepos[:row], allrepos[row+1:]...)
				}
				deletedrepos = append(deletedrepos, r)
			}
		case 'e':
			row, _ := table.GetSelection()
			row--
			if row >= 0 && row < len(allrepos) {
				r := allrepos[row]
				editrepo(r)
			}
		case 'o':
			editserveroptions()
		case 'u':
			showuserlist()
		case 's':
			saveandquit()
			return nil
		}
		return event
	}

	showtable()
	app.Run()
}

func sendcmd(cmd string) error {
	sockname := "humungus.sock"
	sock, err := net.Dial("unix", sockname)
	if err != nil {
		return err
	}
	sock.Write([]byte(cmd))
	sock.Close()
	return nil
}

func controlcommand(args []string) {
	if len(args) == 0 {
		fmt.Printf("missing command\n")
		os.Exit(1)
	}
	switch args[0] {
	case "stop":
		err := sendcmd("stop")
		if err != nil {
			fmt.Printf("could not connect: %s\n", err)
		}
	case "refresh":
		err := sendcmd("init")
		if err != nil {
			fmt.Printf("could not connect: %s\n", err)
		}
	default:
		fmt.Printf("unknown command\n")
		os.Exit(1)
	}
}
