//
// Copyright (c) 2024 Ted Unangst <tedu@tedunangst.com>
//
// Permission to use, copy, modify, and distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

package main

import (
	"fmt"
	"html"
	"net/http"
	"regexp"
	"strings"
	"time"

	"humungus.tedunangst.com/r/gerc"
	"humungus.tedunangst.com/r/webs/junk"
)

var apContentType = `application/ld+json; profile="https://www.w3.org/ns/activitystreams"`
var acceptNames = []string{
	`application/ld+json`,
	`application/activity+json`,
}
var apAcceptType = apContentType
var apContext = "https://www.w3.org/ns/activitystreams"
var apPublic = "https://www.w3.org/ns/activitystreams#Public"

func isAPtype(ct string) bool {
	for _, at := range acceptNames {
		if strings.HasPrefix(ct, at) {
			return true
		}
	}
	return false
}

func isAPrequest(r *http.Request) bool {
	ct := strings.ToLower(r.Header.Get("Accept"))
	return isAPtype(ct)
}
func isAPpost(r *http.Request) bool {
	ct := strings.ToLower(r.Header.Get("Content-Type"))
	return isAPtype(ct)
}

func isMine(id string) bool {
	return strings.HasPrefix(id, serverPrefix)
}

func getType(obj junk.Junk) string {
	if i, ok := obj["type"]; ok {
		if s, ok := i.(string); ok {
			return s
		}
		if a, ok := i.([]interface{}); ok && len(a) > 0 {
			if s, ok := a[0].(string); ok {
				return s
			}
		}
	}
	return ""
}

var re_urlhost = regexp.MustCompile("https://([^/ #)]+)")

func originate(u string) string {
	m := re_urlhost.FindStringSubmatch(u)
	if len(m) > 1 {
		return strings.ToLower(m[1])
	}
	return ""
}

func newActivity(typ string, id string) junk.Junk {
	j := junk.New()
	j["@context"] = apContext
	j["type"] = typ
	j["id"] = id
	return j
}

type PersonID int64

type Person struct {
	ID       PersonID
	APid     string
	Name     string
	Display  string
	About    string
	Format   string
	Inbox    string
	ShareBox string
}

func jsonCommit(reponame string, change *gerc.Change) junk.Junk {
	apid := serverURL("/r/%s/v/%s", reponame, hexchange(change))
	repoid := serverURL("/r/%s", reponame) // todo: user
	j := newActivity("Commit", apid)
	j["attributedTo"] = repoid
	j["to"] = apPublic
	j["cc"] = []string{repoid + "/followers"}
	j["committedBy"] = mapuser(change.User)
	j["published"] = change.Date.Format(time.RFC3339)
	j["committed"] = change.Date.Format(time.RFC3339)
	j["hash"] = fmt.Sprintf("%x", change.NodeID)
	j["summary"] = html.EscapeString(change.Message[0])
	desc := junk.New()
	desc["mediaType"] = "text/plain"
	desc["content"] = strings.Join(change.Message, "\n")
	j["description"] = desc
	j["mediaType"] = "text/plain"
	j["content"] = change.Diff
	j["context"] = repoid

	return j
}

func jsonPerson(username string) junk.Junk {
	apid := serverURL("/u/%s", username)
	j := newActivity("Person", apid)
	j["inbox"] = apid + "/inbox"
	j["outbox"] = apid + "/outbox"
	ep := junk.New()
	ep["sharedInbox"] = serverURL("/inbox")
	j["endpoints"] = ep
	j["followers"] = apid + "/followers"
	j["following"] = apid + "/following"
	j["name"] = username
	j["preferredUsername"], _, _ = strings.Cut(username, "@")
	j["summary"] = ""
	if false { // todo
		pk := junk.New()
		pk["id"] = apid + "#key"
		pk["owner"] = apid
		pk["publicKeyPem"] = dbFindPubkey(apid + "#key")
		j["publicKey"] = pk
	}
	return j
}

func jsonRepository(reponame string) junk.Junk {
	apid := serverURL("/r/%s", reponame)
	j := newActivity("Repository", apid)
	j["inbox"] = apid + "/inbox"
	j["outbox"] = apid + "/outbox"
	ep := junk.New()
	ep["sharedInbox"] = serverURL("/inbox")
	j["endpoints"] = ep
	j["followers"] = apid + "/followers"
	j["following"] = apid + "/following"
	j["name"] = reponame
	j["preferredUsername"] = reponame
	j["summary"] = ""
	pk := junk.New()
	pk["id"] = apid + "#key"
	pk["owner"] = apid
	pk["publicKeyPem"] = dbFindPubkey(apid + "#key")
	j["publicKey"] = pk
	return j
}

func serverURL(u string, args ...interface{}) string {
	return fmt.Sprintf("https://"+serverName+u, args...)
}

func jsonCreate(who string, where string, what junk.Junk) junk.Junk {
	return jsonActivity("Create", who, where, what)
}

func jsonActivity(action, who, where string, what interface{}) junk.Junk {
	j := newActivity(action, who+"/activity/"+generateXid())
	j["actor"] = who
	j["to"] = apPublic
	if where != "" {
		j["cc"] = []string{where}
		j["audience"] = where
	}
	j["object"] = what
	return j
}

func personForAPid(url string) *Person {
	person := findperson(url)
	if person != nil {
		return person
	}
	j, err := fetchActivity(url, slowTimeout)
	if err != nil {
		dlog.Printf("error fetching person %s: %s", url, err)
		return nil
	}
	return apHandlePerson(originate(url), j)
}

func apHandlePerson(origin string, j junk.Junk) *Person {
	typ := getType(j)
	if typ != "Person" && typ != "Service" {
		dlog.Printf("it's not a person! %s", j.ToString())
		return nil
	}
	id, _ := j.GetString("id")
	if originate(id) != origin {
		dlog.Printf("impersonated person %s from %s", id, origin)
		return nil
	}
	if dup := findperson(id); dup != nil {
		dlog.Printf("duplicate person %s", id)
		return dup
	}
	name := getString(j, "preferredUsername", 100)
	display := getString(j, "name", 100)
	summary := getString(j, "summary", 5000)
	inbox, _ := j.GetString("inbox")
	sharebox, _ := j.GetString("endpoints", "sharedInbox")
	person := &Person{
		Name:     fmt.Sprintf("%s@%s", name, origin),
		Display:  display,
		APid:     id,
		About:    summary,
		Format:   "html",
		Inbox:    inbox,
		ShareBox: sharebox,
	}
	err := dbSavePerson(person)
	if err != nil {
		elog.Printf("error saving person: %s", err)
		return nil
	}
	dlog.Printf("saved person: %s", person.Name)

	return person
}

func getString(obj junk.Junk, name string, limit int) string {
	val, _ := obj.GetString(name)
	if len(val) > limit {
		val = val[:limit]
	}
	return val
}
