/*
 *  step_theme_main.c:	GTKstep initialization code
 * 
 *  Written by:		Ullrich Hafner
 *		
 *  Copyright (C) 1999 Ullrich Hafner <hafner@bigfoot.de>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the
 *  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 */

/*
 *  $Date: 1999/09/28 20:06:51 $
 *  $Author: hafner $
 *  $Revision: 1.7 $
 *  $State: Exp $
 */

#include <gtk/gtk.h>
#include <gmodule.h>
#include "../config.h"

#define ARROW_SIZE	16
#define SCROLLBAR_WIDTH 16
#define MINSLIDER_WIDTH 13
#define INDICATOR_SIZE	16
#define SCALE_WIDTH     16
#define LIGHTNESS_MULT  1.5
#define DARKNESS_MULT   0.5
#define SCROLL_DELAY_LENGTH  300
#define SCROLLBAR_SPACING(w) (GTK_SCROLLED_WINDOW_CLASS (GTK_OBJECT (w)->klass)->scrollbar_spacing)
#define RANGE_CLASS(widget)  (GTK_RANGE_CLASS(GTK_OBJECT(widget)->klass))

/* Theme functions to export */
void
theme_init (GtkThemeEngine *engine);
void
theme_exit (void);

/* Exported vtable from th_draw */
extern GtkStyleClass step_class;

void (*old_hpaned_size_allocate) (GtkWidget     *widget,
				  GtkAllocation *allocation);

static void
step_hpaned_size_allocate (GtkWidget     *widget,
			   GtkAllocation *allocation);

void (*old_vpaned_size_allocate) (GtkWidget     *widget,
				  GtkAllocation *allocation);

static void
step_vpaned_size_allocate (GtkWidget     *widget,
			   GtkAllocation *allocation);

static void
step_button_draw (GtkWidget    *widget,
		  GdkRectangle *area);
static gint
step_button_expose (GtkWidget      *widget,
		    GdkEventExpose *event);
static void
step_button_paint (GtkWidget    *widget,
		   GdkRectangle *area);
static void (*old_parent_set) (GtkWidget      *widget,
			       GtkWidget      *previous_parent);
void
step_vscrollbar_parent_set (GtkWidget      *widget,
			    GtkWidget      *previous_parent);
static void
step_range_remove_timer (GtkRange *range);
static void
step_vscrollbar_calc_slider_size (GtkVScrollbar *vscrollbar);
static void
step_hscrollbar_calc_slider_size (GtkHScrollbar *hscrollbar);
static void
step_range_vslider_update (GtkRange *range);
static void
step_range_hslider_update (GtkRange *range);
static gint
step_range_vtrough_click (GtkRange *range,
			  gint      x,
			  gint      y,
			  gfloat   *jump_perc);
static gint
step_range_htrough_click (GtkRange *range,
			  gint      x,
			  gint      y,
			  gfloat   *jump_perc);
static void
step_range_vmotion (GtkRange *range,
		    gint      xdelta,
		    gint      ydelta);
static void
step_range_hmotion (GtkRange *range,
		    gint      xdelta,
		    gint      ydelta);
static void
step_range_trough_vdims (GtkRange *range,
			 gint     *top,
			 gint     *bottom);
static void
step_range_trough_hdims (GtkRange *range,
			 gint     *left,
			 gint     *right);
static void
step_vscrollbar_slider_update (GtkRange *range);
static void
step_hscrollbar_slider_update (GtkRange *range);
static void
step_vscrollbar_size_allocate (GtkWidget     *widget,
                              GtkAllocation *allocation);
static void
step_hscrollbar_size_allocate (GtkWidget     *widget,
			       GtkAllocation *allocation);
static void
step_vscrollbar_realize (GtkWidget *widget);
static void
step_hscrollbar_realize (GtkWidget *widget);

void (*old_scrolled_window_map) (GtkWidget *widget);
static void
step_scrolled_window_map (GtkWidget *widget);
void (*old_scrolled_window_size_allocate) (GtkWidget     *widget,
					   GtkAllocation *allocation);
static void
step_scrolled_window_size_allocate (GtkWidget     *widget,
				    GtkAllocation *allocation);
static void
step_scrolled_window_relative_allocation (GtkWidget     *widget,
					  GtkAllocation *allocation);

static void
step_style_shade (GdkColor *a,
		  GdkColor *b,
		  gdouble   k);
static void
hls_to_rgb (gdouble *h,
	    gdouble *l,
	    gdouble *s);
static void
rgb_to_hls (gdouble *r,
	    gdouble *g,
	    gdouble *b);

static void
step_style_shade (GdkColor *a,
		  GdkColor *b,
		  gdouble   k)
{
  gdouble red;
  gdouble green;
  gdouble blue;

  red = (gdouble) a->red / 65535.0;
  green = (gdouble) a->green / 65535.0;
  blue = (gdouble) a->blue / 65535.0;

  rgb_to_hls (&red, &green, &blue);

  green *= k;
  if (green > 1.0)
    green = 1.0;
  else if (green < 0.0)
    green = 0.0;

  blue *= k;
  if (blue > 1.0)
    blue = 1.0;
  else if (blue < 0.0)
    blue = 0.0;

  hls_to_rgb (&red, &green, &blue);

  b->red = red * 65535.0;
  b->green = green * 65535.0;
  b->blue = blue * 65535.0;
}

static void
hls_to_rgb (gdouble *h,
	    gdouble *l,
	    gdouble *s)
{
  gdouble hue;
  gdouble lightness;
  gdouble saturation;
  gdouble m1, m2;
  gdouble r, g, b;

  lightness = *l;
  saturation = *s;

  if (lightness <= 0.5)
    m2 = lightness * (1 + saturation);
  else
    m2 = lightness + saturation - lightness * saturation;
  m1 = 2 * lightness - m2;

  if (saturation == 0)
    {
      *h = lightness;
      *l = lightness;
      *s = lightness;
    }
  else
    {
      hue = *h + 120;
      while (hue > 360)
	hue -= 360;
      while (hue < 0)
	hue += 360;

      if (hue < 60)
	r = m1 + (m2 - m1) * hue / 60;
      else if (hue < 180)
	r = m2;
      else if (hue < 240)
	r = m1 + (m2 - m1) * (240 - hue) / 60;
      else
	r = m1;

      hue = *h;
      while (hue > 360)
	hue -= 360;
      while (hue < 0)
	hue += 360;

      if (hue < 60)
	g = m1 + (m2 - m1) * hue / 60;
      else if (hue < 180)
	g = m2;
      else if (hue < 240)
	g = m1 + (m2 - m1) * (240 - hue) / 60;
      else
	g = m1;

      hue = *h - 120;
      while (hue > 360)
	hue -= 360;
      while (hue < 0)
	hue += 360;

      if (hue < 60)
	b = m1 + (m2 - m1) * hue / 60;
      else if (hue < 180)
	b = m2;
      else if (hue < 240)
	b = m1 + (m2 - m1) * (240 - hue) / 60;
      else
	b = m1;

      *h = r;
      *l = g;
      *s = b;
    }
}

static void
rgb_to_hls (gdouble *r,
	    gdouble *g,
	    gdouble *b)
{
  gdouble min;
  gdouble max;
  gdouble red;
  gdouble green;
  gdouble blue;
  gdouble h, l, s;
  gdouble delta;

  red = *r;
  green = *g;
  blue = *b;

  if (red > green)
    {
      if (red > blue)
	max = red;
      else
	max = blue;

      if (green < blue)
	min = green;
      else
	min = blue;
    }
  else
    {
      if (green > blue)
	max = green;
      else
	max = blue;

      if (red < blue)
	min = red;
      else
	min = blue;
    }

  l = (max + min) / 2;
  s = 0;
  h = 0;

  if (max != min)
    {
      if (l <= 0.5)
	s = (max - min) / (max + min);
      else
	s = (max - min) / (2 - max - min);

      delta = max -min;
      if (red == max)
	h = (green - blue) / delta;
      else if (green == max)
	h = 2 + (blue - red) / delta;
      else if (blue == max)
	h = 4 + (red - green) / delta;

      h *= 60;
      if (h < 0.0)
	h += 360;
    }

  *r = h;
  *g = l;
  *b = s;
}

/* internals */
static guint
theme_parse_rc_style (GScanner *scanner, GtkRcStyle *rc_style)
{
   guint token;

   token = g_scanner_peek_next_token (scanner);
   while (token != G_TOKEN_RIGHT_CURLY)
   {
      switch (token)
      {
	 default:
	    g_scanner_get_next_token (scanner);
	    token = G_TOKEN_RIGHT_CURLY;
	    break;
      }

      if (token != G_TOKEN_NONE)
	 return token;

      token = g_scanner_peek_next_token (scanner);
   }

   g_scanner_get_next_token (scanner);

   rc_style->engine_data = NULL;

   return G_TOKEN_NONE;
}

static void
theme_merge_rc_style (GtkRcStyle *dest, GtkRcStyle *src)
{
}

static void
theme_rc_style_to_style (GtkStyle *style, GtkRcStyle *rc_style)
{
   style->klass = &step_class;
}

static void
theme_duplicate_style (GtkStyle *dest, GtkStyle *src)
{
}

static void
theme_realize_style (GtkStyle *style)
{
   GdkGCValuesMask gc_values_mask = GDK_GC_FOREGROUND | GDK_GC_FONT;
   GdkGCValues	   gc_values;
   gint		   i;
   extern GdkFont  *default_font;
   
   if (!default_font)
      default_font
	 = gdk_font_load ("-adobe-helvetica-medium-r-normal--*-120-*-*-*-*-*-*");

   if (style->font->type == GDK_FONT_FONT)
      gc_values.font = style->font;
   else if (style->font->type == GDK_FONT_FONTSET)
      gc_values.font = default_font;
   
   for (i = 0; i < 5; i++)
   {
      /*
       *  Release old GC
       */
      gtk_gc_release (style->light_gc [i]);
      gtk_gc_release (style->dark_gc [i]);

      /*
       *  Compute new colors
       */
      step_style_shade (&style->bg [i], &style->light [i], LIGHTNESS_MULT);
      step_style_shade (&style->bg [i], &style->dark [i], DARKNESS_MULT);
      
      style->mid [i].red = (style->light [i].red + style->dark [i].red) / 2;
      style->mid [i].green = (style->light [i].green + style->dark [i].green) / 2;
      style->mid [i].blue = (style->light [i].blue + style->dark [i].blue) / 2;

      /*
       *  Allocate new gc
       */
      if (!gdk_color_alloc (style->colormap, &style->light [i]))
	g_warning ("unable to allocate color: ( %d %d %d )",
		   style->light [i].red, style->light [i].green,
		   style->light [i].blue);
      if (!gdk_color_alloc (style->colormap, &style->dark [i]))
	g_warning ("unable to allocate color: ( %d %d %d )",
		   style->dark [i].red, style->dark [i].green,
		   style->dark [i].blue);
      if (!gdk_color_alloc (style->colormap, &style->mid [i]))
	g_warning ("unable to allocate color: ( %d %d %d )",
		   style->mid [i].red, style->mid [i].green,
		   style->mid [i].blue);

      gc_values.foreground = style->light [i];
      style->light_gc [i] = gtk_gc_get (style->depth, style->colormap,
					&gc_values, gc_values_mask);
      
      gc_values.foreground = style->dark[i];
      style->dark_gc [i] = gtk_gc_get (style->depth, style->colormap,
				       &gc_values, gc_values_mask);
      
      gc_values.foreground = style->mid [i];
      style->mid_gc [i] = gtk_gc_get (style->depth, style->colormap,
				      &gc_values, gc_values_mask);
   }
}

static void
theme_unrealize_style (GtkStyle *style)
{
}

static void
theme_destroy_rc_style (GtkRcStyle *rc_style)
{
}

static void
theme_destroy_style (GtkStyle *style)
{
}

void
theme_init (GtkThemeEngine *engine)
{
   engine->parse_rc_style    = theme_parse_rc_style;
   engine->merge_rc_style    = theme_merge_rc_style;
   engine->rc_style_to_style = theme_rc_style_to_style;
   engine->duplicate_style   = theme_duplicate_style;
   engine->realize_style     = theme_realize_style;
   engine->unrealize_style   = theme_unrealize_style;
   engine->destroy_rc_style  = theme_destroy_rc_style;
   engine->destroy_style     = theme_destroy_style;
   engine->set_background    = NULL;

   /*
    *  Make the scrollbars wider and move both scrollbar buttons
    *  of a horizontal scrollbar to the left and
    *  of a vertical scrollbar to the bottom.
    */
   {
      GtkRangeClass	 *rangeclass;
      GtkHScrollbarClass *hscrollbarclass;
      GtkVScrollbarClass *vscrollbarclass;
      GtkWidgetClass     *widgetclass;

      hscrollbarclass
	 = (GtkHScrollbarClass *) gtk_type_class (gtk_hscrollbar_get_type ());
      rangeclass  = (GtkRangeClass *) hscrollbarclass;
      widgetclass = (GtkWidgetClass *) hscrollbarclass;
      
      rangeclass->slider_width           = SCROLLBAR_WIDTH;
      rangeclass->min_slider_size        = MINSLIDER_WIDTH;
      rangeclass->stepper_size           = SCROLLBAR_WIDTH;
      rangeclass->stepper_slider_spacing = 1;

#ifdef NEXT_SCROLLBAR
      rangeclass->motion 	 = step_range_hmotion;
      rangeclass->trough_click   = step_range_htrough_click;
      rangeclass->slider_update  = step_hscrollbar_slider_update;
      widgetclass->realize       = step_hscrollbar_realize;
      widgetclass->size_allocate = step_hscrollbar_size_allocate;
#endif /* NEXT_SCROLLBAR */
      
      vscrollbarclass
	 = (GtkVScrollbarClass *) gtk_type_class (gtk_vscrollbar_get_type ());
      rangeclass  = (GtkRangeClass *) vscrollbarclass;
      widgetclass = (GtkWidgetClass *) vscrollbarclass;
      
      rangeclass->slider_width           = SCROLLBAR_WIDTH;
      rangeclass->min_slider_size        = MINSLIDER_WIDTH;
      rangeclass->stepper_size           = SCROLLBAR_WIDTH;
      rangeclass->stepper_slider_spacing = 1;

#ifdef NEXT_SCROLLBAR
      rangeclass->motion 	 = step_range_vmotion;
      rangeclass->trough_click   = step_range_vtrough_click;
      rangeclass->slider_update  = step_vscrollbar_slider_update;
      widgetclass->realize       = step_vscrollbar_realize;
      widgetclass->size_allocate = step_vscrollbar_size_allocate;
      old_parent_set             = widgetclass->parent_set;
      widgetclass->parent_set    = step_vscrollbar_parent_set;
#endif /* NEXT_SCROLLBAR */
   }
   
   /*
    *  Set default scrolled window placement to top and right.
    */
   {
      GtkScrolledWindowClass *scrolled;
      GtkWidgetClass	     *widget_class;
      
      scrolled = (GtkScrolledWindowClass *) gtk_type_class (gtk_scrolled_window_get_type ());
      widget_class = (GtkWidgetClass *) scrolled;
      
      scrolled->scrollbar_spacing 	= 1;
      old_scrolled_window_size_allocate = widget_class->size_allocate;
      widget_class->size_allocate 	= step_scrolled_window_size_allocate;
   }

   /*
    *  Don't draw focus for buttons
    */
   {
      GtkButtonClass *buttonclass;
      GtkWidgetClass *widget_class;

      buttonclass  = (GtkButtonClass *) gtk_type_class (gtk_button_get_type ());
      widget_class = (GtkWidgetClass *) buttonclass;

      widget_class->expose_event = step_button_expose;
      widget_class->draw 	 = step_button_draw;
   }
   
   /*
    *  Make radio and check buttons larger
    */
   {
      GtkRadioButtonClass *radiobuttonclass;
      GtkCheckButtonClass *checkbuttonclass;

      radiobuttonclass
	 = (GtkRadioButtonClass *) gtk_type_class (gtk_radio_button_get_type ());
      checkbuttonclass = (GtkCheckButtonClass *) radiobuttonclass;
      checkbuttonclass->indicator_size = INDICATOR_SIZE;

      checkbuttonclass
	 = (GtkCheckButtonClass *) gtk_type_class (gtk_check_button_get_type ());
      checkbuttonclass->indicator_size = INDICATOR_SIZE;
   }

   /*
    *  Move handle in hpaned and vpaned to middle position
    */
   {
      GtkHPanedClass *hpanedclass;
      GtkVPanedClass *vpanedclass;
      GtkWidgetClass *v_widget_class;
      GtkWidgetClass *h_widget_class;
       
      hpanedclass = (GtkHPanedClass *) gtk_type_class (gtk_hpaned_get_type ());
      vpanedclass = (GtkVPanedClass *) gtk_type_class (gtk_vpaned_get_type ());

      h_widget_class 		    = (GtkWidgetClass *) hpanedclass;
      v_widget_class 		    = (GtkWidgetClass *) vpanedclass;
      old_hpaned_size_allocate 	    = h_widget_class->size_allocate;
      old_vpaned_size_allocate 	    = v_widget_class->size_allocate;
      h_widget_class->size_allocate = step_hpaned_size_allocate;
      v_widget_class->size_allocate = step_vpaned_size_allocate;
   }
}

void
theme_exit (void)
{
}

/*
 * The following function will be called by GTK+ when the module
 * is loaded and checks to see if we are compatible with the
 * version of GTK+ that loads us.
 */

G_MODULE_EXPORT const gchar*
g_module_check_init (GModule *module);

const gchar*
g_module_check_init (GModule *module)
{
   return gtk_check_version (GTK_MAJOR_VERSION,
			     GTK_MINOR_VERSION,
			     GTK_MICRO_VERSION - GTK_INTERFACE_AGE);
}

/*
 *  GTKstep drawing functions
 */

static void
step_vpaned_size_allocate (GtkWidget     *widget,
				    GtkAllocation *allocation)
{
   GtkPaned *paned = GTK_PANED (widget);

   if (paned->gutter_size < 10)
      paned->gutter_size = 10;
   old_vpaned_size_allocate (widget, allocation);
   if (paned->gutter_size < 10)
      paned->gutter_size = 10;

   paned->handle_xpos = allocation->width / 2 - 3;

   if (GTK_WIDGET_REALIZED (widget))
   {
      gdk_window_move_resize (widget->window,
			      allocation->x, allocation->y,
			      allocation->width, allocation->height);

      gdk_window_move (paned->handle, paned->handle_xpos, paned->handle_ypos);
   }
}

static void
step_hpaned_size_allocate (GtkWidget     *widget,
			   GtkAllocation *allocation)
{
   GtkPaned *paned = GTK_PANED (widget);

   if (paned->gutter_size < 10)
      paned->gutter_size = 10;
   old_hpaned_size_allocate (widget, allocation);
   if (paned->gutter_size < 10)
      paned->gutter_size = 10;

   paned->handle_ypos = allocation->height / 2 - 3;

   if (GTK_WIDGET_REALIZED (widget))
   {
      gdk_window_move_resize (widget->window,
			      allocation->x, allocation->y,
			      allocation->width, allocation->height);

      gdk_window_move (paned->handle, paned->handle_xpos, paned->handle_ypos);
   }
}

static void
step_button_draw (GtkWidget    *widget,
		  GdkRectangle *area)
{
  GtkButton *button;
  GdkRectangle child_area;
  GdkRectangle tmp_area;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_BUTTON (widget));
  g_return_if_fail (area != NULL);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      button = GTK_BUTTON (widget);

      tmp_area = *area;
      tmp_area.x -= GTK_CONTAINER (button)->border_width;
      tmp_area.y -= GTK_CONTAINER (button)->border_width;

      step_button_paint (widget, &tmp_area);

      if (GTK_BIN (button)->child && gtk_widget_intersect (GTK_BIN (button)->child, &tmp_area, &child_area))
	gtk_widget_draw (GTK_BIN (button)->child, &child_area);
    }
}

static gint
step_button_expose (GtkWidget      *widget,
		    GdkEventExpose *event)
{
  GtkBin *bin;
  GdkEventExpose child_event;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_BUTTON (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      bin = GTK_BIN (widget);
      
      step_button_paint (widget, &event->area);

      child_event = *event;
      if (bin->child && GTK_WIDGET_NO_WINDOW (bin->child) &&
	  gtk_widget_intersect (bin->child, &event->area, &child_event.area))
	gtk_widget_event (bin->child, (GdkEvent*) &child_event);
    }

  return FALSE;
}

#define DEFAULT_LEFT_POS  4
#define DEFAULT_TOP_POS   4
#define DEFAULT_SPACING   7

static void
step_button_paint (GtkWidget    *widget,
		   GdkRectangle *area)
{
  GtkButton *button;
  GtkShadowType shadow_type;
  gint width, height;
  gint x, y;
   
  if (GTK_WIDGET_DRAWABLE (widget))
    {
      button = GTK_BUTTON (widget);
	
      x = 0;
      y = 0;
      width = widget->allocation.width - GTK_CONTAINER (widget)->border_width * 2;
      height = widget->allocation.height - GTK_CONTAINER (widget)->border_width * 2;

      gdk_window_set_back_pixmap (widget->window, NULL, TRUE);
      gdk_window_clear_area (widget->window, area->x, area->y, area->width, area->height);

      if (GTK_WIDGET_HAS_DEFAULT (widget) &&
	  GTK_BUTTON (widget)->relief == GTK_RELIEF_NORMAL)
	{
	  gtk_paint_box (widget->style, widget->window,
			 GTK_STATE_NORMAL, GTK_SHADOW_IN,
			 area, widget, "buttondefault",
			 x, y, width, height);
	}

      if (GTK_WIDGET_CAN_DEFAULT (widget))
	{
	  x += widget->style->klass->xthickness;
	  y += widget->style->klass->ythickness;
	  width -= 2 * x + DEFAULT_SPACING;
	  height -= 2 * y + DEFAULT_SPACING;
	  x += DEFAULT_LEFT_POS;
	  y += DEFAULT_TOP_POS;
	}
       
      if (GTK_WIDGET_STATE (widget) == GTK_STATE_ACTIVE)
	shadow_type = GTK_SHADOW_IN;
      else
	shadow_type = GTK_SHADOW_OUT;

      if ((button->relief != GTK_RELIEF_NONE) ||
	  ((GTK_WIDGET_STATE(widget) != GTK_STATE_NORMAL) &&
	   (GTK_WIDGET_STATE(widget) != GTK_STATE_INSENSITIVE)))
	gtk_paint_box (widget->style, widget->window,
		       GTK_WIDGET_STATE (widget),
		       shadow_type, area, widget, "button",
		       x, y, width, height);
    }
}

static void
step_hscrollbar_realize (GtkWidget *widget)
{
  GtkRange *range;
  GdkWindowAttr attributes;
  gint attributes_mask;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_HSCROLLBAR (widget));
  
  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  range = GTK_RANGE (widget);
  
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y + (widget->allocation.height - widget->requisition.height) / 2;
  attributes.width = widget->allocation.width;
  attributes.height = widget->requisition.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget);
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
                            GDK_BUTTON_PRESS_MASK |
                            GDK_BUTTON_RELEASE_MASK |
                            GDK_ENTER_NOTIFY_MASK |
                            GDK_LEAVE_NOTIFY_MASK);
  
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes, attributes_mask);
  
  range->trough = widget->window;
  gdk_window_ref (range->trough);
  
  attributes.x = widget->style->klass->xthickness;
  attributes.y = widget->style->klass->ythickness;
  attributes.width = RANGE_CLASS (widget)->stepper_size;
  attributes.height = RANGE_CLASS (widget)->stepper_size;
  
  range->step_back = gdk_window_new (range->trough, &attributes, attributes_mask);
  
  attributes.x = widget->style->klass->xthickness
		 + RANGE_CLASS (widget)->stepper_size;
  
  range->step_forw = gdk_window_new (range->trough, &attributes, attributes_mask);
  
  attributes.x = 0;
  attributes.y = widget->style->klass->ythickness;
  attributes.width = RANGE_CLASS (widget)->min_slider_size;
  attributes.height = RANGE_CLASS (widget)->slider_width;
  attributes.event_mask |= (GDK_BUTTON_MOTION_MASK |
                            GDK_POINTER_MOTION_HINT_MASK);
  
  range->slider = gdk_window_new (range->trough, &attributes, attributes_mask);
  
  step_hscrollbar_calc_slider_size (GTK_HSCROLLBAR (widget));
  gtk_range_slider_update (GTK_RANGE (widget));
  
  widget->style = gtk_style_attach (widget->style, widget->window);
  
  gdk_window_set_user_data (range->trough, widget);
  gdk_window_set_user_data (range->slider, widget);
  gdk_window_set_user_data (range->step_forw, widget);
  gdk_window_set_user_data (range->step_back, widget);
  
  gtk_style_set_background (widget->style, range->trough, GTK_STATE_ACTIVE);
  gtk_style_set_background (widget->style, range->slider, GTK_STATE_NORMAL);
  gtk_style_set_background (widget->style, range->step_forw, GTK_STATE_ACTIVE);
  gtk_style_set_background (widget->style, range->step_back, GTK_STATE_ACTIVE);
  
  if (range->adjustment->page_size == 
      (range->adjustment->upper - range->adjustment->lower))
  {
     gdk_window_hide (range->slider);
     gdk_window_hide (range->step_back);
     gdk_window_hide (range->step_forw);
  }
  else
  {
     gdk_window_show (range->slider);
     gdk_window_show (range->step_back);
     gdk_window_show (range->step_forw);
  }                  
}

static void
step_vscrollbar_realize (GtkWidget *widget)
{
  GtkRange *range;
  GdkWindowAttr attributes;
  gint attributes_mask;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_VSCROLLBAR (widget));
  
  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  range = GTK_RANGE (widget);
  
  attributes.x = widget->allocation.x + (widget->allocation.width - widget->requisition.width) / 2;
  attributes.y = widget->allocation.y;
  attributes.width = widget->requisition.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget);
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
                            GDK_BUTTON_PRESS_MASK |
                            GDK_BUTTON_RELEASE_MASK |
                            GDK_ENTER_NOTIFY_MASK |
                            GDK_LEAVE_NOTIFY_MASK);
  
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes, attributes_mask);
  
  range->trough = widget->window;
  gdk_window_ref (range->trough);
  
  attributes.x = widget->style->klass->xthickness;
  attributes.y = (widget->allocation.height -
		  widget->style->klass->ythickness -
		  2 * RANGE_CLASS (widget)->stepper_size);
  attributes.width = RANGE_CLASS (widget)->stepper_size;
  attributes.height = RANGE_CLASS (widget)->stepper_size;
  
  range->step_back = gdk_window_new (range->trough, &attributes, attributes_mask);
  
  attributes.y = (widget->allocation.height -
                  widget->style->klass->ythickness -
                  RANGE_CLASS (widget)->stepper_size);
  
  range->step_forw = gdk_window_new (range->trough, &attributes, attributes_mask);
  
  attributes.x = widget->style->klass->ythickness;
  attributes.y = 0;
  attributes.width = RANGE_CLASS (widget)->slider_width;
  attributes.height = RANGE_CLASS (widget)->min_slider_size;
  attributes.event_mask |= (GDK_BUTTON_MOTION_MASK |
                            GDK_POINTER_MOTION_HINT_MASK);
  
  range->slider = gdk_window_new (range->trough, &attributes, attributes_mask);
  
  step_vscrollbar_calc_slider_size (GTK_VSCROLLBAR (widget));
  gtk_range_slider_update (GTK_RANGE (widget));
  
  widget->style = gtk_style_attach (widget->style, widget->window);
  
  gdk_window_set_user_data (range->trough, widget);
  gdk_window_set_user_data (range->slider, widget);
  gdk_window_set_user_data (range->step_forw, widget);
  gdk_window_set_user_data (range->step_back, widget);
  
  gtk_style_set_background (widget->style, range->trough, GTK_STATE_ACTIVE);
  gtk_style_set_background (widget->style, range->slider, GTK_STATE_NORMAL);
  gtk_style_set_background (widget->style, range->step_forw, GTK_STATE_ACTIVE);
  gtk_style_set_background (widget->style, range->step_back, GTK_STATE_ACTIVE);
  
  if (range->adjustment->page_size == 
      (range->adjustment->upper - range->adjustment->lower))
  {
     gdk_window_hide (range->slider);
     gdk_window_hide (range->step_back);
     gdk_window_hide (range->step_forw);
  }
  else
  {
     gdk_window_show (range->slider);
     gdk_window_show (range->step_back);
     gdk_window_show (range->step_forw);
  }                  
}

static void
step_hscrollbar_size_allocate (GtkWidget     *widget,
			       GtkAllocation *allocation)
{
  GtkRange *range;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_HSCROLLBAR (widget));
  g_return_if_fail (allocation != NULL);
  
  widget->allocation = *allocation;
  if (GTK_WIDGET_REALIZED (widget))
    {
      range = GTK_RANGE (widget);
      
      gdk_window_move_resize (range->trough,
                              allocation->x,
                              allocation->y + (allocation->height - widget->requisition.height) / 2,
                              allocation->width, widget->requisition.height);
      gdk_window_move_resize (range->step_back,
			      widget->style->klass->xthickness,
                              widget->style->klass->ythickness,
                              RANGE_CLASS (widget)->stepper_size,
                              widget->requisition.height - widget->style->klass->ythickness * 2);
      gdk_window_move_resize (range->step_forw,
                              widget->style->klass->xthickness +
                              RANGE_CLASS (widget)->stepper_size,
                              widget->style->klass->ythickness,
                              RANGE_CLASS (widget)->stepper_size,
                              widget->requisition.height - widget->style->klass->ythickness * 2);
      gdk_window_resize (range->slider,
                         RANGE_CLASS (widget)->min_slider_size,
                         widget->requisition.height - widget->style->klass->ythickness * 2);
      
      gtk_range_slider_update (GTK_RANGE (widget));
    }
}

static void
step_vscrollbar_size_allocate (GtkWidget     *widget,
                              GtkAllocation *allocation)
{
  GtkRange *range;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_VSCROLLBAR (widget));
  g_return_if_fail (allocation != NULL);
  
  widget->allocation = *allocation;
  if (GTK_WIDGET_REALIZED (widget))
    {
      range = GTK_RANGE (widget);
      
      gdk_window_move_resize (range->trough,
                              allocation->x + (allocation->width - widget->requisition.width) / 2,
                              allocation->y,
                              widget->requisition.width, allocation->height);
      gdk_window_move_resize (range->step_back,
                              widget->style->klass->xthickness,
			      allocation->height
			      - widget->style->klass->ythickness -
			      2 * RANGE_CLASS (widget)->stepper_size,
                              widget->requisition.width - widget->style->klass->xthickness * 2,
                              RANGE_CLASS (widget)->stepper_size);
      gdk_window_move_resize (range->step_forw,
                              widget->style->klass->xthickness,
                              allocation->height - widget->style->klass->ythickness -
                              RANGE_CLASS (widget)->stepper_size,
                              widget->requisition.width - widget->style->klass->xthickness * 2,
                              RANGE_CLASS (widget)->stepper_size);
      gdk_window_resize (range->slider,
                         widget->requisition.width - widget->style->klass->xthickness * 2,
                         RANGE_CLASS (range)->min_slider_size);
      
      gtk_range_slider_update (GTK_RANGE (widget));
    }
}

static void
step_hscrollbar_slider_update (GtkRange *range)
{
  g_return_if_fail (range != NULL);
  g_return_if_fail (GTK_IS_HSCROLLBAR (range));
  
  step_hscrollbar_calc_slider_size (GTK_HSCROLLBAR (range));
  step_range_hslider_update (range);
}

static void
step_vscrollbar_slider_update (GtkRange *range)
{
  g_return_if_fail (range != NULL);
  g_return_if_fail (GTK_IS_VSCROLLBAR (range));
  
  step_vscrollbar_calc_slider_size (GTK_VSCROLLBAR (range));
  step_range_vslider_update (range);
}

static void
step_range_trough_hdims (GtkRange *range,
			 gint     *left,
			 gint     *right)
{
  gint trough_width;
  gint slider_length;
  gint tmp_width;
  gint tleft;
  gint tright;

  g_return_if_fail (range != NULL);

  gdk_window_get_size (range->trough, &trough_width, NULL);
  gdk_window_get_size (range->slider, &slider_length, NULL);

  tleft = GTK_WIDGET (range)->style->klass->xthickness +
	  RANGE_CLASS (range)->stepper_slider_spacing;
  tright = trough_width - slider_length - GTK_WIDGET (range)->style->klass->xthickness;

  if (range->step_back)
    {
      gdk_window_get_size (range->step_back, &tmp_width, NULL);
      tleft += (tmp_width);
    }

  if (range->step_forw)
    {
      gdk_window_get_size (range->step_forw, &tmp_width, NULL);
      tleft += (tmp_width);
    }

  if (left)
    *left = tleft;
  if (right)
    *right = tright;
}

static void
step_range_trough_vdims (GtkRange *range,
			 gint     *top,
			 gint     *bottom)
{
  gint trough_height;
  gint slider_length;
  gint tmp_height;
  gint ttop;
  gint tbottom;

  g_return_if_fail (range != NULL);

  gdk_window_get_size (range->trough, NULL, &trough_height);
  gdk_window_get_size (range->slider, NULL, &slider_length);

  ttop = GTK_WIDGET (range)->style->klass->ythickness
	 + RANGE_CLASS (range)->stepper_slider_spacing;
  tbottom = trough_height - slider_length - GTK_WIDGET (range)->style->klass->ythickness;

  if (range->step_back)
    {
      gdk_window_get_size (range->step_back, NULL, &tmp_height);
      tbottom -= (tmp_height + RANGE_CLASS (range)->stepper_slider_spacing);
    }

  if (range->step_forw)
    {
      gdk_window_get_size (range->step_forw, NULL, &tmp_height);
      tbottom -= (tmp_height);
    }

  if (top)
    *top = ttop;
  if (bottom)
    *bottom = tbottom;
}

static void
step_range_hmotion (GtkRange *range,
		    gint      xdelta,
		    gint      ydelta)
{
  gdouble old_value;
  gint left, right;
  gint slider_x, slider_y;
  gint new_pos;

  g_return_if_fail (range != NULL);
  g_return_if_fail (GTK_IS_RANGE (range));

  range = GTK_RANGE (range);

  gdk_window_get_position (range->slider, &slider_x, &slider_y);
  step_range_trough_hdims (range, &left, &right);

  if (left == right)
    return;

  new_pos = slider_x + xdelta;

  if (new_pos < left)
    new_pos = left;
  else if (new_pos > right)
    new_pos = right;

  old_value = range->adjustment->value;
  range->adjustment->value = ((range->adjustment->upper -
			       range->adjustment->lower -
			       range->adjustment->page_size) *
			      (new_pos - left) / (right - left) +
			      range->adjustment->lower);

  if (range->digits >= 0)
    {
      char buffer[64];

      sprintf (buffer, "%0.*f", range->digits, range->adjustment->value);
      sscanf (buffer, "%f", &range->adjustment->value);
    }

  if (old_value != range->adjustment->value)
    {
      if (range->policy == GTK_UPDATE_CONTINUOUS)
	{
	  gtk_signal_emit_by_name (GTK_OBJECT (range->adjustment), "value_changed");
	}
      else
	{
	  gtk_range_slider_update (range);
	  gtk_range_clear_background (range);

	  if (range->policy == GTK_UPDATE_DELAYED)
	    {
	      step_range_remove_timer (range);
	      range->timer = gtk_timeout_add (SCROLL_DELAY_LENGTH,
					      (GtkFunction) RANGE_CLASS (range)->timer,
					      (gpointer) range);
	    }
	}
    }
}

static void
step_range_vmotion (GtkRange *range,
		    gint      xdelta,
		    gint      ydelta)
{
  gdouble old_value;
  gint top, bottom;
  gint slider_x, slider_y;
  gint new_pos;

  g_return_if_fail (range != NULL);
  g_return_if_fail (GTK_IS_RANGE (range));

  range = GTK_RANGE (range);

  gdk_window_get_position (range->slider, &slider_x, &slider_y);
  step_range_trough_vdims (range, &top, &bottom);

  if (bottom == top)
    return;

  new_pos = slider_y + ydelta;

  if (new_pos < top)
    new_pos = top;
  else if (new_pos > bottom)
    new_pos = bottom;

  old_value = range->adjustment->value;
  range->adjustment->value = ((range->adjustment->upper -
			       range->adjustment->lower -
			       range->adjustment->page_size) *
			      (new_pos - top) / (bottom - top) +
			      range->adjustment->lower);

  if (range->digits >= 0)
    {
      char buffer[64];

      sprintf (buffer, "%0.*f", range->digits, range->adjustment->value);
      sscanf (buffer, "%f", &range->adjustment->value);
    }

  if (old_value != range->adjustment->value)
    {
      if (range->policy == GTK_UPDATE_CONTINUOUS)
	{
	  gtk_signal_emit_by_name (GTK_OBJECT (range->adjustment), "value_changed");
	}
      else
	{
	  gtk_range_slider_update (range);
	  gtk_range_clear_background (range);

	  if (range->policy == GTK_UPDATE_DELAYED)
	    {
	      step_range_remove_timer (range);
	      range->timer = gtk_timeout_add (SCROLL_DELAY_LENGTH,
					      (GtkFunction) RANGE_CLASS (range)->timer,
					      (gpointer) range);
	    }
	}
    }
}

static gint
step_range_htrough_click (GtkRange *range,
			  gint      x,
			  gint      y,
			  gfloat   *jump_perc)
{
  gint ythickness;
  gint trough_width;
  gint trough_height;
  gint slider_x;
  gint slider_length;
  gint left, right;

  g_return_val_if_fail (range != NULL, GTK_TROUGH_NONE);
  g_return_val_if_fail (GTK_IS_RANGE (range), GTK_TROUGH_NONE);

  ythickness = GTK_WIDGET (range)->style->klass->ythickness;

  step_range_trough_hdims (range, &left, &right);
  gdk_window_get_size (range->slider, &slider_length, NULL);
  right += slider_length;
	      
  if ((x > left) && (y > ythickness))
    {
      gdk_window_get_size (range->trough, &trough_width, &trough_height);

      if ((x < right) && (y < (trough_height - ythickness)))
	{
	  if (jump_perc)
	    {
	      *jump_perc = ((gdouble) (x - left)) / ((gdouble) (right - left));
	      return GTK_TROUGH_JUMP;
	    }
	  
	  gdk_window_get_position (range->slider, &slider_x, NULL);
	  
	  if (x < slider_x)
	    return GTK_TROUGH_START;
	  else
	    return GTK_TROUGH_END;
	}
    }

  return GTK_TROUGH_NONE;
}

static gint
step_range_vtrough_click (GtkRange *range,
			  gint      x,
			  gint      y,
			  gfloat   *jump_perc)
{
  gint xthickness;
  gint trough_width;
  gint trough_height;
  gint slider_y;
  gint top, bottom;
  gint slider_length;

  g_return_val_if_fail (range != NULL, GTK_TROUGH_NONE);
  g_return_val_if_fail (GTK_IS_RANGE (range), GTK_TROUGH_NONE);

  xthickness = GTK_WIDGET (range)->style->klass->xthickness;

  step_range_trough_vdims (range, &top, &bottom);
  gdk_window_get_size (range->slider, NULL, &slider_length);
  bottom += slider_length;
	      
  if ((x > xthickness) && (y > top))
    {
      gdk_window_get_size (range->trough, &trough_width, &trough_height);

      if ((x < (trough_width - xthickness) && (y < bottom)))
	{
	  if (jump_perc)
	    {
	      *jump_perc = ((gdouble) (y - top)) / ((gdouble) (bottom - top));

	      return GTK_TROUGH_JUMP;
	    }
	  
	  gdk_window_get_position (range->slider, NULL, &slider_y);
	  
	  if (y < slider_y)
	    return GTK_TROUGH_START;
	  else
	    return GTK_TROUGH_END;
	}
    }

  return GTK_TROUGH_NONE;
}

static void
step_range_hslider_update (GtkRange *range)
{
  gint left;
  gint right;
  gint x;

  g_return_if_fail (range != NULL);
  g_return_if_fail (GTK_IS_RANGE (range));

  if (GTK_WIDGET_REALIZED (range))
    {
      step_range_trough_hdims (range, &left, &right);
      x = left;

      if (range->adjustment->value < range->adjustment->lower)
	{
	  range->adjustment->value = range->adjustment->lower;
	  gtk_signal_emit_by_name (GTK_OBJECT (range->adjustment), "value_changed");
	}
      else if (range->adjustment->value > range->adjustment->upper)
	{
	  range->adjustment->value = range->adjustment->upper;
	  gtk_signal_emit_by_name (GTK_OBJECT (range->adjustment), "value_changed");
	}

      if (range->adjustment->lower != (range->adjustment->upper - range->adjustment->page_size))
	x += ((right - left) * (range->adjustment->value - range->adjustment->lower) /
	      (range->adjustment->upper - range->adjustment->lower - range->adjustment->page_size));

      if (x < left)
	x = left;
      else if (x > right)
	x = right;

      if (range->adjustment->page_size == 
	  (range->adjustment->upper - range->adjustment->lower))
      {
	 gdk_window_hide (range->slider);
	 gdk_window_hide (range->step_back);
	 gdk_window_hide (range->step_forw);
      }
      else
      {
	 gdk_window_show (range->slider);
	 gdk_window_show (range->step_back);
	 gdk_window_show (range->step_forw);
      }                  
      gdk_window_move (range->slider, x, GTK_WIDGET (range)->style->klass->ythickness);
    }
}

static void
step_range_vslider_update (GtkRange *range)
{
  gint top;
  gint bottom;
  gint y;

  g_return_if_fail (range != NULL);
  g_return_if_fail (GTK_IS_RANGE (range));

  if (GTK_WIDGET_REALIZED (range))
    {
      step_range_trough_vdims (range, &top, &bottom);
      y = top;

      if (range->adjustment->value < range->adjustment->lower)
	{
	  range->adjustment->value = range->adjustment->lower;
	  gtk_signal_emit_by_name (GTK_OBJECT (range->adjustment), "value_changed");
	}
      else if (range->adjustment->value > range->adjustment->upper)
	{
	  range->adjustment->value = range->adjustment->upper;
	  gtk_signal_emit_by_name (GTK_OBJECT (range->adjustment), "value_changed");
	}

      if (range->adjustment->lower != (range->adjustment->upper - range->adjustment->page_size))
	y += ((bottom - top) * (range->adjustment->value - range->adjustment->lower) /
	      (range->adjustment->upper - range->adjustment->lower - range->adjustment->page_size));

      if (y < top)
	y = top;
      else if (y > bottom)
	y = bottom;

      if (range->adjustment->page_size == 
	  (range->adjustment->upper - range->adjustment->lower))
      {
	 gdk_window_hide (range->slider);
	 gdk_window_hide (range->step_back);
	 gdk_window_hide (range->step_forw);
      }
      else
      {
	 gdk_window_show (range->slider);
	 gdk_window_show (range->step_back);
	 gdk_window_show (range->step_forw);
      }                  

      gdk_window_move (range->slider, GTK_WIDGET (range)->style->klass->xthickness, y);
    }
}

static void
step_hscrollbar_calc_slider_size (GtkHScrollbar *hscrollbar)
{
  GtkRange *range;
  gint step_back_x;
  gint step_back_width;
  gint step_forw_x;
  gint step_forw_width;
  gint trough_width;
  gint slider_width;
  gint slider_height;
  gint left, right;
  gint width;
  
  g_return_if_fail (hscrollbar != NULL);
  g_return_if_fail (GTK_IS_HSCROLLBAR (hscrollbar));
  
  if (GTK_WIDGET_REALIZED (hscrollbar))
    {
      range = GTK_RANGE (hscrollbar);
      
      gdk_window_get_size (range->step_back, &step_back_width, NULL);
      gdk_window_get_size (range->step_forw, &step_forw_width, NULL);
      gdk_window_get_size (range->trough, &trough_width, NULL);
      gdk_window_get_position (range->step_back, &step_back_x, NULL);
      gdk_window_get_position (range->step_forw, &step_forw_x, NULL);
      
      left = step_forw_x + RANGE_CLASS (hscrollbar)->stepper_slider_spacing
	     + RANGE_CLASS (hscrollbar)->stepper_size;
      right = GTK_WIDGET (hscrollbar)->allocation.width
	      - GTK_WIDGET (hscrollbar)->style->klass->xthickness;
      width = right - left;
      
      if ((range->adjustment->page_size > 0) &&
          (range->adjustment->lower != range->adjustment->upper))
        {
          if (range->adjustment->page_size >
              (range->adjustment->upper - range->adjustment->lower))
            range->adjustment->page_size = range->adjustment->upper - range->adjustment->lower;
          
          width = (width * range->adjustment->page_size /
                   (range->adjustment->upper - range->adjustment->lower));
          
          if (width < RANGE_CLASS (hscrollbar)->min_slider_size)
            width = RANGE_CLASS (hscrollbar)->min_slider_size;
        }
      
      gdk_window_get_size (range->slider, &slider_width, &slider_height);
      
      if (slider_width != width)
        gdk_window_resize (range->slider, width, slider_height);
    }
}

static void
step_vscrollbar_calc_slider_size (GtkVScrollbar *vscrollbar)
{
  GtkRange *range;
  gint step_back_y;
  gint step_back_height;
  gint step_forw_y;
  gint step_forw_width;
  gint trough_width;
  gint slider_width;
  gint slider_height;
  gint top, bottom;
  gint height;
  
  g_return_if_fail (vscrollbar != NULL);
  g_return_if_fail (GTK_IS_VSCROLLBAR (vscrollbar));
  
  if (GTK_WIDGET_REALIZED (vscrollbar))
    {
      range = GTK_RANGE (vscrollbar);
      
      gdk_window_get_size (range->step_back, NULL, &step_back_height);
      gdk_window_get_size (range->step_forw, &step_forw_width, NULL);
      gdk_window_get_size (range->trough, &trough_width, NULL);
      gdk_window_get_position (range->step_back, NULL, &step_back_y);
      gdk_window_get_position (range->step_forw, NULL, &step_forw_y);
      
      top = GTK_WIDGET (vscrollbar)->style->klass->ythickness +
	    RANGE_CLASS (vscrollbar)->stepper_slider_spacing;
      bottom = step_back_y - RANGE_CLASS (vscrollbar)->stepper_slider_spacing;
      height = bottom - top;
      
      if ((range->adjustment->page_size > 0) &&
          (range->adjustment->lower != range->adjustment->upper))
        {
          if (range->adjustment->page_size >
              (range->adjustment->upper - range->adjustment->lower))
            range->adjustment->page_size = range->adjustment->upper - range->adjustment->lower;
          
          height = (height * range->adjustment->page_size /
                    (range->adjustment->upper - range->adjustment->lower));
          
          if (height < RANGE_CLASS (vscrollbar)->min_slider_size)
            height = RANGE_CLASS (vscrollbar)->min_slider_size;
        }
      
      gdk_window_get_size (range->slider, &slider_width, &slider_height);
      
      if (slider_height != height)
        gdk_window_resize (range->slider, slider_width, height);
    }
}

void
step_vscrollbar_parent_set (GtkWidget      *widget,
			    GtkWidget      *previous_parent)
{
   if (widget && widget->parent && GTK_IS_SCROLLED_WINDOW (widget->parent))
      gtk_scrolled_window_set_placement (GTK_SCROLLED_WINDOW (widget->parent),
					 GTK_CORNER_TOP_RIGHT);
   if (old_parent_set)
      old_parent_set (widget, previous_parent);
}

static void
step_scrolled_window_relative_allocation (GtkWidget     *widget,
					  GtkAllocation *allocation)
{
  GtkScrolledWindow *scrolled_window;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (allocation != NULL);

  scrolled_window = GTK_SCROLLED_WINDOW (widget);

  allocation->x = GTK_CONTAINER (widget)->border_width;
  allocation->y = GTK_CONTAINER (widget)->border_width;
  allocation->width = MAX (1, (gint)widget->allocation.width - allocation->x * 2);
  allocation->height = MAX (1, (gint)widget->allocation.height - allocation->y * 2);

  if (scrolled_window->vscrollbar_visible)
    {
      GtkRequisition vscrollbar_requisition;
      gtk_widget_get_child_requisition (scrolled_window->vscrollbar,
					&vscrollbar_requisition);
  
      if (scrolled_window->window_placement == GTK_CORNER_TOP_RIGHT ||
	  scrolled_window->window_placement == GTK_CORNER_BOTTOM_RIGHT)
	allocation->x += (vscrollbar_requisition.width +
			  SCROLLBAR_SPACING (scrolled_window));

      allocation->width = MAX (1, (gint)allocation->width -
			       ((gint)vscrollbar_requisition.width +
				(gint)SCROLLBAR_SPACING (scrolled_window)));
    }
  if (scrolled_window->hscrollbar_visible)
    {
      GtkRequisition hscrollbar_requisition;
      gtk_widget_get_child_requisition (scrolled_window->hscrollbar,
					&hscrollbar_requisition);
  
      if (scrolled_window->window_placement == GTK_CORNER_BOTTOM_LEFT ||
	  scrolled_window->window_placement == GTK_CORNER_BOTTOM_RIGHT)
	allocation->y += (hscrollbar_requisition.height +
			  SCROLLBAR_SPACING (scrolled_window));

      allocation->height = MAX (1, (gint)allocation->height -
				((gint)hscrollbar_requisition.height +
				 (gint)SCROLLBAR_SPACING (scrolled_window)));
    }
}

static void
step_scrolled_window_size_allocate (GtkWidget     *widget,
				    GtkAllocation *allocation)
{
  GtkScrolledWindow *scrolled_window;
  GtkBin *bin;
  GtkAllocation relative_allocation;
  GtkAllocation child_allocation;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SCROLLED_WINDOW (widget));
  g_return_if_fail (allocation != NULL);

  scrolled_window = GTK_SCROLLED_WINDOW (widget);
  bin = GTK_BIN (scrolled_window);

  widget->allocation = *allocation;

  if (scrolled_window->hscrollbar_policy == GTK_POLICY_ALWAYS)
    scrolled_window->hscrollbar_visible = TRUE;
  else if (scrolled_window->hscrollbar_policy == GTK_POLICY_NEVER)
    scrolled_window->hscrollbar_visible = FALSE;
  if (scrolled_window->vscrollbar_policy == GTK_POLICY_ALWAYS)
    scrolled_window->vscrollbar_visible = TRUE;
  else if (scrolled_window->vscrollbar_policy == GTK_POLICY_NEVER)
    scrolled_window->vscrollbar_visible = FALSE;

  if (bin->child && GTK_WIDGET_VISIBLE (bin->child))
    {
      gboolean previous_hvis;
      gboolean previous_vvis;
      guint count = 0;
      
      do
	{
	  step_scrolled_window_relative_allocation (widget, &relative_allocation);
	  
	  child_allocation.x = relative_allocation.x + allocation->x;
	  child_allocation.y = relative_allocation.y + allocation->y;
	  child_allocation.width = relative_allocation.width;
	  child_allocation.height = relative_allocation.height;
	  
	  previous_hvis = scrolled_window->hscrollbar_visible;
	  previous_vvis = scrolled_window->vscrollbar_visible;
	  
	  gtk_widget_size_allocate (bin->child, &child_allocation);

	  /* If, after the first iteration, the hscrollbar and the
	   * vscrollbar flip visiblity, then we need both.
	   */
	  if (count &&
	      previous_hvis != scrolled_window->hscrollbar_visible &&
	      previous_vvis != scrolled_window->vscrollbar_visible)
	    {
	      scrolled_window->hscrollbar_visible = TRUE;
	      scrolled_window->vscrollbar_visible = TRUE;

	      /* a new resize is already queued at this point,
	       * so we will immediatedly get reinvoked
	       */
	      return;
	    }
	  
	  count++;
	}
      while (previous_hvis != scrolled_window->hscrollbar_visible ||
	     previous_vvis != scrolled_window->vscrollbar_visible);
    }
  else
    step_scrolled_window_relative_allocation (widget, &relative_allocation);
  
  if (scrolled_window->hscrollbar_visible)
    {
      GtkRequisition hscrollbar_requisition;
      gtk_widget_get_child_requisition (scrolled_window->hscrollbar,
					&hscrollbar_requisition);
  
      if (!GTK_WIDGET_VISIBLE (scrolled_window->hscrollbar))
	gtk_widget_show (scrolled_window->hscrollbar);

      child_allocation.x = relative_allocation.x;
      if (scrolled_window->window_placement == GTK_CORNER_TOP_LEFT ||
	  scrolled_window->window_placement == GTK_CORNER_TOP_RIGHT)
	child_allocation.y = (relative_allocation.y +
			      relative_allocation.height +
			      SCROLLBAR_SPACING (scrolled_window));
      else
	child_allocation.y = GTK_CONTAINER (scrolled_window)->border_width;

      child_allocation.width = relative_allocation.width;
      child_allocation.height = hscrollbar_requisition.height;
      child_allocation.x += allocation->x;
      child_allocation.y += allocation->y;

      gtk_widget_size_allocate (scrolled_window->hscrollbar, &child_allocation);
    }
  else if (GTK_WIDGET_VISIBLE (scrolled_window->hscrollbar))
    gtk_widget_hide (scrolled_window->hscrollbar);

  if (scrolled_window->vscrollbar_visible)
    {
      GtkRequisition vscrollbar_requisition;
      if (!GTK_WIDGET_VISIBLE (scrolled_window->vscrollbar))
	gtk_widget_show (scrolled_window->vscrollbar);

      gtk_widget_get_child_requisition (scrolled_window->vscrollbar,
					&vscrollbar_requisition);

      if (scrolled_window->window_placement == GTK_CORNER_TOP_LEFT ||
	  scrolled_window->window_placement == GTK_CORNER_BOTTOM_LEFT)
	child_allocation.x = (relative_allocation.x +
			      relative_allocation.width +
			      SCROLLBAR_SPACING (scrolled_window));
      else
	child_allocation.x = GTK_CONTAINER (scrolled_window)->border_width;

      child_allocation.y = relative_allocation.y;
      child_allocation.width = vscrollbar_requisition.width;
      child_allocation.height = relative_allocation.height;
      child_allocation.x += allocation->x;
      child_allocation.y += allocation->y;
      if (scrolled_window->window_placement == GTK_CORNER_TOP_RIGHT
	  && scrolled_window->hscrollbar_visible)
	 child_allocation.height += SCROLLBAR_SPACING (scrolled_window)
				    + scrolled_window->hscrollbar->requisition.height;
      
      gtk_widget_size_allocate (scrolled_window->vscrollbar, &child_allocation);
    }
  else if (GTK_WIDGET_VISIBLE (scrolled_window->vscrollbar))
    gtk_widget_hide (scrolled_window->vscrollbar);
}

static void
step_range_remove_timer (GtkRange *range)
{
  g_return_if_fail (range != NULL);
  g_return_if_fail (GTK_IS_RANGE (range));

  if (range->timer)
    {
      gtk_timeout_remove (range->timer);
      range->timer = 0;
    }
  range->need_timer = FALSE;
}
